/**
 * TopRanker AI Admin JavaScript
 *
 * Handles settings page interactions including:
 * - API key testing with live feedback
 * - Model selection with info display
 * - Usage counter display and refresh
 * - Tab navigation
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

/* global jQuery, toprankerAdmin */

(function($) {
	'use strict';

	/**
	 * Model information for display.
	 */
	var modelInfo = {
		'gpt-5-nano': {
			speed: 'Fastest',
			cost: 'Lowest',
			description: 'Best for most SEO optimization tasks. Fast responses, very cost-effective.'
		},
		'gpt-5-mini': {
			speed: 'Fast',
			cost: 'Low',
			description: 'Good balance of speed and capability. Slightly better for complex content.'
		},
		'gpt-5': {
			speed: 'Standard',
			cost: 'Higher',
			description: 'Most capable model. Best for nuanced content or when quality is critical.'
		}
	};

	/**
	 * TopRanker Admin module.
	 */
	var TopRankerAdmin = {

		/**
		 * Initialize the module.
		 */
		init: function() {
			this.bindEvents();
			this.initModelInfo();
			this.initUsageDisplay();
		},

		/**
		 * Bind event handlers.
		 */
		bindEvents: function() {
			// API key test connection.
			$('#topranker-test-connection').on('click', this.testConnection.bind(this));

			// Model selection change.
			$('#topranker_model').on('change', this.onModelChange.bind(this));

			// Usage refresh button.
			$('#topranker-refresh-usage').on('click', this.refreshUsage.bind(this));

			// Tab navigation keyboard support.
			$('.topranker-tabs .nav-tab').on('keydown', this.onTabKeydown.bind(this));

			// Dashboard stats refresh.
			$('#topranker-refresh-stats').on('click', this.refreshDashboardStats.bind(this));
		},

		/**
		 * Initialize model info display.
		 */
		initModelInfo: function() {
			var $select = $('#topranker_model');
			if ($select.length) {
				this.showModelInfo($select.val());
			}
		},

		/**
		 * Initialize usage display on settings page.
		 */
		initUsageDisplay: function() {
			// If on the usage tab, optionally refresh data.
			if (window.location.search.indexOf('tab=usage') > -1) {
				// Auto-refresh usage data on page load.
				this.refreshUsage();
			}
		},

		/**
		 * Test the API connection.
		 *
		 * @param {Event} e Click event.
		 */
		testConnection: function(e) {
			e.preventDefault();

			var $button = $(e.currentTarget);
			var $status = $('#topranker-connection-status');
			var $input = $('#topranker_api_key');
			var apiKey = $input.val().trim();

			// Validate input.
			if (!apiKey) {
				this.showStatus($status, 'error', toprankerAdmin.i18n.invalidKey);
				return;
			}

			// Disable button and show loading.
			$button.prop('disabled', true);
			this.showStatus($status, 'loading', toprankerAdmin.i18n.testing);

			// Make AJAX request.
			$.ajax({
				url: toprankerAdmin.restUrl + 'test-api-key',
				method: 'POST',
				beforeSend: function(xhr) {
					xhr.setRequestHeader('X-WP-Nonce', toprankerAdmin.restNonce);
				},
				data: {
					api_key: apiKey
				},
				success: function(response) {
					if (response.success) {
						this.showStatus($status, 'success', toprankerAdmin.i18n.success);
					} else {
						this.showStatus($status, 'error', response.message || toprankerAdmin.i18n.error);
					}
				}.bind(this),
				error: function(xhr) {
					var message = toprankerAdmin.i18n.networkError;

					if (xhr.responseJSON && xhr.responseJSON.message) {
						message = xhr.responseJSON.message;
					}

					this.showStatus($status, 'error', message);
				}.bind(this),
				complete: function() {
					$button.prop('disabled', false);
				}
			});
		},

		/**
		 * Show status message.
		 *
		 * @param {jQuery} $element Status element.
		 * @param {string} type     Status type: loading, success, error.
		 * @param {string} message  Status message.
		 */
		showStatus: function($element, type, message) {
			$element
				.removeClass('is-loading is-success is-error')
				.addClass('is-' + type);

			var icon = '';
			switch (type) {
				case 'loading':
					icon = '<span class="spinner is-active" style="float:none;margin:0 5px 0 0;"></span>';
					break;
				case 'success':
					icon = '<span class="dashicons dashicons-yes-alt"></span>';
					break;
				case 'error':
					icon = '<span class="dashicons dashicons-warning"></span>';
					break;
			}

			$element.html(icon + message);
		},

		/**
		 * Handle model selection change.
		 *
		 * @param {Event} e Change event.
		 */
		onModelChange: function(e) {
			var selectedModel = $(e.currentTarget).val();
			this.showModelInfo(selectedModel);
		},

		/**
		 * Show model information.
		 *
		 * @param {string} model Model identifier.
		 */
		showModelInfo: function(model) {
			var $infoContainer = $('#topranker-model-info');

			// Create container if it doesn't exist.
			if (!$infoContainer.length) {
				$infoContainer = $('<div id="topranker-model-info" class="topranker-model-info" aria-live="polite"></div>');
				$('#topranker_model').parent().append($infoContainer);
			}

			var info = modelInfo[model];
			if (info) {
				var html = '<div class="topranker-model-info-inner">' +
					'<span class="topranker-model-speed"><strong>' + toprankerAdmin.i18n.speed + ':</strong> ' + info.speed + '</span>' +
					'<span class="topranker-model-cost"><strong>' + toprankerAdmin.i18n.cost + ':</strong> ' + info.cost + '</span>' +
					'<span class="topranker-model-desc">' + info.description + '</span>' +
					'</div>';
				$infoContainer.html(html).show();
			} else {
				$infoContainer.hide();
			}
		},

		/**
		 * Refresh usage counter via AJAX.
		 *
		 * @param {Event} e Click event (optional).
		 */
		refreshUsage: function(e) {
			if (e) {
				e.preventDefault();
			}

			var $container = $('.topranker-usage-counter');
			var $refreshBtn = $('#topranker-refresh-usage');

			if (!$container.length) {
				return;
			}

			// Show loading state.
			if ($refreshBtn.length) {
				$refreshBtn.prop('disabled', true);
				$refreshBtn.find('.dashicons').addClass('topranker-spin');
			}

			$.ajax({
				url: toprankerAdmin.restUrl + 'usage',
				method: 'GET',
				beforeSend: function(xhr) {
					xhr.setRequestHeader('X-WP-Nonce', toprankerAdmin.restNonce);
				},
				success: function(response) {
					this.updateUsageDisplay(response);
				}.bind(this),
				error: function(xhr) {
					console.error('Failed to fetch usage data:', xhr);
				},
				complete: function() {
					if ($refreshBtn.length) {
						$refreshBtn.prop('disabled', false);
						$refreshBtn.find('.dashicons').removeClass('topranker-spin');
					}
				}
			});
		},

		/**
		 * Update usage display with new data.
		 *
		 * @param {Object} data Usage data from API.
		 */
		updateUsageDisplay: function(data) {
			if (data.is_pro) {
				// Pro users - show unlimited message.
				return;
			}

			var $bar = $('.topranker-usage-bar');
			var $text = $('.topranker-usage-text');
			var $reset = $('.topranker-usage-reset');

			if ($bar.length && data.limit !== 'unlimited') {
				var percentage = Math.min(100, (data.used / data.limit) * 100);
				$bar.css('width', percentage + '%');

				// Update ARIA attributes.
				$bar.attr({
					'aria-valuenow': data.used,
					'aria-valuemax': data.limit
				});
			}

			if ($text.length && data.limit !== 'unlimited') {
				$text.text(
					toprankerAdmin.i18n.usageText
						.replace('%1$d', data.used)
						.replace('%2$d', data.limit)
				);
			}

			if ($reset.length && data.resets_on) {
				$reset.text(toprankerAdmin.i18n.resetsOn.replace('%s', data.resets_on));
			}

			// Show/hide warning or limit messages.
			this.updateUsageWarnings(data);
		},

		/**
		 * Update usage warning/limit messages.
		 *
		 * @param {Object} data Usage data.
		 */
		updateUsageWarnings: function(data) {
			var $warningNotice = $('.topranker-usage-warning');
			var $limitNotice = $('.topranker-usage-limit');

			// Calculate remaining.
			var remaining = data.limit - data.used;
			var isWarning = remaining <= 5 && remaining > 0;
			var isAtLimit = remaining <= 0;

			// Handle warning state.
			if ($warningNotice.length) {
				if (isWarning && !isAtLimit) {
					$warningNotice.show().find('p').text(
						toprankerAdmin.i18n.remainingWarning.replace('%d', remaining)
					);
				} else {
					$warningNotice.hide();
				}
			}

			// Handle limit reached state.
			if ($limitNotice.length) {
				if (isAtLimit) {
					$limitNotice.show();
				} else {
					$limitNotice.hide();
				}
			}
		},

		/**
		 * Handle keyboard navigation in tabs.
		 *
		 * @param {Event} e Keydown event.
		 */
		onTabKeydown: function(e) {
			var $tabs = $('.topranker-tabs .nav-tab');
			var $current = $(e.currentTarget);
			var currentIndex = $tabs.index($current);
			var newIndex;

			switch (e.key) {
				case 'ArrowRight':
				case 'ArrowDown':
					e.preventDefault();
					newIndex = (currentIndex + 1) % $tabs.length;
					$tabs.eq(newIndex).focus();
					break;

				case 'ArrowLeft':
				case 'ArrowUp':
					e.preventDefault();
					newIndex = (currentIndex - 1 + $tabs.length) % $tabs.length;
					$tabs.eq(newIndex).focus();
					break;

				case 'Home':
					e.preventDefault();
					$tabs.first().focus();
					break;

				case 'End':
					e.preventDefault();
					$tabs.last().focus();
					break;
			}
		},

		/**
		 * Refresh dashboard statistics.
		 *
		 * @param {Event} e Click event.
		 */
		refreshDashboardStats: function(e) {
			e.preventDefault();

			var $button = $(e.currentTarget);
			var $container = $('.topranker-stats-grid');

			if (!$container.length) {
				return;
			}

			// Show loading state.
			$button.prop('disabled', true);
			$button.find('.dashicons').addClass('topranker-spin');

			$.ajax({
				url: toprankerAdmin.restUrl + 'dashboard-stats',
				method: 'GET',
				data: {
					refresh: true
				},
				beforeSend: function(xhr) {
					xhr.setRequestHeader('X-WP-Nonce', toprankerAdmin.restNonce);
				},
				success: function(response) {
					this.updateDashboardStats(response);
				}.bind(this),
				error: function(xhr) {
					console.error('Failed to refresh dashboard stats:', xhr);
				},
				complete: function() {
					$button.prop('disabled', false);
					$button.find('.dashicons').removeClass('topranker-spin');
				}
			});
		},

		/**
		 * Update dashboard stats display.
		 *
		 * @param {Object} data Dashboard stats data.
		 */
		updateDashboardStats: function(data) {
			// Update total posts.
			$('#topranker-stat-total-posts').text(data.total_posts);

			// Update meta title stats.
			$('#topranker-stat-meta-title-count').text(data.meta_title.count);
			$('#topranker-stat-meta-title-missing').text(data.meta_title.missing);
			$('.topranker-meta-title-bar').css('width', data.meta_title.percent + '%');
			$('.topranker-meta-title-percent').text(data.meta_title.percent + '%');

			// Update meta description stats.
			$('#topranker-stat-meta-desc-count').text(data.meta_description.count);
			$('#topranker-stat-meta-desc-missing').text(data.meta_description.missing);
			$('.topranker-meta-desc-bar').css('width', data.meta_description.percent + '%');
			$('.topranker-meta-desc-percent').text(data.meta_description.percent + '%');

			// Update keyphrase stats.
			$('#topranker-stat-keyphrase-count').text(data.keyphrase.count);
			$('#topranker-stat-keyphrase-missing').text(data.keyphrase.missing);
			$('.topranker-keyphrase-bar').css('width', data.keyphrase.percent + '%');
			$('.topranker-keyphrase-percent').text(data.keyphrase.percent + '%');

			// Update overall coverage.
			$('#topranker-stat-overall').text(data.overall_coverage + '%');
			$('.topranker-overall-bar').css('width', data.overall_coverage + '%');

			// Update coverage bar colors based on percentage.
			this.updateCoverageBarColors(data);
		},

		/**
		 * Update coverage bar colors based on percentages.
		 *
		 * @param {Object} data Dashboard stats data.
		 */
		updateCoverageBarColors: function(data) {
			var fields = ['meta_title', 'meta_description', 'keyphrase'];

			fields.forEach(function(field) {
				var percent = data[field].percent;
				var $bar = $('.topranker-' + field.replace('_', '-') + '-bar');
				var className = 'is-bad';

				if (percent >= 80) {
					className = 'is-good';
				} else if (percent >= 50) {
					className = 'is-warning';
				}

				$bar.removeClass('is-good is-warning is-bad').addClass(className);
			});

			// Overall coverage bar.
			var overallClass = 'is-bad';
			if (data.overall_coverage >= 80) {
				overallClass = 'is-good';
			} else if (data.overall_coverage >= 50) {
				overallClass = 'is-warning';
			}
			$('.topranker-overall-bar').removeClass('is-good is-warning is-bad').addClass(overallClass);
		}
	};

	// Initialize on document ready.
	$(document).ready(function() {
		// Extend i18n with defaults if not provided.
		toprankerAdmin.i18n = $.extend({
			testing: 'Testing connection...',
			success: 'Connection successful!',
			error: 'Connection failed',
			invalidKey: 'Invalid API key. Please check your key.',
			networkError: 'Network error. Please try again.',
			speed: 'Speed',
			cost: 'Cost',
			usageText: '%1$d of %2$d optimizations used this month',
			resetsOn: 'Resets on %s',
			remainingWarning: 'You have %d optimizations remaining this month.'
		}, toprankerAdmin.i18n || {});

		TopRankerAdmin.init();
	});

})(jQuery);
