/**
 * TopRanker AI Classic Editor Metabox JavaScript
 *
 * Handles optimization workflow, suggestion rendering, AJAX calls,
 * SERP preview updates, and user interactions in the Classic Editor.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

/* global jQuery, toprankerEditor */

( function( $ ) {
	'use strict';

	/**
	 * TopRanker Editor Module
	 */
	var TopRankerEditor = {

		/**
		 * Configuration from localized data
		 */
		config: {},

		/**
		 * DOM element cache
		 */
		elements: {},

		/**
		 * State variables
		 */
		state: {
			isOptimizing: false,
			isApplying: false,
			isReverting: false,
			debounceTimer: null,
			lastOptimizeTime: 0
		},

		/**
		 * Initialize the module
		 */
		init: function() {
			// Bail if not on Classic Editor or missing data.
			if ( typeof toprankerEditor === 'undefined' ) {
				return;
			}

			this.config = toprankerEditor;
			this.cacheElements();

			if ( ! this.elements.metabox.length ) {
				return;
			}

			// Bail if metabox is in API-key-required state (no form elements).
			if ( ! this.elements.optimizeBtn.length ) {
				return;
			}

			this.bindEvents();
			this.initSerpPreview();
			this.initCharCounters();
		},

		/**
		 * Cache DOM elements
		 */
		cacheElements: function() {
			this.elements = {
				metabox: $( '#topranker-metabox' ),
				optimizeBtn: $( '#topranker-optimize-btn' ),
				retryBtn: $( '#topranker-retry-btn' ),
				applyBtn: $( '#topranker-apply-btn' ),
				loadingContainer: $( '#topranker-loading' ),
				errorContainer: $( '#topranker-error' ),
				errorMessage: $( '#topranker-error-message' ),
				resultsContainer: $( '#topranker-results' ),
				emptyState: $( '#topranker-empty-state' ),

				// Meta title elements.
				metaTitleSuggestions: $( '#topranker-meta-title-suggestions' ),
				metaTitleCustom: $( '#topranker-meta-title-custom' ),
				metaTitleCount: $( '#topranker-meta-title-count' ),

				// Meta description elements.
				metaDescriptionSuggestions: $( '#topranker-meta-description-suggestions' ),
				metaDescriptionCustom: $( '#topranker-meta-description-custom' ),
				metaDescriptionCount: $( '#topranker-meta-description-count' ),

				// Excerpt elements.
				excerptResult: $( '#topranker-excerpt-result' ),
				excerptCustom: $( '#topranker-excerpt-custom' ),
				excerptCount: $( '#topranker-excerpt-count' ),

				// Keyphrase elements.
				keyphrases: $( '#topranker-keyphrases' ),
				keyphrasePrimary: $( '#topranker-keyphrase-primary' ),
				keyphraseSecondary: $( '#topranker-keyphrase-secondary' ),
				keyphraseCustom: $( '#topranker-keyphrase-custom' ),

				// Social meta elements.
				ogTitle: $( '#topranker-og-title' ),
				ogDescription: $( '#topranker-og-description' ),
				twitterTitle: $( '#topranker-twitter-title' ),
				twitterDescription: $( '#topranker-twitter-description' ),

				// SERP preview elements.
				serpTitle: $( '#topranker-serp-title' ),
				serpUrl: $( '#topranker-serp-url' ),
				serpDescription: $( '#topranker-serp-description' ),
				serpTitleCount: $( '#topranker-serp-title-count' ),
				serpDescriptionCount: $( '#topranker-serp-description-count' ),

				// History elements (Pro feature).
				historySection: $( '.topranker-history-section' ),
				historyList: $( '.topranker-history-list' ),
				historyEmpty: $( '.topranker-history-empty' ),
				historyModal: $( '#topranker-history-modal' ),
				historyModalTitle: $( '#topranker-history-modal-title' ),
				historyModalContent: $( '#topranker-history-modal-content' ),
				historyModalClose: $( '.topranker-modal-close' ),
				historyModalRevert: $( '#topranker-history-modal-revert' ),

				// Copy to SEO plugin button.
				copyToSeoBtn: $( '#topranker-copy-to-seo-btn' )
			};
		},

		/**
		 * Bind event handlers
		 */
		bindEvents: function() {
			var self = this;

			// Optimize button click.
			this.elements.optimizeBtn.on( 'click', function( e ) {
				e.preventDefault();
				self.handleOptimize();
			} );

			// Retry button click.
			this.elements.retryBtn.on( 'click', function( e ) {
				e.preventDefault();
				self.handleOptimize( true );
			} );

			// Apply button click.
			this.elements.applyBtn.on( 'click', function( e ) {
				e.preventDefault();
				self.handleApply();
			} );

			// Copy to SEO plugin button click.
			this.elements.copyToSeoBtn.on( 'click', function( e ) {
				e.preventDefault();
				self.handleCopyToSeoPlugin();
			} );

			// Radio suggestion selection.
			this.elements.metabox.on( 'change', 'input[type="radio"]', function() {
				self.handleSuggestionSelect( $( this ) );
			} );

			// Custom field updates.
			this.elements.metaTitleCustom.on( 'input', function() {
				self.updateCharCounter( $( this ), self.elements.metaTitleCount, 60 );
				self.updateSerpPreview();
			} );

			this.elements.metaDescriptionCustom.on( 'input', function() {
				self.updateCharCounter( $( this ), self.elements.metaDescriptionCount, 155 );
				self.updateSerpPreview();
			} );

			this.elements.excerptCustom.on( 'input', function() {
				self.updateCharCounter( $( this ), self.elements.excerptCount, 300 );
			} );

			// Social meta field updates.
			this.elements.ogTitle.on( 'input', function() {
				self.updateCharCounter( $( this ), null, 60 );
			} );

			this.elements.ogDescription.on( 'input', function() {
				self.updateCharCounter( $( this ), null, 200 );
			} );

			// History feature events (Pro only).
			if ( this.elements.historySection.length ) {
				this.bindHistoryEvents();
			}
		},

		/**
		 * Bind history-related events (Pro feature)
		 */
		bindHistoryEvents: function() {
			var self = this;

			// Preview button click.
			this.elements.historyList.on( 'click', '.topranker-history-preview-btn', function( e ) {
				e.preventDefault();
				var index = $( this ).closest( '.topranker-history-entry' ).data( 'index' );
				self.showHistoryPreview( index );
			} );

			// Revert button click (in list).
			this.elements.historyList.on( 'click', '.topranker-history-revert-btn', function( e ) {
				e.preventDefault();
				var index = $( this ).closest( '.topranker-history-entry' ).data( 'index' );
				self.confirmHistoryRevert( index );
			} );

			// Modal close button.
			this.elements.historyModalClose.on( 'click', function( e ) {
				e.preventDefault();
				self.closeHistoryModal();
			} );

			// Modal overlay click to close.
			this.elements.historyModal.on( 'click', function( e ) {
				if ( $( e.target ).hasClass( 'topranker-modal' ) ) {
					self.closeHistoryModal();
				}
			} );

			// Modal revert button.
			this.elements.historyModalRevert.on( 'click', function( e ) {
				e.preventDefault();
				var index = $( this ).data( 'index' );
				self.closeHistoryModal();
				self.confirmHistoryRevert( index );
			} );

			// ESC key to close modal.
			$( document ).on( 'keydown', function( e ) {
				if ( 27 === e.keyCode && self.elements.historyModal.is( ':visible' ) ) {
					self.closeHistoryModal();
				}
			} );
		},

		/**
		 * Handle optimize button click
		 *
		 * @param {boolean} forceRefresh Whether to force refresh.
		 */
		handleOptimize: function( forceRefresh ) {
			var self = this;
			var now = Date.now();

			// Debounce: prevent double-click (3 second delay).
			if ( ! forceRefresh && now - this.state.lastOptimizeTime < 3000 ) {
				return;
			}

			if ( this.state.isOptimizing ) {
				return;
			}

			this.state.isOptimizing = true;
			this.state.lastOptimizeTime = now;

			this.showLoading();
			this.disableOptimizeButton();

			var postId = this.elements.metabox.data( 'post-id' );
			var nonce = this.elements.metabox.data( 'nonce' );

			$.ajax( {
				url: this.config.ajax_url,
				type: 'POST',
				data: {
					action: 'topranker_optimize',
					post_id: postId,
					nonce: nonce,
					force_refresh: forceRefresh ? 'true' : 'false'
				},
				success: function( response ) {
					if ( response.success ) {
						self.handleOptimizeSuccess( response.data );
					} else {
						self.handleOptimizeError( response.data );
					}
				},
				error: function( xhr, status, error ) {
					self.handleOptimizeError( {
						message: self.config.strings.error + ' (' + error + ')'
					} );
				},
				complete: function() {
					self.state.isOptimizing = false;
					self.hideLoading();
					self.enableOptimizeButton();
				}
			} );
		},

		/**
		 * Handle successful optimization response
		 *
		 * @param {Object} data Response data.
		 */
		handleOptimizeSuccess: function( data ) {
			this.hideError();
			this.hideEmptyState();

			if ( data.results ) {
				this.renderSuggestions( data.results );
			}

			if ( data.usage ) {
				this.updateUsageDisplay( data.usage );
			}

			this.showResults();
			this.updateSerpPreview();
		},

		/**
		 * Handle optimization error
		 *
		 * @param {Object} error Error data.
		 */
		handleOptimizeError: function( error ) {
			var message = error && error.message ? error.message : this.config.strings.error;
			this.showError( message );
		},

		/**
		 * Render suggestions into the UI
		 *
		 * @param {Object} results Results from the API.
		 */
		renderSuggestions: function( results ) {
			// Meta title suggestions.
			if ( results.meta_title && results.meta_title.suggestions ) {
				this.renderRadioSuggestions(
					this.elements.metaTitleSuggestions,
					results.meta_title.suggestions,
					'topranker_meta_title_choice',
					60,
					'title'
				);

				// Update custom field with first suggestion.
				if ( results.meta_title.suggestions.length > 0 ) {
					var firstTitle = results.meta_title.suggestions[0];
					this.elements.metaTitleCustom.val( typeof firstTitle === 'object' ? ( firstTitle.title || '' ) : firstTitle );
					this.updateCharCounter( this.elements.metaTitleCustom, this.elements.metaTitleCount, 60 );
				}
			}

			// Meta description suggestions.
			if ( results.meta_description && results.meta_description.suggestions ) {
				this.renderRadioSuggestions(
					this.elements.metaDescriptionSuggestions,
					results.meta_description.suggestions,
					'topranker_meta_description_choice',
					155,
					'description'
				);

				// Update custom field with first suggestion.
				if ( results.meta_description.suggestions.length > 0 ) {
					var firstDesc = results.meta_description.suggestions[0];
					this.elements.metaDescriptionCustom.val( typeof firstDesc === 'object' ? ( firstDesc.description || '' ) : firstDesc );
					this.updateCharCounter( this.elements.metaDescriptionCustom, this.elements.metaDescriptionCount, 155 );
				}
			}

			// Excerpt result.
			if ( results.excerpt && ( results.excerpt.excerpt || results.excerpt.text ) ) {
				var excerptText = results.excerpt.excerpt || results.excerpt.text;
				this.renderExcerptResult( excerptText );
				this.elements.excerptCustom.val( excerptText );
				this.updateCharCounter( this.elements.excerptCustom, this.elements.excerptCount, 300 );
			}

			// Keyphrases.
			if ( results.keyphrases ) {
				this.renderKeyphrases( results.keyphrases );

				// Update custom field with primary keyphrase.
				if ( results.keyphrases.primary ) {
					this.elements.keyphraseCustom.val( results.keyphrases.primary );
				}
			}

			// Social meta.
			if ( results.social_meta ) {
				this.renderSocialMeta( results.social_meta );
			}

			// Alt tags (Pro).
			if ( results.alt_tags && results.alt_tags.results ) {
				this.renderAltTags( results.alt_tags );
			}

			// Update SERP preview.
			this.updateSerpPreview();
		},

		/**
		 * Render radio suggestion cards
		 *
		 * @param {jQuery}   container      Container element.
		 * @param {Array}    suggestions    Array of suggestions (strings or objects).
		 * @param {string}   name           Radio input name.
		 * @param {number}   maxLength      Maximum character length.
		 * @param {string}   textKey        Optional. Key to extract text from object suggestions.
		 */
		renderRadioSuggestions: function( container, suggestions, name, maxLength, textKey ) {
			container.empty();

			var self = this;

			suggestions.forEach( function( suggestion, index ) {
				var text = ( textKey && typeof suggestion === 'object' ) ? ( suggestion[ textKey ] || '' ) : suggestion;
				var charCount = text.length;
				var charClass = self.getCharCountClass( charCount, maxLength );
				var isSelected = index === 0;

				var charsLabel = self.config.strings.chars || '%d chars';
				var html = '<label class="topranker-suggestion' + ( isSelected ? ' is-selected' : '' ) + '">' +
					'<input type="radio" name="' + name + '" value="' + self.escapeHtml( text ) + '"' + ( isSelected ? ' checked' : '' ) + ' />' +
					'<div class="topranker-suggestion-content">' +
						'<div class="topranker-suggestion-text">' + self.escapeHtml( text ) + '</div>' +
						'<div class="topranker-suggestion-meta">' +
							'<span class="topranker-char-count ' + charClass + '">' +
								charsLabel.replace( '%d', charCount ) +
							'</span>' +
						'</div>' +
					'</div>' +
				'</label>';

				container.append( html );
			} );
		},

		/**
		 * Render excerpt result
		 *
		 * @param {string} text Excerpt text.
		 */
		renderExcerptResult: function( text ) {
			this.elements.excerptResult.html(
				'<div class="topranker-suggestion is-selected">' +
					'<div class="topranker-suggestion-content">' +
						'<div class="topranker-suggestion-text">' + this.escapeHtml( text ) + '</div>' +
					'</div>' +
				'</div>'
			);
		},

		/**
		 * Render keyphrases
		 *
		 * @param {Object} keyphrases Keyphrases data.
		 */
		renderKeyphrases: function( keyphrases ) {
			var html = '';
			var primaryLabel = this.config.strings.primary_keyphrase || 'Primary';
			var secondaryLabel = this.config.strings.secondary_keyphrases || 'Secondary';

			// Primary keyphrase.
			if ( keyphrases.primary ) {
				html += '<div class="topranker-keyphrase-primary" id="topranker-keyphrase-primary">' +
					'<div class="topranker-keyphrase-label">' + this.escapeHtml( primaryLabel ) + '</div>' +
					'<div class="topranker-keyphrase-value">' + this.escapeHtml( keyphrases.primary ) + '</div>' +
				'</div>';
			}

			// Secondary keyphrases.
			if ( keyphrases.secondary && keyphrases.secondary.length > 0 ) {
				html += '<div class="topranker-keyphrase-secondary" id="topranker-keyphrase-secondary">' +
					'<div class="topranker-keyphrase-label" style="width: 100%; margin-bottom: 6px;">' + this.escapeHtml( secondaryLabel ) + '</div>';

				keyphrases.secondary.forEach( function( keyphrase ) {
					html += '<span class="topranker-keyphrase-tag">' + this.escapeHtml( keyphrase ) + '</span>';
				}.bind( this ) );

				html += '</div>';
			}

			this.elements.keyphrases.html( html );
		},

		/**
		 * Render social meta fields
		 *
		 * @param {Object} socialMeta Social meta data.
		 */
		renderSocialMeta: function( socialMeta ) {
			if ( socialMeta.og_title ) {
				this.elements.ogTitle.val( socialMeta.og_title );
			}

			if ( socialMeta.og_description ) {
				this.elements.ogDescription.val( socialMeta.og_description );
			}

			if ( socialMeta.twitter_title ) {
				this.elements.twitterTitle.val( socialMeta.twitter_title );
			}

			if ( socialMeta.twitter_description ) {
				this.elements.twitterDescription.val( socialMeta.twitter_description );
			}
		},

		/**
		 * Render alt tag results
		 *
		 * @param {Object} altData Alt tag generation results.
		 */
		renderAltTags: function( altData ) {
			var container = $( '#topranker-alt-tags-results' );
			if ( ! container.length || ! altData.results || ! altData.results.length ) {
				return;
			}

			var html = '';

			altData.results.forEach( function( item ) {
				var filename = item.src ? item.src.split( '/' ).pop() : '';
				var oldAlt = item.old_alt || '';
				var newAlt = item.new_alt || '';
				var featured = item.is_featured ? ' <span class="topranker-alt-tag-badge">' + ( this.config.strings.featured_image || 'Featured' ) + '</span>' : '';

				html += '<div class="topranker-alt-tag-item">' +
					'<div class="topranker-alt-tag-filename">' + this.escapeHtml( filename ) + featured + '</div>';

				if ( oldAlt ) {
					html += '<div class="topranker-alt-tag-old">' + this.escapeHtml( oldAlt ) + '</div>';
				}

				html += '<div class="topranker-alt-tag-new">' + this.escapeHtml( newAlt ) + '</div>' +
				'</div>';
			}.bind( this ) );

			// Summary.
			html += '<div class="topranker-alt-tags-summary">' +
				'<span class="dashicons dashicons-yes-alt" aria-hidden="true"></span> ' +
				altData.success + ' / ' + altData.total + ' ' +
				( this.config.strings.images_processed || 'images processed' ) +
			'</div>';

			container.html( html );

			// Store alt tag results for apply.
			this._altTagResults = altData.results;
		},

		/**
		 * Handle suggestion radio selection
		 *
		 * @param {jQuery} radio The selected radio input.
		 */
		handleSuggestionSelect: function( radio ) {
			var label = radio.closest( '.topranker-suggestion' );
			var container = radio.closest( '.topranker-suggestions' );
			var value = radio.val();

			// Update selection styling.
			container.find( '.topranker-suggestion' ).removeClass( 'is-selected' );
			label.addClass( 'is-selected' );

			// Update corresponding custom field.
			var name = radio.attr( 'name' );

			if ( 'topranker_meta_title_choice' === name ) {
				this.elements.metaTitleCustom.val( value );
				this.updateCharCounter( this.elements.metaTitleCustom, this.elements.metaTitleCount, 60 );
				this.updateSerpPreview();
			} else if ( 'topranker_meta_description_choice' === name ) {
				this.elements.metaDescriptionCustom.val( value );
				this.updateCharCounter( this.elements.metaDescriptionCustom, this.elements.metaDescriptionCount, 155 );
				this.updateSerpPreview();
			}
		},

		/**
		 * Handle apply button click
		 */
		handleApply: function() {
			var self = this;

			if ( this.state.isApplying ) {
				return;
			}

			// Collect selected values.
			var values = this.collectSelectedValues();

			// Check if any values are selected.
			if ( Object.keys( values ).length === 0 ) {
				alert( this.config.strings.select_at_least );
				return;
			}

			this.state.isApplying = true;
			this.disableApplyButton();

			var postId = this.elements.metabox.data( 'post-id' );
			var nonce = this.elements.metabox.data( 'nonce' );

			var requestData = $.extend( {
				action: 'topranker_apply',
				post_id: postId,
				nonce: nonce
			}, values );

			$.ajax( {
				url: this.config.ajax_url,
				type: 'POST',
				data: requestData,
				success: function( response ) {
					if ( response.success ) {
						self.handleApplySuccess( response.data );
					} else {
						self.handleApplyError( response.data );
					}
				},
				error: function( xhr, status, error ) {
					self.handleApplyError( {
						message: self.config.strings.error + ' (' + error + ')'
					} );
				},
				complete: function() {
					self.state.isApplying = false;
					self.enableApplyButton();
				}
			} );
		},

		/**
		 * Collect selected values from the form
		 *
		 * @return {Object} Selected values.
		 */
		collectSelectedValues: function() {
			var values = {};

			// Meta title.
			var metaTitle = this.elements.metaTitleCustom.val().trim();
			if ( metaTitle ) {
				values.meta_title = metaTitle;
			}

			// Meta description.
			var metaDescription = this.elements.metaDescriptionCustom.val().trim();
			if ( metaDescription ) {
				values.meta_description = metaDescription;
			}

			// Excerpt.
			var excerpt = this.elements.excerptCustom.val().trim();
			if ( excerpt ) {
				values.excerpt = excerpt;
			}

			// Focus keyphrase.
			var keyphrase = this.elements.keyphraseCustom.val().trim();
			if ( keyphrase ) {
				values.focus_keyphrase = keyphrase;
			}

			// OG title.
			var ogTitle = this.elements.ogTitle.val().trim();
			if ( ogTitle ) {
				values.og_title = ogTitle;
			}

			// OG description.
			var ogDescription = this.elements.ogDescription.val().trim();
			if ( ogDescription ) {
				values.og_description = ogDescription;
			}

			// Twitter title.
			var twitterTitle = this.elements.twitterTitle.val().trim();
			if ( twitterTitle ) {
				values.twitter_title = twitterTitle;
			}

			// Twitter description.
			var twitterDescription = this.elements.twitterDescription.val().trim();
			if ( twitterDescription ) {
				values.twitter_description = twitterDescription;
			}

			// Secondary keyphrases (from the displayed tags).
			var secondaryKeyphrases = [];
			this.elements.keyphrases.find( '.topranker-keyphrase-tag' ).each( function() {
				secondaryKeyphrases.push( $( this ).text().trim() );
			} );
			if ( secondaryKeyphrases.length > 0 ) {
				values.secondary_keyphrases = JSON.stringify( secondaryKeyphrases );
			}

			// Alt tags (stored from renderAltTags).
			if ( this._altTagResults && this._altTagResults.length > 0 ) {
				values.alt_tags = JSON.stringify( this._altTagResults );
			}

			return values;
		},

		/**
		 * Handle successful apply response
		 *
		 * @param {Object} data Response data.
		 */
		handleApplySuccess: function( data ) {
			// Show success notification.
			this.showTemporaryNotice( this.config.strings.applied, 'success' );

			// Update last optimized time display.
			var applyInfo = this.elements.resultsContainer.find( '.topranker-apply-info' );
			if ( applyInfo.length && data.last_optimized ) {
				var lastOptimizedText = this.config.strings.last_optimized_just_now || 'Last optimized: just now';
				applyInfo.text( lastOptimizedText );
			}
		},

		/**
		 * Handle apply error
		 *
		 * @param {Object} error Error data.
		 */
		handleApplyError: function( error ) {
			var message = error && error.message ? error.message : this.config.strings.error;
			this.showTemporaryNotice( message, 'error' );
		},

		/**
		 * Handle copy to SEO plugin
		 */
		handleCopyToSeoPlugin: function() {
			var self = this;
			var btn = this.elements.copyToSeoBtn;
			var originalText = btn.html();
			var pluginName = btn.data( 'seo-plugin' ) || 'SEO plugin';

			btn.prop( 'disabled', true ).html(
				'<span class="dashicons dashicons-update spin" aria-hidden="true"></span> Copying...'
			);

			var postId = this.elements.metabox.data( 'post-id' );
			var nonce = this.elements.metabox.data( 'nonce' );

			$.ajax( {
				url: this.config.ajax_url,
				type: 'POST',
				data: {
					action: 'topranker_copy_to_seo_plugin',
					nonce: nonce,
					post_id: postId
				},
				success: function( response ) {
					if ( response.success ) {
						self.showTemporaryNotice(
							'Data copied to ' + pluginName + ' successfully!',
							'success'
						);
					} else {
						var msg = response.data && response.data.message
							? response.data.message
							: 'Failed to copy data.';
						self.showTemporaryNotice( msg, 'error' );
					}
				},
				error: function() {
					self.showTemporaryNotice( 'Failed to copy data to ' + pluginName + '.', 'error' );
				},
				complete: function() {
					btn.prop( 'disabled', false ).html( originalText );
				}
			} );
		},

		/**
		 * Initialize SERP preview
		 */
		initSerpPreview: function() {
			this.updateSerpPreview();
		},

		/**
		 * Update SERP preview
		 */
		updateSerpPreview: function() {
			// Get values from custom fields or fallback to original.
			var title = this.elements.metaTitleCustom.val().trim();
			var description = this.elements.metaDescriptionCustom.val().trim();

			// Update SERP preview if elements exist.
			if ( this.elements.serpTitle.length ) {
				if ( title ) {
					this.elements.serpTitle.text( title );
				}
			}

			if ( this.elements.serpDescription.length ) {
				if ( description ) {
					this.elements.serpDescription.text( description );
				}
			}

			// Update SERP character counts.
			if ( this.elements.serpTitleCount.length ) {
				var titleLength = title ? title.length : 0;
				var titleClass = this.getCharCountClass( titleLength, 60 );
				this.elements.serpTitleCount.find( '.count' ).text( titleLength );
				this.elements.serpTitleCount
					.removeClass( 'is-good is-warning is-error' )
					.addClass( titleClass );
			}

			if ( this.elements.serpDescriptionCount.length ) {
				var descLength = description ? description.length : 0;
				var descClass = this.getCharCountClass( descLength, 155, 120 );
				this.elements.serpDescriptionCount.find( '.count' ).text( descLength );
				this.elements.serpDescriptionCount
					.removeClass( 'is-good is-warning is-error' )
					.addClass( descClass );
			}
		},

		/**
		 * Initialize character counters
		 */
		initCharCounters: function() {
			this.updateCharCounter( this.elements.metaTitleCustom, this.elements.metaTitleCount, 60 );
			this.updateCharCounter( this.elements.metaDescriptionCustom, this.elements.metaDescriptionCount, 155 );
			this.updateCharCounter( this.elements.excerptCustom, this.elements.excerptCount, 300 );
		},

		/**
		 * Update character counter for a field
		 *
		 * @param {jQuery}      input      Input element.
		 * @param {jQuery|null} counter    Counter element (optional).
		 * @param {number}      maxLength  Maximum character length.
		 * @param {number}      minOptimal Minimum optimal length (optional).
		 */
		updateCharCounter: function( input, counter, maxLength, minOptimal ) {
			var value = input.val();
			var length = value ? value.length : 0;
			var charClass = this.getCharCountClass( length, maxLength, minOptimal );

			if ( counter && counter.length ) {
				counter.find( '.count' ).text( length );
				counter
					.removeClass( 'is-good is-warning is-error' )
					.addClass( charClass );
			}
		},

		/**
		 * Get character count CSS class
		 *
		 * @param {number} length     Current length.
		 * @param {number} maxLength  Maximum length.
		 * @param {number} minOptimal Minimum optimal length (optional).
		 * @return {string} CSS class name.
		 */
		getCharCountClass: function( length, maxLength, minOptimal ) {
			if ( length > maxLength ) {
				return 'is-error';
			}

			minOptimal = minOptimal || Math.floor( maxLength * 0.83 );

			if ( length < minOptimal ) {
				return 'is-warning';
			}

			return 'is-good';
		},

		/**
		 * Update usage display
		 *
		 * @param {Object} usage Usage data.
		 */
		updateUsageDisplay: function( usage ) {
			var badge = this.elements.metabox.find( '.topranker-usage-badge' );

			if ( ! badge.length || this.config.is_pro ) {
				return;
			}

			badge.text( usage.used + ' / ' + usage.limit + ' this month' );

			badge.removeClass( 'is-warning is-limit' );

			if ( usage.remaining <= 0 ) {
				badge.addClass( 'is-limit' );
				this.disableOptimizeButton();
			} else if ( usage.remaining <= 5 ) {
				badge.addClass( 'is-warning' );
			}
		},

		/**
		 * Show loading state
		 */
		showLoading: function() {
			this.elements.loadingContainer.show();
			this.elements.errorContainer.hide();
			this.elements.resultsContainer.hide();
			this.elements.emptyState.hide();
		},

		/**
		 * Hide loading state
		 */
		hideLoading: function() {
			this.elements.loadingContainer.hide();
		},

		/**
		 * Show error state
		 *
		 * @param {string} message Error message.
		 */
		showError: function( message ) {
			this.elements.errorMessage.text( message );
			this.elements.errorContainer.show();
			this.elements.resultsContainer.hide();
			this.elements.emptyState.hide();
		},

		/**
		 * Hide error state
		 */
		hideError: function() {
			this.elements.errorContainer.hide();
		},

		/**
		 * Show results container
		 */
		showResults: function() {
			this.elements.resultsContainer.show();
		},

		/**
		 * Hide empty state
		 */
		hideEmptyState: function() {
			this.elements.emptyState.hide();
		},

		/**
		 * Disable optimize button
		 */
		disableOptimizeButton: function() {
			this.elements.optimizeBtn.prop( 'disabled', true );
			this.elements.optimizeBtn.find( '.topranker-btn-text' ).text( this.config.strings.optimizing );
		},

		/**
		 * Enable optimize button
		 */
		enableOptimizeButton: function() {
			// Check if usage limit is reached.
			if ( ! this.config.is_pro && this.config.usage && this.config.usage.remaining <= 0 ) {
				return;
			}

			this.elements.optimizeBtn.prop( 'disabled', false );
			this.elements.optimizeBtn.find( '.topranker-btn-text' ).text( this.config.strings.optimize_btn );
		},

		/**
		 * Disable apply button
		 */
		disableApplyButton: function() {
			this.elements.applyBtn.prop( 'disabled', true );
			this.elements.applyBtn.html(
				'<span class="dashicons dashicons-update topranker-spin" aria-hidden="true"></span> ' +
				this.config.strings.applying
			);
		},

		/**
		 * Enable apply button
		 */
		enableApplyButton: function() {
			this.elements.applyBtn.prop( 'disabled', false );
			this.elements.applyBtn.html(
				'<span class="dashicons dashicons-yes" aria-hidden="true"></span> ' +
				this.config.strings.apply_btn
			);
		},

		/**
		 * Show history preview modal (Pro feature)
		 *
		 * @param {number} index History entry index.
		 */
		showHistoryPreview: function( index ) {
			var self = this;
			var postId = this.elements.metabox.data( 'post-id' );
			var nonce = this.elements.metabox.data( 'nonce' );

			// Show loading state in modal.
			this.elements.historyModalTitle.text( this.config.strings.history_loading || 'Loading...' );
			this.elements.historyModalContent.html( '<div class="topranker-history-loading"><span class="spinner is-active"></span></div>' );
			this.elements.historyModalRevert.hide();
			this.elements.historyModal.show();

			$.ajax( {
				url: this.config.ajax_url,
				type: 'POST',
				data: {
					action: 'topranker_get_history',
					post_id: postId,
					nonce: nonce
				},
				success: function( response ) {
					if ( response.success && response.data.history && response.data.history[ index ] ) {
						var entry = response.data.history[ index ];
						self.renderHistoryPreview( entry, index );
					} else {
						self.elements.historyModalTitle.text( self.config.strings.history_error || 'Error' );
						self.elements.historyModalContent.html(
							'<p class="topranker-history-error">' +
							( self.config.strings.history_load_error || 'Failed to load history entry.' ) +
							'</p>'
						);
					}
				},
				error: function() {
					self.elements.historyModalTitle.text( self.config.strings.history_error || 'Error' );
					self.elements.historyModalContent.html(
						'<p class="topranker-history-error">' +
						( self.config.strings.history_load_error || 'Failed to load history entry.' ) +
						'</p>'
					);
				}
			} );
		},

		/**
		 * Render history preview content in modal
		 *
		 * @param {Object} entry History entry data.
		 * @param {number} index History entry index.
		 */
		renderHistoryPreview: function( entry, index ) {
			var timestamp = entry.timestamp || '';
			var dateStr = timestamp ? new Date( timestamp * 1000 ).toLocaleString() : '';

			this.elements.historyModalTitle.text(
				( this.config.strings.history_preview_title || 'Optimization from' ) + ' ' + dateStr
			);

			var html = '<div class="topranker-history-diff">';

			// Meta title.
			if ( entry.data && entry.data.meta_title ) {
				html += '<div class="topranker-diff-section">' +
					'<div class="topranker-diff-label">' + ( this.config.strings.meta_title || 'Meta Title' ) + '</div>' +
					'<div class="topranker-diff-value">' + this.escapeHtml( entry.data.meta_title ) + '</div>' +
				'</div>';
			}

			// Meta description.
			if ( entry.data && entry.data.meta_description ) {
				html += '<div class="topranker-diff-section">' +
					'<div class="topranker-diff-label">' + ( this.config.strings.meta_description || 'Meta Description' ) + '</div>' +
					'<div class="topranker-diff-value">' + this.escapeHtml( entry.data.meta_description ) + '</div>' +
				'</div>';
			}

			// Excerpt.
			if ( entry.data && entry.data.excerpt ) {
				html += '<div class="topranker-diff-section">' +
					'<div class="topranker-diff-label">' + ( this.config.strings.excerpt || 'Excerpt' ) + '</div>' +
					'<div class="topranker-diff-value">' + this.escapeHtml( entry.data.excerpt ) + '</div>' +
				'</div>';
			}

			// Focus keyphrase.
			if ( entry.data && entry.data.focus_keyphrase ) {
				html += '<div class="topranker-diff-section">' +
					'<div class="topranker-diff-label">' + ( this.config.strings.focus_keyphrase || 'Focus Keyphrase' ) + '</div>' +
					'<div class="topranker-diff-value">' + this.escapeHtml( entry.data.focus_keyphrase ) + '</div>' +
				'</div>';
			}

			// OG Title.
			if ( entry.data && entry.data.og_title ) {
				html += '<div class="topranker-diff-section">' +
					'<div class="topranker-diff-label">' + ( this.config.strings.og_title || 'OG Title' ) + '</div>' +
					'<div class="topranker-diff-value">' + this.escapeHtml( entry.data.og_title ) + '</div>' +
				'</div>';
			}

			// OG Description.
			if ( entry.data && entry.data.og_description ) {
				html += '<div class="topranker-diff-section">' +
					'<div class="topranker-diff-label">' + ( this.config.strings.og_description || 'OG Description' ) + '</div>' +
					'<div class="topranker-diff-value">' + this.escapeHtml( entry.data.og_description ) + '</div>' +
				'</div>';
			}

			html += '</div>';

			this.elements.historyModalContent.html( html );
			this.elements.historyModalRevert.data( 'index', index ).show();
		},

		/**
		 * Close history modal
		 */
		closeHistoryModal: function() {
			this.elements.historyModal.hide();
		},

		/**
		 * Confirm and execute history revert
		 *
		 * @param {number} index History entry index.
		 */
		confirmHistoryRevert: function( index ) {
			var confirmMsg = this.config.strings.history_revert_confirm ||
				'Are you sure you want to revert to this optimization? Current values will be replaced.';

			if ( ! confirm( confirmMsg ) ) {
				return;
			}

			this.executeHistoryRevert( index );
		},

		/**
		 * Execute history revert via AJAX
		 *
		 * @param {number} index History entry index.
		 */
		executeHistoryRevert: function( index ) {
			var self = this;

			if ( this.state.isReverting ) {
				return;
			}

			this.state.isReverting = true;

			var postId = this.elements.metabox.data( 'post-id' );
			var nonce = this.elements.metabox.data( 'nonce' );

			// Disable revert buttons.
			this.elements.historyList.find( '.topranker-history-revert-btn' ).prop( 'disabled', true );

			$.ajax( {
				url: this.config.ajax_url,
				type: 'POST',
				data: {
					action: 'topranker_revert_history',
					post_id: postId,
					index: index,
					nonce: nonce
				},
				success: function( response ) {
					if ( response.success ) {
						self.handleRevertSuccess( response.data );
					} else {
						self.handleRevertError( response.data );
					}
				},
				error: function( xhr, status, error ) {
					self.handleRevertError( {
						message: self.config.strings.error + ' (' + error + ')'
					} );
				},
				complete: function() {
					self.state.isReverting = false;
					self.elements.historyList.find( '.topranker-history-revert-btn' ).prop( 'disabled', false );
				}
			} );
		},

		/**
		 * Handle successful revert
		 *
		 * @param {Object} data Response data.
		 */
		handleRevertSuccess: function( data ) {
			// Update form fields with reverted values.
			if ( data.reverted ) {
				if ( data.reverted.meta_title ) {
					this.elements.metaTitleCustom.val( data.reverted.meta_title );
					this.updateCharCounter( this.elements.metaTitleCustom, this.elements.metaTitleCount, 60 );
				}

				if ( data.reverted.meta_description ) {
					this.elements.metaDescriptionCustom.val( data.reverted.meta_description );
					this.updateCharCounter( this.elements.metaDescriptionCustom, this.elements.metaDescriptionCount, 155 );
				}

				if ( data.reverted.excerpt ) {
					this.elements.excerptCustom.val( data.reverted.excerpt );
					this.updateCharCounter( this.elements.excerptCustom, this.elements.excerptCount, 300 );
				}

				if ( data.reverted.focus_keyphrase ) {
					this.elements.keyphraseCustom.val( data.reverted.focus_keyphrase );
				}

				if ( data.reverted.og_title ) {
					this.elements.ogTitle.val( data.reverted.og_title );
				}

				if ( data.reverted.og_description ) {
					this.elements.ogDescription.val( data.reverted.og_description );
				}

				if ( data.reverted.twitter_title ) {
					this.elements.twitterTitle.val( data.reverted.twitter_title );
				}

				if ( data.reverted.twitter_description ) {
					this.elements.twitterDescription.val( data.reverted.twitter_description );
				}

				// Update SERP preview.
				this.updateSerpPreview();
			}

			// Show success message.
			this.showTemporaryNotice(
				this.config.strings.history_reverted || 'Successfully reverted to previous optimization.',
				'success'
			);
		},

		/**
		 * Handle revert error
		 *
		 * @param {Object} error Error data.
		 */
		handleRevertError: function( error ) {
			var message = error && error.message ? error.message :
				( this.config.strings.history_revert_error || 'Failed to revert optimization.' );
			this.showTemporaryNotice( message, 'error' );
		},

		/**
		 * Show temporary notice
		 *
		 * @param {string} message Notice message.
		 * @param {string} type    Notice type (success, error).
		 */
		showTemporaryNotice: function( message, type ) {
			var notice = $( '<div class="topranker-notice is-' + type + '">' +
				'<div class="topranker-notice-icon">' +
					'<span class="dashicons dashicons-' + ( 'success' === type ? 'yes-alt' : 'warning' ) + '" aria-hidden="true"></span>' +
				'</div>' +
				'<div class="topranker-notice-content">' +
					'<div class="topranker-notice-message">' + this.escapeHtml( message ) + '</div>' +
				'</div>' +
			'</div>' );

			this.elements.metabox.prepend( notice );

			// Auto-remove after 3 seconds.
			setTimeout( function() {
				notice.fadeOut( 300, function() {
					$( this ).remove();
				} );
			}, 3000 );
		},

		/**
		 * Escape HTML special characters
		 *
		 * @param {string} text Text to escape.
		 * @return {string} Escaped text.
		 */
		escapeHtml: function( text ) {
			if ( ! text ) {
				return '';
			}

			var div = document.createElement( 'div' );
			div.appendChild( document.createTextNode( text ) );
			return div.innerHTML;
		}
	};

	// Initialize on DOM ready.
	$( document ).ready( function() {
		TopRankerEditor.init();
	} );

} )( jQuery );
