/**
 * TopRanker AI Media Library Integration.
 *
 * Handles bulk and single image alt tag generation in the Media Library.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

/* global jQuery, toprankerMedia */

(function($) {
	'use strict';

	var TopRankerMediaLibrary = {
		/**
		 * Processing state.
		 */
		isProcessing: false,
		isCancelled: false,
		currentBatch: [],
		currentIndex: 0,
		successCount: 0,
		failedCount: 0,
		results: [],

		/**
		 * Initialize the module.
		 */
		init: function() {
			this.bindEvents();

			// Check if we should show the bulk modal.
			if (toprankerMedia.showModal && toprankerMedia.batchIds.length > 0) {
				this.showBulkModal(toprankerMedia.batchIds);
			}
		},

		/**
		 * Bind event handlers.
		 */
		bindEvents: function() {
			var self = this;

			// Row action clicks.
			$(document).on('click', '.topranker-generate-alt', function(e) {
				e.preventDefault();
				if ($(this).hasClass('is-loading')) {
					return;
				}
				self.handleSingleGenerate($(this));
			});

			// Modal close.
			$(document).on('click', '.topranker-media-modal-close, .topranker-media-modal-backdrop', function() {
				if (!self.isProcessing) {
					self.closeModal();
				}
			});

			// Cancel button.
			$(document).on('click', '#topranker-media-cancel', function() {
				if (self.isProcessing) {
					self.isCancelled = true;
					$(this).prop('disabled', true).text(toprankerMedia.i18n.cancel + '...');
				} else {
					self.closeModal();
				}
			});

			// Proceed button.
			$(document).on('click', '#topranker-media-proceed', function() {
				self.startBulkProcessing();
			});

			// Done button (redirect).
			$(document).on('click', '#topranker-media-done', function() {
				var successCount = self.successCount;
				self.closeModal();
				// Redirect with success count.
				window.location.href = self.getRedirectUrl(successCount);
			});

			// Apply alt buttons in results.
			$(document).on('click', '.topranker-apply-alt-btn', function() {
				self.applyAlt($(this));
			});

			// Escape key to close.
			$(document).on('keydown', function(e) {
				if (e.key === 'Escape' && !self.isProcessing) {
					self.closeModal();
				}
			});
		},

		/**
		 * Handle single image alt generation from row action.
		 *
		 * @param {jQuery} $link The clicked link.
		 */
		handleSingleGenerate: function($link) {
			var self = this;
			var attachmentId = $link.data('attachment-id');
			var nonce = $link.data('nonce');

			// Add loading state.
			$link.addClass('is-loading').text(toprankerMedia.i18n.generating);

			// Remove any previous result.
			$link.siblings('.topranker-alt-result').remove();

			$.ajax({
				url: toprankerMedia.ajaxUrl,
				type: 'POST',
				data: {
					action: 'topranker_media_single_alt',
					attachment_id: attachmentId,
					nonce: nonce
				},
				success: function(response) {
					$link.removeClass('is-loading');

					if (response.success) {
						var result = response.data;
						// Show success with the alt text and apply button.
						$link.text(toprankerMedia.i18n.applyAlt);
						$link.after(
							'<span class="topranker-alt-result is-success">' +
							self.escapeHtml(result.new_alt.substring(0, 50)) +
							(result.new_alt.length > 50 ? '...' : '') +
							' <button type="button" class="button button-small topranker-apply-single-alt" ' +
							'data-attachment-id="' + attachmentId + '" ' +
							'data-alt-text="' + self.escapeHtml(result.new_alt) + '">' +
							toprankerMedia.i18n.applyAlt +
							'</button></span>'
						);

						// Bind apply button.
						$link.siblings('.topranker-alt-result').find('.topranker-apply-single-alt').on('click', function() {
							self.applySingleAlt($(this), attachmentId, result.new_alt);
						});

						$link.text(toprankerMedia.i18n.applyAlt.replace('Apply', 'Regenerate'));
					} else {
						$link.text('Generate Alt');
						$link.after(
							'<span class="topranker-alt-result is-error">' +
							self.escapeHtml(response.data.message) +
							'</span>'
						);
					}
				},
				error: function() {
					$link.removeClass('is-loading').text('Generate Alt');
					$link.after(
						'<span class="topranker-alt-result is-error">' +
						toprankerMedia.i18n.networkError +
						'</span>'
					);
				}
			});
		},

		/**
		 * Apply single alt text from row action result.
		 *
		 * @param {jQuery} $button     The apply button.
		 * @param {number} attachmentId Attachment ID.
		 * @param {string} altText     Alt text to apply.
		 */
		applySingleAlt: function($button, attachmentId, altText) {
			var self = this;

			$button.prop('disabled', true).text(toprankerMedia.i18n.processing);

			$.ajax({
				url: toprankerMedia.ajaxUrl,
				type: 'POST',
				data: {
					action: 'topranker_media_apply_alt',
					attachment_id: attachmentId,
					alt_text: altText,
					nonce: toprankerMedia.nonce
				},
				success: function(response) {
					if (response.success) {
						$button.parent().removeClass('is-success').html(
							'<span class="dashicons dashicons-yes" style="color: #00a32a;"></span> ' +
							toprankerMedia.i18n.applied
						);
					} else {
						$button.prop('disabled', false).text(toprankerMedia.i18n.applyAlt);
						alert(response.data.message || toprankerMedia.i18n.errorApply);
					}
				},
				error: function() {
					$button.prop('disabled', false).text(toprankerMedia.i18n.applyAlt);
					alert(toprankerMedia.i18n.networkError);
				}
			});
		},

		/**
		 * Show the bulk processing modal.
		 *
		 * @param {Array} batchIds Array of attachment IDs to process.
		 */
		showBulkModal: function(batchIds) {
			this.currentBatch = batchIds;
			this.currentIndex = 0;
			this.successCount = 0;
			this.failedCount = 0;
			this.results = [];

			var count = batchIds.length;
			var estimatedCost = '$' + (count * 0.01).toFixed(2) + ' - $' + (count * 0.03).toFixed(2);

			var modalHtml = this.getModalHtml(count, estimatedCost);
			$('body').append(modalHtml);

			// Focus the modal for accessibility.
			$('.topranker-media-modal-content').attr('tabindex', '-1').focus();
		},

		/**
		 * Get the modal HTML.
		 *
		 * @param {number} count         Number of images to process.
		 * @param {string} estimatedCost Estimated cost string.
		 * @return {string} Modal HTML.
		 */
		getModalHtml: function(count, estimatedCost) {
			return '<div id="topranker-media-modal" class="topranker-media-modal">' +
				'<div class="topranker-media-modal-backdrop"></div>' +
				'<div class="topranker-media-modal-content" role="dialog" aria-modal="true" aria-labelledby="topranker-modal-title">' +
					'<div class="topranker-media-modal-header">' +
						'<h2 id="topranker-modal-title">' + toprankerMedia.i18n.modalTitle + '</h2>' +
						'<button type="button" class="topranker-media-modal-close" aria-label="' + toprankerMedia.i18n.close + '">&times;</button>' +
					'</div>' +
					'<div class="topranker-media-modal-body">' +
						'<div id="topranker-media-confirm" class="topranker-media-confirm">' +
							'<p><strong>' + count + '</strong> ' + (count === 1 ? toprankerMedia.i18n.imageAlt : toprankerMedia.i18n.imagesAlt) + ' selected</p>' +
							'<p><strong>' + toprankerMedia.i18n.estimatedCalls + '</strong> ' + count + '</p>' +
							'<p><strong>' + toprankerMedia.i18n.estimatedCost + '</strong> ' + estimatedCost + '</p>' +
						'</div>' +
						'<div id="topranker-media-progress" class="topranker-media-progress" style="display: none;">' +
							'<div class="topranker-media-progress-header">' +
								'<span id="topranker-progress-title">' + toprankerMedia.i18n.processing + '</span>' +
								'<span><span id="topranker-progress-current">0</span> ' + toprankerMedia.i18n.of + ' <span id="topranker-progress-total">' + count + '</span></span>' +
							'</div>' +
							'<div class="topranker-media-progress-bar">' +
								'<div id="topranker-progress-fill" class="topranker-media-progress-fill" style="width: 0%;"></div>' +
							'</div>' +
							'<div id="topranker-progress-text" class="topranker-media-progress-text"></div>' +
						'</div>' +
						'<div id="topranker-media-results" class="topranker-media-results" style="display: none;"></div>' +
					'</div>' +
					'<div class="topranker-media-modal-footer">' +
						'<button type="button" id="topranker-media-cancel" class="button">' + toprankerMedia.i18n.confirmCancel + '</button>' +
						'<button type="button" id="topranker-media-proceed" class="button button-primary">' + toprankerMedia.i18n.confirmProceed + '</button>' +
						'<button type="button" id="topranker-media-done" class="button button-primary" style="display: none;">' + toprankerMedia.i18n.done + '</button>' +
					'</div>' +
				'</div>' +
			'</div>';
		},

		/**
		 * Close the modal.
		 */
		closeModal: function() {
			$('#topranker-media-modal').remove();
			this.isProcessing = false;
			this.isCancelled = false;
		},

		/**
		 * Start bulk processing.
		 */
		startBulkProcessing: function() {
			this.isProcessing = true;
			this.isCancelled = false;

			// Hide confirm, show progress.
			$('#topranker-media-confirm').hide();
			$('#topranker-media-progress').show();
			$('#topranker-media-results').show();

			// Update buttons.
			$('#topranker-media-proceed').hide();
			$('#topranker-media-cancel').text(toprankerMedia.i18n.cancel);

			// Start processing.
			this.processNext();
		},

		/**
		 * Process the next image in the queue.
		 */
		processNext: function() {
			var self = this;

			if (this.isCancelled || this.currentIndex >= this.currentBatch.length) {
				this.finishProcessing();
				return;
			}

			var attachmentId = this.currentBatch[this.currentIndex];

			// Update progress.
			$('#topranker-progress-current').text(this.currentIndex + 1);
			var percent = Math.round(((this.currentIndex + 1) / this.currentBatch.length) * 100);
			$('#topranker-progress-fill').css('width', percent + '%');
			$('#topranker-progress-text').text(toprankerMedia.i18n.generating);

			// Add processing result row.
			this.addResultRow(attachmentId, 'processing');

			$.ajax({
				url: toprankerMedia.ajaxUrl,
				type: 'POST',
				data: {
					action: 'topranker_media_bulk_alt',
					attachment_id: attachmentId,
					nonce: toprankerMedia.nonce
				},
				success: function(response) {
					if (response.success) {
						self.successCount++;
						self.results.push({
							id: attachmentId,
							success: true,
							data: response.data
						});
						self.updateResultRow(attachmentId, 'success', response.data);
					} else {
						self.failedCount++;
						self.results.push({
							id: attachmentId,
							success: false,
							error: response.data.message
						});
						self.updateResultRow(attachmentId, 'error', null, response.data.message);
					}
				},
				error: function() {
					self.failedCount++;
					self.results.push({
						id: attachmentId,
						success: false,
						error: toprankerMedia.i18n.networkError
					});
					self.updateResultRow(attachmentId, 'error', null, toprankerMedia.i18n.networkError);
				},
				complete: function() {
					self.currentIndex++;
					self.processNext();
				}
			});
		},

		/**
		 * Add a result row to the results list.
		 *
		 * @param {number} attachmentId Attachment ID.
		 * @param {string} status       Status (processing, success, error).
		 */
		addResultRow: function(attachmentId, status) {
			var statusText = status === 'processing' ? toprankerMedia.i18n.processing : '';
			var statusIcon = status === 'processing' ? 'dashicons-update spin' : '';

			var html = '<div id="topranker-result-' + attachmentId + '" class="topranker-media-result is-' + status + '" data-attachment-id="' + attachmentId + '">' +
				'<img class="topranker-media-result-thumb" src="" alt="" />' +
				'<div class="topranker-media-result-info">' +
					'<div class="topranker-media-result-title">ID: ' + attachmentId + '</div>' +
					'<div class="topranker-media-result-alt"></div>' +
				'</div>' +
				'<div class="topranker-media-result-status">' +
					'<span class="dashicons ' + statusIcon + '"></span> ' + statusText +
				'</div>' +
			'</div>';

			$('#topranker-media-results').append(html);

			// Scroll to bottom.
			var $results = $('#topranker-media-results');
			$results.scrollTop($results[0].scrollHeight);
		},

		/**
		 * Update a result row.
		 *
		 * @param {number}      attachmentId Attachment ID.
		 * @param {string}      status       Status (success, error).
		 * @param {Object|null} data         Response data for success.
		 * @param {string|null} errorMsg     Error message.
		 */
		updateResultRow: function(attachmentId, status, data, errorMsg) {
			var $row = $('#topranker-result-' + attachmentId);
			$row.removeClass('is-processing').addClass('is-' + status);

			var statusIcon = status === 'success' ? 'dashicons-yes' : 'dashicons-no';
			var statusText = status === 'success' ? '' : (errorMsg || toprankerMedia.i18n.error);

			if (status === 'success' && data) {
				// Update thumb and title.
				if (data.src) {
					$row.find('.topranker-media-result-thumb').attr('src', data.src);
				}
				if (data.new_alt) {
					$row.find('.topranker-media-result-alt').text(data.new_alt);
					// Add apply button.
					$row.find('.topranker-media-result-status').html(
						'<button type="button" class="button button-small topranker-apply-alt-btn" ' +
						'data-attachment-id="' + attachmentId + '" ' +
						'data-alt-text="' + this.escapeHtml(data.new_alt) + '">' +
						toprankerMedia.i18n.applyAlt +
						'</button>'
					);
				}
			} else {
				$row.find('.topranker-media-result-status').html(
					'<span class="dashicons ' + statusIcon + '"></span> ' + this.escapeHtml(statusText)
				);
			}
		},

		/**
		 * Apply alt text from bulk results.
		 *
		 * @param {jQuery} $button The apply button.
		 */
		applyAlt: function($button) {
			var self = this;
			var attachmentId = $button.data('attachment-id');
			var altText = $button.data('alt-text');

			$button.prop('disabled', true).text(toprankerMedia.i18n.processing);

			$.ajax({
				url: toprankerMedia.ajaxUrl,
				type: 'POST',
				data: {
					action: 'topranker_media_apply_alt',
					attachment_id: attachmentId,
					alt_text: altText,
					nonce: toprankerMedia.nonce
				},
				success: function(response) {
					if (response.success) {
						$button.replaceWith(
							'<span class="dashicons dashicons-yes" style="color: #00a32a;"></span> ' +
							toprankerMedia.i18n.applied
						);
					} else {
						$button.prop('disabled', false).text(toprankerMedia.i18n.applyAlt);
						alert(response.data.message || toprankerMedia.i18n.errorApply);
					}
				},
				error: function() {
					$button.prop('disabled', false).text(toprankerMedia.i18n.applyAlt);
					alert(toprankerMedia.i18n.networkError);
				}
			});
		},

		/**
		 * Finish bulk processing.
		 */
		finishProcessing: function() {
			this.isProcessing = false;

			// Update progress.
			$('#topranker-progress-fill').css('width', '100%');

			var statusText = this.isCancelled ? toprankerMedia.i18n.cancelled : toprankerMedia.i18n.complete;
			statusText += ' ' + this.successCount + ' ' + toprankerMedia.i18n.optimized;
			if (this.failedCount > 0) {
				statusText += ', ' + this.failedCount + ' ' + toprankerMedia.i18n.failed;
			}

			$('#topranker-progress-title').text(statusText);
			$('#topranker-progress-text').text('');

			// Update buttons.
			$('#topranker-media-cancel').hide();
			$('#topranker-media-done').show();

			// Apply all successful results automatically.
			if (this.successCount > 0) {
				this.applyAllSuccessful();
			}
		},

		/**
		 * Apply all successful alt texts automatically.
		 */
		applyAllSuccessful: function() {
			var self = this;

			// Get all apply buttons.
			$('.topranker-apply-alt-btn').each(function() {
				self.applyAlt($(this));
			});
		},

		/**
		 * Get redirect URL after processing.
		 *
		 * @param {number} successCount Number of successful generations.
		 * @return {string} Redirect URL.
		 */
		getRedirectUrl: function(successCount) {
			var url = window.location.href;
			// Remove existing topranker params.
			url = url.replace(/[?&]topranker_media_alt=[^&]*/g, '');
			url = url.replace(/[?&]topranker_batch_count=[^&]*/g, '');
			url = url.replace(/[?&]topranker_success=[^&]*/g, '');
			url = url.replace(/[?&]topranker_error=[^&]*/g, '');

			// Clean up double && or ?&.
			url = url.replace(/\?&/, '?').replace(/&&/g, '&').replace(/[?&]$/, '');

			// Add success count.
			if (successCount > 0) {
				url += (url.indexOf('?') === -1 ? '?' : '&') + 'topranker_success=' + successCount;
			}

			return url;
		},

		/**
		 * Escape HTML entities.
		 *
		 * @param {string} text Text to escape.
		 * @return {string} Escaped text.
		 */
		escapeHtml: function(text) {
			if (!text) {
				return '';
			}
			var map = {
				'&': '&amp;',
				'<': '&lt;',
				'>': '&gt;',
				'"': '&quot;',
				"'": '&#039;'
			};
			return String(text).replace(/[&<>"']/g, function(m) { return map[m]; });
		}
	};

	// Initialize on document ready.
	$(document).ready(function() {
		TopRankerMediaLibrary.init();
	});

})(jQuery);
