<?php
/**
 * Bulk Optimization page template (Pro).
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Ensure Pro feature.
if ( ! topranker_is_pro() ) {
	wp_die( esc_html__( 'This feature requires TopRanker Pro.', 'topranker-ai' ) );
}

// Check if bulk class exists.
if ( ! class_exists( 'TopRanker_Bulk' ) ) {
	wp_die( esc_html__( 'Bulk optimization is not available.', 'topranker-ai' ) );
}

// Get query params.
// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Display only.
$post_type = isset( $_GET['tr_type'] ) ? sanitize_key( $_GET['tr_type'] ) : '';
// phpcs:ignore WordPress.Security.NonceVerification.Recommended
$paged = isset( $_GET['paged'] ) ? absint( $_GET['paged'] ) : 1;
// phpcs:ignore WordPress.Security.NonceVerification.Recommended
$filter = isset( $_GET['filter'] ) ? sanitize_key( $_GET['filter'] ) : 'all';
// phpcs:ignore WordPress.Security.NonceVerification.Recommended
$search = isset( $_GET['s'] ) ? sanitize_text_field( wp_unslash( $_GET['s'] ) ) : '';

// Get bulk instance and view data.
$bulk = new TopRanker_Bulk();
$data = $bulk->get_view_data(
	array(
		'post_type' => $post_type,
		'paged'     => $paged,
		'filter'    => $filter,
		's'         => $search,
	)
);

$posts_data        = $data['posts_data'];
$counts            = $data['counts'];
$post_type_options = $data['post_type_options'];
$current_type      = $data['current_type'];
$nonce             = $data['nonce'];

// Build base URL for pagination/filtering.
$base_url = add_query_arg(
	array(
		'page'      => 'topranker-ai-bulk',
		'tr_type' => $current_type,
	),
	admin_url( 'admin.php' )
);
?>
<div class="wrap topranker-bulk-wrap">
	<div class="topranker-bulk-header">
		<h1 class="topranker-bulk-title">
			<span class="dashicons dashicons-update" aria-hidden="true"></span>
			<?php esc_html_e( 'Bulk Optimize', 'topranker-ai' ); ?>
		</h1>
	</div>

	<!-- Quick Stats -->
	<div class="topranker-bulk-stats">
		<div class="topranker-bulk-stat">
			<span class="topranker-bulk-stat-label"><?php esc_html_e( 'Total', 'topranker-ai' ); ?></span>
			<span class="topranker-bulk-stat-value"><?php echo esc_html( number_format_i18n( $counts['total'] ) ); ?></span>
		</div>
		<div class="topranker-bulk-stat">
			<span class="topranker-bulk-stat-label"><?php esc_html_e( 'Optimized', 'topranker-ai' ); ?></span>
			<span class="topranker-bulk-stat-value is-good"><?php echo esc_html( number_format_i18n( $counts['with_meta'] ) ); ?></span>
		</div>
		<div class="topranker-bulk-stat">
			<span class="topranker-bulk-stat-label"><?php esc_html_e( 'Need Optimization', 'topranker-ai' ); ?></span>
			<span class="topranker-bulk-stat-value is-warning"><?php echo esc_html( number_format_i18n( $counts['missing_meta'] ) ); ?></span>
		</div>
		<div class="topranker-bulk-stat">
			<span class="topranker-bulk-stat-label"><?php esc_html_e( 'Coverage', 'topranker-ai' ); ?></span>
			<span class="topranker-bulk-stat-value"><?php echo esc_html( $counts['percent'] ); ?>%</span>
		</div>
	</div>

	<!-- Filters and Search -->
	<div class="topranker-bulk-filters">
		<form method="get" action="<?php echo esc_url( admin_url( 'admin.php' ) ); ?>" class="topranker-bulk-filter-form">
			<input type="hidden" name="page" value="topranker-ai-bulk" />

			<!-- Post Type Selector -->
			<select name="tr_type" id="topranker-post-type-filter" class="topranker-filter-select">
				<?php foreach ( $post_type_options as $type => $label ) : ?>
					<option value="<?php echo esc_attr( $type ); ?>" <?php selected( $current_type, $type ); ?>>
						<?php echo esc_html( $label ); ?>
					</option>
				<?php endforeach; ?>
			</select>

			<!-- Filter Selector -->
			<select name="filter" id="topranker-filter" class="topranker-filter-select">
				<option value="all" <?php selected( $filter, 'all' ); ?>><?php esc_html_e( 'All Posts', 'topranker-ai' ); ?></option>
				<option value="missing_meta" <?php selected( $filter, 'missing_meta' ); ?>><?php esc_html_e( 'Missing Meta', 'topranker-ai' ); ?></option>
			</select>

			<!-- Search -->
			<input
				type="search"
				name="s"
				id="topranker-search"
				class="topranker-filter-search"
				value="<?php echo esc_attr( $search ); ?>"
				placeholder="<?php esc_attr_e( 'Search posts...', 'topranker-ai' ); ?>"
			/>

			<button type="submit" class="button">
				<?php esc_html_e( 'Filter', 'topranker-ai' ); ?>
			</button>
		</form>

		<div class="topranker-bulk-actions">
			<span class="topranker-bulk-select-info" id="topranker-select-info">
				<span id="topranker-selected-count">0</span> <?php esc_html_e( 'selected', 'topranker-ai' ); ?>
			</span>
			<button
				type="button"
				id="topranker-bulk-optimize-btn"
				class="button button-primary"
				disabled
				aria-label="<?php esc_attr_e( 'Optimize selected posts', 'topranker-ai' ); ?>"
			>
				<span class="dashicons dashicons-update" aria-hidden="true"></span>
				<?php esc_html_e( 'Bulk Optimize Selected', 'topranker-ai' ); ?>
			</button>
		</div>
	</div>

	<!-- Progress Panel (hidden by default) -->
	<div id="topranker-bulk-progress" class="topranker-bulk-progress" style="display: none;">
		<div class="topranker-bulk-progress-header">
			<span class="topranker-bulk-progress-title"><?php esc_html_e( 'Optimization in Progress', 'topranker-ai' ); ?></span>
			<span class="topranker-bulk-progress-stats">
				<span id="topranker-progress-current">0</span> / <span id="topranker-progress-total">0</span>
			</span>
		</div>
		<div class="topranker-bulk-progress-bar">
			<div id="topranker-progress-fill" class="topranker-bulk-progress-fill" style="width: 0%;"></div>
		</div>
		<div class="topranker-bulk-progress-text" id="topranker-progress-text">
			<?php esc_html_e( 'Preparing...', 'topranker-ai' ); ?>
		</div>
		<div class="topranker-bulk-progress-actions">
			<button
				type="button"
				id="topranker-cancel-btn"
				class="button button-secondary"
			>
				<?php esc_html_e( 'Cancel', 'topranker-ai' ); ?>
			</button>
		</div>
		<div id="topranker-bulk-results" class="topranker-bulk-results" style="display: none;"></div>
	</div>

	<!-- Confirmation Modal (hidden by default) -->
	<div id="topranker-confirm-modal" class="topranker-modal" style="display: none;">
		<div class="topranker-modal-backdrop"></div>
		<div class="topranker-modal-content">
			<div class="topranker-modal-header">
				<h2><?php esc_html_e( 'Confirm Bulk Optimization', 'topranker-ai' ); ?></h2>
			</div>
			<div class="topranker-modal-body">
				<p>
					<?php esc_html_e( 'You are about to optimize:', 'topranker-ai' ); ?>
					<strong id="topranker-confirm-count">0</strong> <?php esc_html_e( 'posts', 'topranker-ai' ); ?>
				</p>
				<div class="topranker-modal-estimate">
					<p>
						<span class="dashicons dashicons-info" aria-hidden="true"></span>
						<strong><?php esc_html_e( 'Estimated API Calls:', 'topranker-ai' ); ?></strong>
						<span id="topranker-confirm-api-calls">0</span>
					</p>
					<p>
						<strong><?php esc_html_e( 'Estimated Cost:', 'topranker-ai' ); ?></strong>
						<span id="topranker-confirm-cost">$0.00 - $0.00</span>
					</p>
				</div>
				<p class="topranker-modal-note">
					<?php esc_html_e( 'The first suggestion will be auto-applied for each field. You can edit individual posts later.', 'topranker-ai' ); ?>
				</p>
			</div>
			<div class="topranker-modal-footer">
				<button type="button" id="topranker-confirm-cancel" class="button button-secondary">
					<?php esc_html_e( 'Cancel', 'topranker-ai' ); ?>
				</button>
				<button type="button" id="topranker-confirm-proceed" class="button button-primary">
					<?php esc_html_e( 'Proceed', 'topranker-ai' ); ?>
				</button>
			</div>
		</div>
	</div>

	<!-- Posts Table -->
	<?php if ( empty( $posts_data['posts'] ) ) : ?>
		<div class="topranker-notice is-info">
			<span class="topranker-notice-icon">
				<span class="dashicons dashicons-info" aria-hidden="true"></span>
			</span>
			<div class="topranker-notice-content">
				<p class="topranker-notice-title"><?php esc_html_e( 'No posts found', 'topranker-ai' ); ?></p>
				<p class="topranker-notice-message">
					<?php
					if ( ! empty( $search ) || 'all' !== $filter ) {
						esc_html_e( 'No posts match your current filters. Try adjusting the filters.', 'topranker-ai' );
					} else {
						esc_html_e( 'Publish some posts to start optimizing.', 'topranker-ai' );
					}
					?>
				</p>
			</div>
		</div>
	<?php else : ?>
		<div class="topranker-bulk-table-wrap">
			<table class="topranker-bulk-table widefat striped">
				<thead>
					<tr>
						<td id="cb" class="manage-column column-cb check-column">
							<label class="screen-reader-text" for="cb-select-all"><?php esc_html_e( 'Select All', 'topranker-ai' ); ?></label>
							<input id="cb-select-all" type="checkbox" />
						</td>
						<th scope="col" class="column-title"><?php esc_html_e( 'Title', 'topranker-ai' ); ?></th>
						<th scope="col" class="column-status"><?php esc_html_e( 'SEO Status', 'topranker-ai' ); ?></th>
						<th scope="col" class="column-meta"><?php esc_html_e( 'Meta', 'topranker-ai' ); ?></th>
						<th scope="col" class="column-alt"><?php esc_html_e( 'Alt Tags', 'topranker-ai' ); ?></th>
						<th scope="col" class="column-score"><?php esc_html_e( 'SEO Score', 'topranker-ai' ); ?></th>
						<th scope="col" class="column-optimized"><?php esc_html_e( 'Last Optimized', 'topranker-ai' ); ?></th>
					</tr>
				</thead>
				<tbody id="topranker-posts-tbody">
					<?php foreach ( $posts_data['posts'] as $post_item ) : ?>
						<tr id="post-<?php echo esc_attr( $post_item['ID'] ); ?>" data-post-id="<?php echo esc_attr( $post_item['ID'] ); ?>">
							<th scope="row" class="check-column">
								<label class="screen-reader-text" for="cb-select-<?php echo esc_attr( $post_item['ID'] ); ?>">
									<?php
									/* translators: %s: Post title */
									printf( esc_html__( 'Select %s', 'topranker-ai' ), esc_html( $post_item['title'] ) );
									?>
								</label>
								<input
									type="checkbox"
									id="cb-select-<?php echo esc_attr( $post_item['ID'] ); ?>"
									class="topranker-post-checkbox"
									value="<?php echo esc_attr( $post_item['ID'] ); ?>"
								/>
							</th>
							<td class="column-title">
								<a href="<?php echo esc_url( $post_item['edit_link'] ); ?>" class="row-title">
									<?php echo esc_html( $post_item['title'] ); ?>
								</a>
								<div class="row-actions">
									<span class="edit">
										<a href="<?php echo esc_url( $post_item['edit_link'] ); ?>">
											<?php esc_html_e( 'Edit', 'topranker-ai' ); ?>
										</a> |
									</span>
									<span class="optimize">
										<a href="<?php echo esc_url( add_query_arg( 'topranker_optimize', '1', $post_item['edit_link'] ) ); ?>">
											<?php esc_html_e( 'Optimize', 'topranker-ai' ); ?>
										</a>
									</span>
								</div>
							</td>
							<td class="column-status">
								<?php
								$status_class = 'is-missing';
								$status_text  = __( 'Missing', 'topranker-ai' );
								if ( 'complete' === $post_item['seo_status'] ) {
									$status_class = 'is-complete';
									$status_text  = __( 'Complete', 'topranker-ai' );
								} elseif ( 'partial' === $post_item['seo_status'] ) {
									$status_class = 'is-partial';
									$status_text  = __( 'Partial', 'topranker-ai' );
								}
								?>
								<span class="topranker-seo-status <?php echo esc_attr( $status_class ); ?>">
									<?php echo esc_html( $status_text ); ?>
								</span>
							</td>
							<td class="column-meta">
								<span class="topranker-meta-indicators">
									<span
										class="topranker-meta-indicator <?php echo $post_item['has_meta_title'] ? 'is-set' : 'is-missing'; ?>"
										title="<?php esc_attr_e( 'Meta Title', 'topranker-ai' ); ?>"
									>
										T
									</span>
									<span
										class="topranker-meta-indicator <?php echo $post_item['has_meta_desc'] ? 'is-set' : 'is-missing'; ?>"
										title="<?php esc_attr_e( 'Meta Description', 'topranker-ai' ); ?>"
									>
										D
									</span>
									<span
										class="topranker-meta-indicator <?php echo $post_item['has_keyphrase'] ? 'is-set' : 'is-missing'; ?>"
										title="<?php esc_attr_e( 'Focus Keyphrase', 'topranker-ai' ); ?>"
									>
										K
									</span>
								</span>
							</td>
							<td class="column-alt">
								<?php if ( $post_item['images_missing_alt'] > 0 ) : ?>
									<span class="topranker-alt-missing">
										<?php echo esc_html( $post_item['images_missing_alt'] ); ?>
										<?php esc_html_e( 'missing', 'topranker-ai' ); ?>
									</span>
								<?php else : ?>
									<span class="topranker-alt-complete">
										<span class="dashicons dashicons-yes" aria-hidden="true"></span>
									</span>
								<?php endif; ?>
							</td>
							<td class="column-score">
								<?php if ( null !== $post_item['seo_score'] ) : ?>
									<?php
									$score       = $post_item['seo_score'];
									$score_class = 'score-low';
									if ( $score >= 80 ) {
										$score_class = 'score-high';
									} elseif ( $score >= 50 ) {
										$score_class = 'score-medium';
									}
									?>
									<span class="topranker-seo-score <?php echo esc_attr( $score_class ); ?>">
										<?php echo esc_html( $score ); ?>
									</span>
								<?php else : ?>
									<span class="topranker-no-score">—</span>
								<?php endif; ?>
							</td>
							<td class="column-optimized">
								<?php if ( $post_item['last_optimized'] ) : ?>
									<span class="topranker-last-optimized">
										<?php echo esc_html( $post_item['last_optimized'] ); ?>
									</span>
								<?php else : ?>
									<span class="topranker-never-optimized"><?php esc_html_e( 'Never', 'topranker-ai' ); ?></span>
								<?php endif; ?>
							</td>
						</tr>
					<?php endforeach; ?>
				</tbody>
			</table>
		</div>

		<!-- Pagination -->
		<?php if ( $posts_data['total_pages'] > 1 ) : ?>
			<div class="topranker-bulk-pagination tablenav bottom">
				<div class="tablenav-pages">
					<span class="displaying-num">
						<?php
						printf(
							/* translators: %s: Number of posts */
							esc_html( _n( '%s item', '%s items', $posts_data['total_posts'], 'topranker-ai' ) ),
							esc_html( number_format_i18n( $posts_data['total_posts'] ) )
						);
						?>
					</span>
					<span class="pagination-links">
						<?php
						$pagination_args = array(
							'base'      => add_query_arg( 'paged', '%#%', $base_url ),
							'format'    => '',
							'current'   => $posts_data['current_page'],
							'total'     => $posts_data['total_pages'],
							'prev_text' => '&lsaquo;',
							'next_text' => '&rsaquo;',
						);

						// Add filter to base URL.
						if ( 'all' !== $filter ) {
							$pagination_args['base'] = add_query_arg( 'filter', $filter, $pagination_args['base'] );
						}
						if ( ! empty( $search ) ) {
							$pagination_args['base'] = add_query_arg( 's', rawurlencode( $search ), $pagination_args['base'] );
						}

						// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- paginate_links() output is safe.
						echo paginate_links( $pagination_args );
						?>
					</span>
				</div>
			</div>
		<?php endif; ?>
	<?php endif; ?>
</div>

<style>
/* Bulk Page Specific Styles */
.topranker-bulk-title {
	display: flex;
	align-items: center;
	gap: 8px;
}

.topranker-bulk-title .dashicons {
	font-size: 28px;
	width: 28px;
	height: 28px;
	color: #2271b1;
}

.topranker-bulk-stats {
	display: flex;
	gap: 20px;
	margin: 20px 0;
	padding: 15px 20px;
	background: #fff;
	border: 1px solid #c3c4c7;
	border-radius: 4px;
}

.topranker-bulk-stat {
	display: flex;
	flex-direction: column;
	gap: 4px;
}

.topranker-bulk-stat-label {
	font-size: 12px;
	color: #646970;
	text-transform: uppercase;
	letter-spacing: 0.5px;
}

.topranker-bulk-stat-value {
	font-size: 24px;
	font-weight: 700;
	color: #1d2327;
}

.topranker-bulk-stat-value.is-good {
	color: #00a32a;
}

.topranker-bulk-stat-value.is-warning {
	color: #dba617;
}

.topranker-bulk-filters {
	display: flex;
	justify-content: space-between;
	align-items: center;
	flex-wrap: wrap;
	gap: 15px;
	margin-bottom: 20px;
}

.topranker-bulk-filter-form {
	display: flex;
	gap: 10px;
	align-items: center;
	flex-wrap: wrap;
}

.topranker-filter-select {
	min-width: 150px;
}

.topranker-filter-search {
	min-width: 200px;
}

.topranker-bulk-actions {
	display: flex;
	gap: 15px;
	align-items: center;
}

.topranker-bulk-select-info {
	font-size: 13px;
	color: #646970;
}

#topranker-bulk-optimize-btn .dashicons {
	font-size: 16px;
	width: 16px;
	height: 16px;
	margin-right: 4px;
}

/* Table Styles */
.topranker-bulk-table-wrap {
	background: #fff;
	border: 1px solid #c3c4c7;
	border-radius: 4px;
}

.topranker-bulk-table {
	margin: 0;
	border: none;
}

.topranker-bulk-table th,
.topranker-bulk-table td {
	padding: 10px 12px;
	vertical-align: middle;
}

.topranker-bulk-table .column-title {
	width: 30%;
}

.topranker-bulk-table .column-status,
.topranker-bulk-table .column-meta,
.topranker-bulk-table .column-alt,
.topranker-bulk-table .column-score,
.topranker-bulk-table .column-optimized {
	width: 12%;
	text-align: center;
}

.topranker-bulk-table .row-title {
	font-weight: 600;
	color: #1d2327;
}

/* Meta Indicators */
.topranker-meta-indicators {
	display: flex;
	gap: 4px;
	justify-content: center;
}

.topranker-meta-indicator {
	display: inline-flex;
	align-items: center;
	justify-content: center;
	width: 20px;
	height: 20px;
	font-size: 10px;
	font-weight: 600;
	border-radius: 3px;
}

.topranker-meta-indicator.is-set {
	background: #d4edda;
	color: #155724;
}

.topranker-meta-indicator.is-missing {
	background: #f8d7da;
	color: #721c24;
}

/* Alt Tags */
.topranker-alt-missing {
	color: #d63638;
	font-size: 12px;
}

.topranker-alt-complete {
	color: #00a32a;
}

.topranker-alt-complete .dashicons {
	font-size: 16px;
	width: 16px;
	height: 16px;
}

/* Score */
.topranker-no-score {
	color: #646970;
}

/* Last Optimized */
.topranker-never-optimized {
	color: #646970;
	font-style: italic;
}

/* Progress Panel - supplementing existing CSS */
.topranker-bulk-progress-actions {
	margin-top: 15px;
	text-align: center;
}

/* Modal Styles */
.topranker-modal {
	position: fixed;
	inset: 0;
	z-index: 100050;
	display: flex;
	align-items: center;
	justify-content: center;
}

.topranker-modal-backdrop {
	position: absolute;
	inset: 0;
	background: rgba(0, 0, 0, 0.5);
}

.topranker-modal-content {
	position: relative;
	background: #fff;
	border-radius: 4px;
	box-shadow: 0 3px 30px rgba(0, 0, 0, 0.2);
	max-width: 500px;
	width: 90%;
}

.topranker-modal-header {
	padding: 15px 20px;
	border-bottom: 1px solid #dcdcde;
}

.topranker-modal-header h2 {
	margin: 0;
	font-size: 18px;
}

.topranker-modal-body {
	padding: 20px;
}

.topranker-modal-estimate {
	margin: 15px 0;
	padding: 12px 15px;
	background: #f0f6fc;
	border-radius: 4px;
}

.topranker-modal-estimate p {
	margin: 5px 0;
	display: flex;
	align-items: center;
	gap: 6px;
}

.topranker-modal-estimate .dashicons {
	color: #2271b1;
}

.topranker-modal-note {
	font-size: 12px;
	color: #646970;
	font-style: italic;
}

.topranker-modal-footer {
	display: flex;
	justify-content: flex-end;
	gap: 10px;
	padding: 15px 20px;
	border-top: 1px solid #dcdcde;
	background: #f6f7f7;
}

/* Responsive */
@media screen and (max-width: 782px) {
	.topranker-bulk-stats {
		flex-wrap: wrap;
	}

	.topranker-bulk-filters {
		flex-direction: column;
		align-items: flex-start;
	}

	.topranker-bulk-filter-form {
		width: 100%;
	}

	.topranker-bulk-actions {
		width: 100%;
		justify-content: flex-start;
	}

	.topranker-bulk-table .column-alt,
	.topranker-bulk-table .column-score {
		display: none;
	}
}
</style>

<script type="text/javascript">
(function($) {
	'use strict';

	var selectedPosts = [];
	var isProcessing = false;
	var isCancelled = false;
	var nonce = '<?php echo esc_js( $nonce ); ?>';

	$(document).ready(function() {
		// Select all checkbox.
		$('#cb-select-all').on('change', function() {
			var isChecked = $(this).is(':checked');
			$('.topranker-post-checkbox').prop('checked', isChecked);
			updateSelectedCount();
		});

		// Individual checkboxes.
		$(document).on('change', '.topranker-post-checkbox', function() {
			updateSelectedCount();
		});

		// Bulk optimize button click.
		$('#topranker-bulk-optimize-btn').on('click', function() {
			if (selectedPosts.length === 0) {
				return;
			}
			showConfirmModal();
		});

		// Confirm modal cancel.
		$('#topranker-confirm-cancel, .topranker-modal-backdrop').on('click', function() {
			hideConfirmModal();
		});

		// Confirm modal proceed.
		$('#topranker-confirm-proceed').on('click', function() {
			hideConfirmModal();
			startBulkOptimization();
		});

		// Cancel button.
		$('#topranker-cancel-btn').on('click', function() {
			isCancelled = true;
			$(this).prop('disabled', true).text('<?php echo esc_js( __( 'Cancelling...', 'topranker-ai' ) ); ?>');
		});

		// Auto-submit on filter change.
		$('#topranker-post-type-filter, #topranker-filter').on('change', function() {
			$(this).closest('form').submit();
		});
	});

	function updateSelectedCount() {
		selectedPosts = [];
		$('.topranker-post-checkbox:checked').each(function() {
			selectedPosts.push($(this).val());
		});

		$('#topranker-selected-count').text(selectedPosts.length);
		$('#topranker-bulk-optimize-btn').prop('disabled', selectedPosts.length === 0);
	}

	function showConfirmModal() {
		var count = selectedPosts.length;
		var apiCalls = count; // 1 call per post for basic optimization.
		var costMin = (apiCalls * 0.01).toFixed(2);
		var costMax = (apiCalls * 0.03).toFixed(2);

		$('#topranker-confirm-count').text(count);
		$('#topranker-confirm-api-calls').text(apiCalls);
		$('#topranker-confirm-cost').text('$' + costMin + ' - $' + costMax);
		$('#topranker-confirm-modal').show();
	}

	function hideConfirmModal() {
		$('#topranker-confirm-modal').hide();
	}

	function startBulkOptimization() {
		if (selectedPosts.length === 0) {
			return;
		}

		isProcessing = true;
		isCancelled = false;

		var total = selectedPosts.length;
		var current = 0;
		var success = 0;
		var failed = 0;
		var results = [];

		// Show progress panel.
		$('#topranker-bulk-progress').show();
		$('#topranker-progress-total').text(total);
		$('#topranker-progress-current').text(0);
		$('#topranker-progress-fill').css('width', '0%');
		$('#topranker-progress-text').text('<?php echo esc_js( __( 'Starting optimization...', 'topranker-ai' ) ); ?>');
		$('#topranker-bulk-results').hide().empty();
		$('#topranker-cancel-btn').prop('disabled', false).text('<?php echo esc_js( __( 'Cancel', 'topranker-ai' ) ); ?>');

		// Disable table interactions.
		$('.topranker-bulk-table-wrap').css('opacity', '0.5').css('pointer-events', 'none');
		$('#topranker-bulk-optimize-btn').prop('disabled', true);

		// Process posts sequentially.
		function processNext() {
			if (isCancelled || current >= selectedPosts.length) {
				finishOptimization(success, failed, results);
				return;
			}

			var postId = selectedPosts[current];
			var $row = $('#post-' + postId);
			var postTitle = $row.find('.row-title').text();

			// Update progress.
			$('#topranker-progress-current').text(current + 1);
			var percent = Math.round(((current + 1) / total) * 100);
			$('#topranker-progress-fill').css('width', percent + '%');
			$('#topranker-progress-text').text(
				'<?php echo esc_js( __( 'Optimizing:', 'topranker-ai' ) ); ?> ' + postTitle
			);

			// Add processing indicator.
			addResultRow(postId, postTitle, 'processing', '<?php echo esc_js( __( 'Processing...', 'topranker-ai' ) ); ?>');

			// Make AJAX call.
			$.ajax({
				url: ajaxurl,
				type: 'POST',
				data: {
					action: 'topranker_bulk_optimize',
					post_id: postId,
					nonce: nonce
				},
				success: function(response) {
					if (response.success && (!response.data || response.data.success !== false)) {
						success++;
						updateResultRow(postId, 'success', '<?php echo esc_js( __( 'Optimized', 'topranker-ai' ) ); ?>');
						results.push({ id: postId, success: true });

						// Update table row.
						updateTableRow(postId, response.data);
					} else {
						failed++;
						var errorMsg = response.data && response.data.message
							? response.data.message
							: (response.data && response.data.error
								? response.data.error
								: '<?php echo esc_js( __( 'Failed', 'topranker-ai' ) ); ?>');
						updateResultRow(postId, 'error', errorMsg);
						results.push({ id: postId, success: false, error: errorMsg });
					}
				},
				error: function() {
					failed++;
					updateResultRow(postId, 'error', '<?php echo esc_js( __( 'Network error', 'topranker-ai' ) ); ?>');
					results.push({ id: postId, success: false, error: 'Network error' });
				},
				complete: function() {
					current++;
					processNext();
				}
			});
		}

		processNext();
	}

	function addResultRow(postId, title, status, message) {
		var $results = $('#topranker-bulk-results');
		$results.show();

		var iconClass = 'dashicons-update';
		if (status === 'success') {
			iconClass = 'dashicons-yes';
		} else if (status === 'error') {
			iconClass = 'dashicons-no';
		}

		var html = '<div id="result-' + postId + '" class="topranker-bulk-result is-' + status + '">' +
			'<span class="topranker-bulk-result-icon"><span class="dashicons ' + iconClass + '"></span></span>' +
			'<span class="topranker-bulk-result-title">' + escapeHtml(title) + '</span>' +
			'<span class="topranker-bulk-result-status">' + escapeHtml(message) + '</span>' +
			'</div>';

		$results.append(html);

		// Scroll to bottom.
		$results.scrollTop($results[0].scrollHeight);
	}

	function updateResultRow(postId, status, message) {
		var $row = $('#result-' + postId);
		$row.removeClass('is-processing is-success is-error').addClass('is-' + status);

		var iconClass = status === 'success' ? 'dashicons-yes' : 'dashicons-no';
		$row.find('.dashicons').removeClass('dashicons-update dashicons-yes dashicons-no').addClass(iconClass);
		$row.find('.topranker-bulk-result-status').text(message);
	}

	function updateTableRow(postId, data) {
		var $row = $('#post-' + postId);

		// Update SEO status.
		var $status = $row.find('.topranker-seo-status');
		$status.removeClass('is-missing is-partial is-complete').addClass('is-complete');
		$status.text('<?php echo esc_js( __( 'Complete', 'topranker-ai' ) ); ?>');

		// Update meta indicators.
		$row.find('.topranker-meta-indicator').removeClass('is-missing').addClass('is-set');

		// Update last optimized.
		$row.find('.column-optimized').html(
			'<span class="topranker-last-optimized"><?php echo esc_js( __( 'Just now', 'topranker-ai' ) ); ?></span>'
		);
	}

	function finishOptimization(success, failed, results) {
		isProcessing = false;

		$('#topranker-progress-fill').css('width', '100%');

		var summaryMsg;
		if (isCancelled) {
			summaryMsg = '<?php echo esc_js( __( 'Cancelled.', 'topranker-ai' ) ); ?> ';
		} else {
			summaryMsg = '<?php echo esc_js( __( 'Complete!', 'topranker-ai' ) ); ?> ';
		}

		summaryMsg += success + ' <?php echo esc_js( __( 'optimized', 'topranker-ai' ) ); ?>';
		if (failed > 0) {
			summaryMsg += ', ' + failed + ' <?php echo esc_js( __( 'failed', 'topranker-ai' ) ); ?>';
		}

		$('#topranker-progress-text').text(summaryMsg);
		$('#topranker-cancel-btn').text('<?php echo esc_js( __( 'Done', 'topranker-ai' ) ); ?>').on('click', function() {
			location.reload();
		});

		// Re-enable table.
		$('.topranker-bulk-table-wrap').css('opacity', '1').css('pointer-events', 'auto');

		// Clear selection.
		$('.topranker-post-checkbox').prop('checked', false);
		$('#cb-select-all').prop('checked', false);
		updateSelectedCount();
	}

	function escapeHtml(text) {
		var map = {
			'&': '&amp;',
			'<': '&lt;',
			'>': '&gt;',
			'"': '&quot;',
			"'": '&#039;'
		};
		return text.replace(/[&<>"']/g, function(m) { return map[m]; });
	}
})(jQuery);
</script>
<?php
