<?php
/**
 * SEO Coverage Dashboard page template.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Get dashboard instance and view data.
$dashboard = new TopRanker_Dashboard();
$data      = $dashboard->get_view_data();
$stats     = $data['stats'];
$grade     = $data['grade'];
$is_pro    = $data['is_pro'];

// Nonce for AJAX refresh.
$refresh_nonce = wp_create_nonce( 'topranker_admin' );
?>
<div class="wrap topranker-dashboard-wrap">
	<div class="topranker-dashboard-header">
		<h1 class="topranker-dashboard-title">
			<span class="dashicons dashicons-chart-bar" aria-hidden="true"></span>
			<?php esc_html_e( 'SEO Coverage Dashboard', 'topranker-ai' ); ?>
		</h1>
		<div class="topranker-dashboard-actions">
			<button
				type="button"
				id="topranker-refresh-stats"
				class="button button-secondary"
				data-nonce="<?php echo esc_attr( $refresh_nonce ); ?>"
				aria-label="<?php esc_attr_e( 'Refresh statistics', 'topranker-ai' ); ?>"
			>
				<span class="dashicons dashicons-update" aria-hidden="true"></span>
				<?php esc_html_e( 'Refresh Stats', 'topranker-ai' ); ?>
			</button>
			<?php if ( $is_pro ) : ?>
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=topranker-ai-bulk' ) ); ?>" class="button button-primary">
					<?php esc_html_e( 'Bulk Optimize', 'topranker-ai' ); ?>
				</a>
			<?php else : ?>
				<a href="<?php echo esc_url( $data['upgrade_url'] ); ?>" class="button button-primary">
					<?php esc_html_e( 'Upgrade to Pro', 'topranker-ai' ); ?>
				</a>
			<?php endif; ?>
		</div>
	</div>

	<?php if ( 0 === $stats['total_posts'] ) : ?>
		<div class="topranker-notice is-info">
			<span class="topranker-notice-icon">
				<span class="dashicons dashicons-info" aria-hidden="true"></span>
			</span>
			<div class="topranker-notice-content">
				<p class="topranker-notice-title"><?php esc_html_e( 'No content found', 'topranker-ai' ); ?></p>
				<p class="topranker-notice-message">
					<?php esc_html_e( 'Publish some posts or pages to see your SEO coverage statistics.', 'topranker-ai' ); ?>
				</p>
			</div>
		</div>
	<?php else : ?>
		<!-- Overall Grade Card -->
		<div class="topranker-stats-grid">
			<div class="topranker-stat-card topranker-stat-card-grade">
				<div class="topranker-stat-card-header">
					<span class="topranker-stat-card-title"><?php esc_html_e( 'Overall SEO Coverage', 'topranker-ai' ); ?></span>
					<span class="topranker-stat-card-icon">
						<span class="dashicons dashicons-<?php echo esc_attr( $grade['icon'] ); ?>" aria-hidden="true"></span>
					</span>
				</div>
				<div class="topranker-stat-card-value">
					<?php echo esc_html( $stats['overall_coverage'] ); ?>%
				</div>
				<div class="topranker-coverage-bar">
					<div class="topranker-coverage-bar-track">
						<div
							class="topranker-coverage-bar-fill <?php echo esc_attr( $stats['overall_coverage'] >= 70 ? 'is-good' : ( $stats['overall_coverage'] >= 40 ? 'is-warning' : 'is-bad' ) ); ?>"
							style="width: <?php echo esc_attr( $stats['overall_coverage'] ); ?>%;"
							role="progressbar"
							aria-valuenow="<?php echo esc_attr( $stats['overall_coverage'] ); ?>"
							aria-valuemin="0"
							aria-valuemax="100"
						></div>
					</div>
				</div>
				<div class="topranker-stat-card-meta">
					<span class="<?php echo esc_attr( 'green' === $grade['color'] ? 'is-good' : '' ); ?>">
						<?php echo esc_html( $grade['label'] ); ?>
					</span>
				</div>
			</div>

			<!-- Total Posts -->
			<div class="topranker-stat-card">
				<div class="topranker-stat-card-header">
					<span class="topranker-stat-card-title"><?php esc_html_e( 'Total Content', 'topranker-ai' ); ?></span>
					<span class="topranker-stat-card-icon">
						<span class="dashicons dashicons-admin-post" aria-hidden="true"></span>
					</span>
				</div>
				<div class="topranker-stat-card-value"><?php echo esc_html( number_format_i18n( $stats['total_posts'] ) ); ?></div>
				<div class="topranker-stat-card-meta">
					<span>
						<?php
						$post_type_labels = array();
						foreach ( $stats['post_types'] as $post_type ) {
							$post_type_obj = get_post_type_object( $post_type );
							if ( $post_type_obj ) {
								$post_type_labels[] = $post_type_obj->labels->name;
							}
						}
						echo esc_html( implode( ', ', $post_type_labels ) );
						?>
					</span>
				</div>
			</div>

			<!-- Meta Titles -->
			<div class="topranker-stat-card">
				<div class="topranker-stat-card-header">
					<span class="topranker-stat-card-title"><?php esc_html_e( 'Meta Titles', 'topranker-ai' ); ?></span>
					<span class="topranker-stat-card-icon">
						<span class="dashicons dashicons-heading" aria-hidden="true"></span>
					</span>
				</div>
				<div class="topranker-stat-card-value"><?php echo esc_html( $stats['meta_title']['percent'] ); ?>%</div>
				<div class="topranker-coverage-bar">
					<div class="topranker-coverage-bar-track">
						<div
							class="topranker-coverage-bar-fill <?php echo esc_attr( $stats['meta_title']['percent'] >= 70 ? 'is-good' : ( $stats['meta_title']['percent'] >= 40 ? 'is-warning' : 'is-bad' ) ); ?>"
							style="width: <?php echo esc_attr( $stats['meta_title']['percent'] ); ?>%;"
						></div>
					</div>
				</div>
				<div class="topranker-stat-card-meta">
					<span class="is-good"><?php echo esc_html( $stats['meta_title']['count'] ); ?> <?php esc_html_e( 'optimized', 'topranker-ai' ); ?></span>
					<span class="is-bad"><?php echo esc_html( $stats['meta_title']['missing'] ); ?> <?php esc_html_e( 'missing', 'topranker-ai' ); ?></span>
				</div>
			</div>

			<!-- Meta Descriptions -->
			<div class="topranker-stat-card">
				<div class="topranker-stat-card-header">
					<span class="topranker-stat-card-title"><?php esc_html_e( 'Meta Descriptions', 'topranker-ai' ); ?></span>
					<span class="topranker-stat-card-icon">
						<span class="dashicons dashicons-text" aria-hidden="true"></span>
					</span>
				</div>
				<div class="topranker-stat-card-value"><?php echo esc_html( $stats['meta_description']['percent'] ); ?>%</div>
				<div class="topranker-coverage-bar">
					<div class="topranker-coverage-bar-track">
						<div
							class="topranker-coverage-bar-fill <?php echo esc_attr( $stats['meta_description']['percent'] >= 70 ? 'is-good' : ( $stats['meta_description']['percent'] >= 40 ? 'is-warning' : 'is-bad' ) ); ?>"
							style="width: <?php echo esc_attr( $stats['meta_description']['percent'] ); ?>%;"
						></div>
					</div>
				</div>
				<div class="topranker-stat-card-meta">
					<span class="is-good"><?php echo esc_html( $stats['meta_description']['count'] ); ?> <?php esc_html_e( 'optimized', 'topranker-ai' ); ?></span>
					<span class="is-bad"><?php echo esc_html( $stats['meta_description']['missing'] ); ?> <?php esc_html_e( 'missing', 'topranker-ai' ); ?></span>
				</div>
			</div>

			<!-- Focus Keyphrases -->
			<div class="topranker-stat-card">
				<div class="topranker-stat-card-header">
					<span class="topranker-stat-card-title"><?php esc_html_e( 'Focus Keyphrases', 'topranker-ai' ); ?></span>
					<span class="topranker-stat-card-icon">
						<span class="dashicons dashicons-tag" aria-hidden="true"></span>
					</span>
				</div>
				<div class="topranker-stat-card-value"><?php echo esc_html( $stats['keyphrase']['percent'] ); ?>%</div>
				<div class="topranker-coverage-bar">
					<div class="topranker-coverage-bar-track">
						<div
							class="topranker-coverage-bar-fill <?php echo esc_attr( $stats['keyphrase']['percent'] >= 70 ? 'is-good' : ( $stats['keyphrase']['percent'] >= 40 ? 'is-warning' : 'is-bad' ) ); ?>"
							style="width: <?php echo esc_attr( $stats['keyphrase']['percent'] ); ?>%;"
						></div>
					</div>
				</div>
				<div class="topranker-stat-card-meta">
					<span class="is-good"><?php echo esc_html( $stats['keyphrase']['count'] ); ?> <?php esc_html_e( 'set', 'topranker-ai' ); ?></span>
					<span class="is-bad"><?php echo esc_html( $stats['keyphrase']['missing'] ); ?> <?php esc_html_e( 'missing', 'topranker-ai' ); ?></span>
				</div>
			</div>

			<!-- Image Alt Tags (Pro teaser) -->
			<div class="topranker-stat-card">
				<div class="topranker-stat-card-header">
					<span class="topranker-stat-card-title">
						<?php esc_html_e( 'Images Missing Alt Text', 'topranker-ai' ); ?>
						<?php if ( ! $is_pro ) : ?>
							<span class="topranker-pro-badge">
								<span class="dashicons dashicons-lock" aria-hidden="true"></span>
								<?php esc_html_e( 'Pro', 'topranker-ai' ); ?>
							</span>
						<?php endif; ?>
					</span>
					<span class="topranker-stat-card-icon">
						<span class="dashicons dashicons-format-image" aria-hidden="true"></span>
					</span>
				</div>
				<?php if ( $is_pro ) : ?>
					<div class="topranker-stat-card-value"><?php echo esc_html( number_format_i18n( $stats['images_missing_alt'] ) ); ?></div>
					<div class="topranker-stat-card-meta">
						<span class="<?php echo 0 === $stats['images_missing_alt'] ? 'is-good' : 'is-bad'; ?>">
							<?php
							if ( 0 === $stats['images_missing_alt'] ) {
								esc_html_e( 'All images have alt text', 'topranker-ai' );
							} else {
								esc_html_e( 'images need alt text', 'topranker-ai' );
							}
							?>
						</span>
					</div>
				<?php else : ?>
					<div class="topranker-pro-lock">
						<span class="topranker-pro-lock-icon">
							<span class="dashicons dashicons-lock" aria-hidden="true"></span>
						</span>
						<div class="topranker-pro-lock-content">
							<span class="topranker-pro-lock-description">
								<?php esc_html_e( 'Upgrade to Pro to see how many images are missing alt text.', 'topranker-ai' ); ?>
							</span>
							<a href="<?php echo esc_url( $data['upgrade_url'] ); ?>" class="topranker-pro-lock-link">
								<?php esc_html_e( 'Upgrade to Pro', 'topranker-ai' ); ?>
								<span class="dashicons dashicons-arrow-right-alt2" aria-hidden="true"></span>
							</a>
						</div>
					</div>
				<?php endif; ?>
			</div>
		</div>

		<!-- Posts Missing Meta Titles -->
		<?php if ( ! empty( $data['missing_titles'] ) ) : ?>
			<div class="topranker-dashboard-table">
				<div class="topranker-dashboard-table-header">
					<h2 class="topranker-dashboard-table-title">
						<?php esc_html_e( 'Posts Missing Meta Titles', 'topranker-ai' ); ?>
						<span class="count">(<?php echo esc_html( $stats['meta_title']['missing'] ); ?>)</span>
					</h2>
					<?php if ( $is_pro ) : ?>
						<a href="<?php echo esc_url( admin_url( 'admin.php?page=topranker-ai-bulk' ) ); ?>" class="button button-secondary button-small">
							<?php esc_html_e( 'Fix All', 'topranker-ai' ); ?>
						</a>
					<?php endif; ?>
				</div>
				<table>
					<thead>
						<tr>
							<th><?php esc_html_e( 'Title', 'topranker-ai' ); ?></th>
							<th><?php esc_html_e( 'Type', 'topranker-ai' ); ?></th>
							<th><?php esc_html_e( 'Action', 'topranker-ai' ); ?></th>
						</tr>
					</thead>
					<tbody>
						<?php foreach ( $data['missing_titles'] as $post_item ) : ?>
							<tr>
								<td>
									<a href="<?php echo esc_url( $post_item['edit_link'] ); ?>">
										<?php echo esc_html( $post_item['title'] ); ?>
									</a>
								</td>
								<td>
									<?php
									$post_type_obj = get_post_type_object( $post_item['post_type'] );
									echo esc_html( $post_type_obj ? $post_type_obj->labels->singular_name : $post_item['post_type'] );
									?>
								</td>
								<td>
									<a href="<?php echo esc_url( add_query_arg( 'topranker_optimize', '1', $post_item['edit_link'] ) ); ?>" class="button button-small">
										<?php esc_html_e( 'Optimize', 'topranker-ai' ); ?>
									</a>
								</td>
							</tr>
						<?php endforeach; ?>
					</tbody>
				</table>
			</div>
		<?php endif; ?>

		<!-- Posts Missing Meta Descriptions -->
		<?php if ( ! empty( $data['missing_desc'] ) ) : ?>
			<div class="topranker-dashboard-table" style="margin-top: 20px;">
				<div class="topranker-dashboard-table-header">
					<h2 class="topranker-dashboard-table-title">
						<?php esc_html_e( 'Posts Missing Meta Descriptions', 'topranker-ai' ); ?>
						<span class="count">(<?php echo esc_html( $stats['meta_description']['missing'] ); ?>)</span>
					</h2>
					<?php if ( $is_pro ) : ?>
						<a href="<?php echo esc_url( admin_url( 'admin.php?page=topranker-ai-bulk' ) ); ?>" class="button button-secondary button-small">
							<?php esc_html_e( 'Fix All', 'topranker-ai' ); ?>
						</a>
					<?php endif; ?>
				</div>
				<table>
					<thead>
						<tr>
							<th><?php esc_html_e( 'Title', 'topranker-ai' ); ?></th>
							<th><?php esc_html_e( 'Type', 'topranker-ai' ); ?></th>
							<th><?php esc_html_e( 'Action', 'topranker-ai' ); ?></th>
						</tr>
					</thead>
					<tbody>
						<?php foreach ( $data['missing_desc'] as $post_item ) : ?>
							<tr>
								<td>
									<a href="<?php echo esc_url( $post_item['edit_link'] ); ?>">
										<?php echo esc_html( $post_item['title'] ); ?>
									</a>
								</td>
								<td>
									<?php
									$post_type_obj = get_post_type_object( $post_item['post_type'] );
									echo esc_html( $post_type_obj ? $post_type_obj->labels->singular_name : $post_item['post_type'] );
									?>
								</td>
								<td>
									<a href="<?php echo esc_url( add_query_arg( 'topranker_optimize', '1', $post_item['edit_link'] ) ); ?>" class="button button-small">
										<?php esc_html_e( 'Optimize', 'topranker-ai' ); ?>
									</a>
								</td>
							</tr>
						<?php endforeach; ?>
					</tbody>
				</table>
			</div>
		<?php endif; ?>

		<!-- Last Updated -->
		<p class="topranker-dashboard-footer" style="margin-top: 20px; color: #646970; font-size: 13px;">
			<?php
			printf(
				/* translators: %s: time ago text */
				esc_html__( 'Statistics last updated: %s', 'topranker-ai' ),
				esc_html( $data['last_updated'] )
			);
			?>
		</p>
	<?php endif; ?>
</div>

<script type="text/javascript">
(function($) {
	'use strict';

	$(document).ready(function() {
		var $refreshBtn = $('#topranker-refresh-stats');
		var $dashboardWrap = $('.topranker-dashboard-wrap');

		$refreshBtn.on('click', function() {
			var $btn = $(this);
			var nonce = $btn.data('nonce');

			// Disable button and show loading state.
			$btn.prop('disabled', true);
			$btn.find('.dashicons').addClass('topranker-spin');

			$.ajax({
				url: ajaxurl,
				type: 'POST',
				data: {
					action: 'topranker_refresh_dashboard_stats',
					nonce: nonce
				},
				success: function(response) {
					if (response.success) {
						// Reload the page to show updated stats.
						location.reload();
					} else {
						alert(response.data.message || '<?php echo esc_js( __( 'Failed to refresh stats.', 'topranker-ai' ) ); ?>');
						$btn.prop('disabled', false);
						$btn.find('.dashicons').removeClass('topranker-spin');
					}
				},
				error: function() {
					alert('<?php echo esc_js( __( 'An error occurred. Please try again.', 'topranker-ai' ) ); ?>');
					$btn.prop('disabled', false);
					$btn.find('.dashicons').removeClass('topranker-spin');
				}
			});
		});
	});
})(jQuery);
</script>
