<?php
/**
 * Classic Editor metabox template.
 *
 * Displays the TopRanker AI optimization interface in the Classic Editor.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 *
 * @var WP_Post $post The current post object.
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

$api_key          = get_option( 'topranker_api_key', '' );
$has_api_key      = ! empty( $api_key );
$is_pro           = topranker_is_pro();
$usage            = new TopRanker_Usage();
$usage_data       = $usage->get_stats();
$optimizer        = new TopRanker_Optimizer();
$suggestions_data = $optimizer->get_suggestions_for_editor( $post->ID );
$word_count       = $optimizer->get_word_count( $post );
$has_suggestions  = ! empty( $suggestions_data['suggestions'] );
$current          = $suggestions_data['current'];
$seo_compat       = new TopRanker_SEO_Compat();
$has_seo_plugin   = $seo_compat->has_seo_plugin();
$seo_plugin_name  = $seo_compat->get_detected_plugin_name();
$is_suggest_mode  = $seo_compat->is_suggest_mode();
$is_sync_mode     = $seo_compat->is_sync_mode();

// Check if auto-optimizing is in progress.
$is_auto_optimizing = false;
if ( class_exists( 'TopRanker_Cron' ) ) {
	$cron               = new TopRanker_Cron();
	$is_auto_optimizing = $cron->is_auto_optimizing( $post->ID );
}

// Get nonce for AJAX.
$nonce = TopRanker_AJAX::get_nonce();
?>

<div class="topranker-metabox" id="topranker-metabox" data-post-id="<?php echo esc_attr( $post->ID ); ?>" data-nonce="<?php echo esc_attr( $nonce ); ?>">

	<?php if ( ! $has_api_key ) : ?>
		<!-- No API Key Warning -->
		<div class="topranker-notice is-warning">
			<div class="topranker-notice-icon">
				<span class="dashicons dashicons-warning" aria-hidden="true"></span>
			</div>
			<div class="topranker-notice-content">
				<div class="topranker-notice-title"><?php esc_html_e( 'API Key Required', 'topranker-ai' ); ?></div>
				<div class="topranker-notice-message">
					<?php
					printf(
						/* translators: %s: Settings page URL */
						esc_html__( 'Please configure your OpenAI API key in %s to enable optimization.', 'topranker-ai' ),
						'<a href="' . esc_url( admin_url( 'admin.php?page=topranker-ai' ) ) . '">' . esc_html__( 'TopRanker Settings', 'topranker-ai' ) . '</a>'
					);
					?>
				</div>
			</div>
		</div>
	<?php else : ?>

		<!-- Header with Optimize Button and Usage -->
		<div class="topranker-metabox-header">
			<button type="button" id="topranker-optimize-btn" class="button button-primary topranker-optimize-btn" <?php disabled( ! $usage->can_optimize() ); ?>>
				<span class="dashicons dashicons-update" aria-hidden="true"></span>
				<span class="topranker-btn-text"><?php esc_html_e( 'Optimize This Post', 'topranker-ai' ); ?></span>
			</button>

			<?php
			$usage_class = '';
			if ( $usage_data['remaining'] <= 0 && ! $is_pro ) {
				$usage_class = 'is-limit';
			} elseif ( $usage_data['remaining'] <= 5 && ! $is_pro ) {
				$usage_class = 'is-warning';
			}
			?>

			<?php if ( ! $is_pro ) : ?>
				<span class="topranker-usage-badge <?php echo esc_attr( $usage_class ); ?>">
					<?php
					printf(
						/* translators: 1: Used count, 2: Total limit */
						esc_html__( '%1$d / %2$d this month', 'topranker-ai' ),
						$usage_data['used'],
						$usage_data['limit']
					);
					?>
				</span>
			<?php else : ?>
				<span class="topranker-pro-badge">
					<span class="dashicons dashicons-star-filled" aria-hidden="true"></span>
					<?php esc_html_e( 'PRO', 'topranker-ai' ); ?>
				</span>
			<?php endif; ?>
		</div>

		<?php if ( $is_auto_optimizing ) : ?>
			<!-- Auto-Optimizing Badge -->
			<div class="topranker-notice is-info" style="margin: 10px 0;">
				<div class="topranker-notice-icon">
					<span class="dashicons dashicons-update topranker-spin" aria-hidden="true"></span>
				</div>
				<div class="topranker-notice-content">
					<div class="topranker-notice-title"><?php esc_html_e( 'Auto-Optimizing...', 'topranker-ai' ); ?></div>
					<div class="topranker-notice-message">
						<?php esc_html_e( 'This post is being optimized in the background. Refresh the page to see results.', 'topranker-ai' ); ?>
					</div>
				</div>
			</div>
		<?php endif; ?>

		<?php if ( $word_count < 50 ) : ?>
			<!-- Low Content Warning -->
			<div class="topranker-notice is-info">
				<div class="topranker-notice-icon">
					<span class="dashicons dashicons-info" aria-hidden="true"></span>
				</div>
				<div class="topranker-notice-content">
					<div class="topranker-notice-message">
						<?php esc_html_e( 'Add more content for better results. TopRanker works best with 100+ words.', 'topranker-ai' ); ?>
					</div>
				</div>
			</div>
		<?php endif; ?>

		<?php if ( ! $usage->can_optimize() && ! $is_pro ) : ?>
			<!-- Usage Limit Reached -->
			<div class="topranker-notice is-warning">
				<div class="topranker-notice-icon">
					<span class="dashicons dashicons-warning" aria-hidden="true"></span>
				</div>
				<div class="topranker-notice-content">
					<div class="topranker-notice-title"><?php esc_html_e( 'Monthly Limit Reached', 'topranker-ai' ); ?></div>
					<div class="topranker-notice-message">
						<?php echo esc_html( $usage->get_limit_message() ); ?>
					</div>
					<?php if ( function_exists( 'topranker_fs' ) ) : ?>
						<a href="<?php echo esc_url( topranker_fs()->get_upgrade_url() ); ?>" class="button button-primary topranker-upgrade-button">
							<?php esc_html_e( 'Upgrade to Pro', 'topranker-ai' ); ?>
						</a>
					<?php endif; ?>
				</div>
			</div>
		<?php endif; ?>

		<!-- Loading State -->
		<div id="topranker-loading" class="topranker-loading" style="display: none;">
			<div class="topranker-spinner"></div>
			<div class="topranker-loading-text"><?php esc_html_e( 'Generating suggestions...', 'topranker-ai' ); ?></div>
		</div>

		<!-- Error State -->
		<div id="topranker-error" class="topranker-error" style="display: none;">
			<div class="topranker-error-icon">
				<span class="dashicons dashicons-warning" aria-hidden="true"></span>
			</div>
			<div class="topranker-error-title"><?php esc_html_e( 'Optimization Failed', 'topranker-ai' ); ?></div>
			<div id="topranker-error-message" class="topranker-error-message"></div>
			<button type="button" id="topranker-retry-btn" class="button topranker-retry-btn">
				<span class="dashicons dashicons-update" aria-hidden="true"></span>
				<?php esc_html_e( 'Retry', 'topranker-ai' ); ?>
			</button>
		</div>

		<!-- Results Container -->
		<div id="topranker-results" class="topranker-results" <?php echo ( ! $has_suggestions ) ? 'style="display: none;"' : ''; ?>>

			<!-- Meta Title Section -->
			<div class="topranker-section" id="topranker-meta-title-section">
				<div class="topranker-section-title">
					<span class="dashicons dashicons-heading" aria-hidden="true"></span>
					<?php esc_html_e( 'Meta Title', 'topranker-ai' ); ?>
				</div>

				<div id="topranker-meta-title-suggestions" class="topranker-suggestions">
					<?php if ( $has_suggestions && ! empty( $suggestions_data['suggestions']['meta_title']['suggestions'] ) ) : ?>
						<?php foreach ( $suggestions_data['suggestions']['meta_title']['suggestions'] as $index => $suggestion ) : ?>
							<?php $suggestion_text = is_array( $suggestion ) ? ( $suggestion['title'] ?? '' ) : $suggestion; ?>
							<label class="topranker-suggestion <?php echo ( 0 === $index ) ? 'is-selected' : ''; ?>">
								<input type="radio" name="topranker_meta_title_choice" value="<?php echo esc_attr( $suggestion_text ); ?>" <?php checked( 0 === $index ); ?> />
								<div class="topranker-suggestion-content">
									<div class="topranker-suggestion-text"><?php echo esc_html( $suggestion_text ); ?></div>
									<div class="topranker-suggestion-meta">
										<?php
										$char_count = mb_strlen( $suggestion_text );
										$char_class = 'is-good';
										if ( $char_count > 60 ) {
											$char_class = 'is-error';
										} elseif ( $char_count < 50 ) {
											$char_class = 'is-warning';
										}
										?>
										<span class="topranker-char-count <?php echo esc_attr( $char_class ); ?>">
											<?php
											printf(
												/* translators: %d: Character count */
												esc_html__( '%d chars', 'topranker-ai' ),
												$char_count
											);
											?>
										</span>
									</div>
								</div>
							</label>
						<?php endforeach; ?>
					<?php endif; ?>
				</div>

				<div class="topranker-editable-field">
					<label for="topranker-meta-title-custom"><?php esc_html_e( 'Or enter custom title:', 'topranker-ai' ); ?></label>
					<input type="text" id="topranker-meta-title-custom" name="topranker_meta_title_custom" value="<?php echo esc_attr( $current['meta_title'] ); ?>" maxlength="70" />
					<div class="topranker-char-count" id="topranker-meta-title-count">
						<span class="count"><?php echo esc_html( mb_strlen( $current['meta_title'] ) ); ?></span> / 60
					</div>
				</div>
			</div>

			<!-- Meta Description Section -->
			<div class="topranker-section" id="topranker-meta-description-section">
				<div class="topranker-section-title">
					<span class="dashicons dashicons-editor-alignleft" aria-hidden="true"></span>
					<?php esc_html_e( 'Meta Description', 'topranker-ai' ); ?>
				</div>

				<div id="topranker-meta-description-suggestions" class="topranker-suggestions">
					<?php if ( $has_suggestions && ! empty( $suggestions_data['suggestions']['meta_description']['suggestions'] ) ) : ?>
						<?php foreach ( $suggestions_data['suggestions']['meta_description']['suggestions'] as $index => $suggestion ) : ?>
							<?php $suggestion_text = is_array( $suggestion ) ? ( $suggestion['description'] ?? '' ) : $suggestion; ?>
							<label class="topranker-suggestion <?php echo ( 0 === $index ) ? 'is-selected' : ''; ?>">
								<input type="radio" name="topranker_meta_description_choice" value="<?php echo esc_attr( $suggestion_text ); ?>" <?php checked( 0 === $index ); ?> />
								<div class="topranker-suggestion-content">
									<div class="topranker-suggestion-text"><?php echo esc_html( $suggestion_text ); ?></div>
									<div class="topranker-suggestion-meta">
										<?php
										$char_count = mb_strlen( $suggestion_text );
										$char_class = 'is-good';
										if ( $char_count > 155 ) {
											$char_class = 'is-error';
										} elseif ( $char_count < 120 ) {
											$char_class = 'is-warning';
										}
										?>
										<span class="topranker-char-count <?php echo esc_attr( $char_class ); ?>">
											<?php
											printf(
												/* translators: %d: Character count */
												esc_html__( '%d chars', 'topranker-ai' ),
												$char_count
											);
											?>
										</span>
									</div>
								</div>
							</label>
						<?php endforeach; ?>
					<?php endif; ?>
				</div>

				<div class="topranker-editable-field">
					<label for="topranker-meta-description-custom"><?php esc_html_e( 'Or enter custom description:', 'topranker-ai' ); ?></label>
					<textarea id="topranker-meta-description-custom" name="topranker_meta_description_custom" maxlength="200" rows="3"><?php echo esc_textarea( $current['meta_description'] ); ?></textarea>
					<div class="topranker-char-count" id="topranker-meta-description-count">
						<span class="count"><?php echo esc_html( mb_strlen( $current['meta_description'] ) ); ?></span> / 155
					</div>
				</div>
			</div>

			<!-- Excerpt Section -->
			<div class="topranker-section" id="topranker-excerpt-section">
				<div class="topranker-section-title">
					<span class="dashicons dashicons-text" aria-hidden="true"></span>
					<?php esc_html_e( 'Excerpt', 'topranker-ai' ); ?>
				</div>

				<div id="topranker-excerpt-result" class="topranker-excerpt-result">
					<?php if ( $has_suggestions && ! empty( $suggestions_data['suggestions']['excerpt']['excerpt'] ) ) : ?>
						<div class="topranker-suggestion is-selected">
							<div class="topranker-suggestion-content">
								<div class="topranker-suggestion-text"><?php echo esc_html( $suggestions_data['suggestions']['excerpt']['excerpt'] ); ?></div>
							</div>
						</div>
					<?php endif; ?>
				</div>

				<div class="topranker-editable-field">
					<label for="topranker-excerpt-custom"><?php esc_html_e( 'Excerpt:', 'topranker-ai' ); ?></label>
					<textarea id="topranker-excerpt-custom" name="topranker_excerpt_custom" maxlength="350" rows="3"><?php echo esc_textarea( $current['excerpt'] ); ?></textarea>
					<div class="topranker-char-count" id="topranker-excerpt-count">
						<span class="count"><?php echo esc_html( mb_strlen( $current['excerpt'] ) ); ?></span> / 300
					</div>
				</div>
			</div>

			<!-- Focus Keyphrase Section -->
			<div class="topranker-section" id="topranker-keyphrase-section">
				<div class="topranker-section-title">
					<span class="dashicons dashicons-tag" aria-hidden="true"></span>
					<?php esc_html_e( 'Focus Keyphrase', 'topranker-ai' ); ?>
				</div>

				<div id="topranker-keyphrases" class="topranker-keyphrases">
					<?php if ( $has_suggestions && ! empty( $suggestions_data['suggestions']['keyphrases'] ) ) : ?>
						<div class="topranker-keyphrase-primary" id="topranker-keyphrase-primary">
							<div class="topranker-keyphrase-label"><?php esc_html_e( 'Primary', 'topranker-ai' ); ?></div>
							<div class="topranker-keyphrase-value"><?php echo esc_html( $suggestions_data['suggestions']['keyphrases']['primary'] ); ?></div>
						</div>

						<?php if ( ! empty( $suggestions_data['suggestions']['keyphrases']['secondary'] ) ) : ?>
							<div class="topranker-keyphrase-secondary" id="topranker-keyphrase-secondary">
								<div class="topranker-keyphrase-label" style="width: 100%; margin-bottom: 6px;"><?php esc_html_e( 'Secondary', 'topranker-ai' ); ?></div>
								<?php foreach ( $suggestions_data['suggestions']['keyphrases']['secondary'] as $keyphrase ) : ?>
									<span class="topranker-keyphrase-tag"><?php echo esc_html( $keyphrase ); ?></span>
								<?php endforeach; ?>
							</div>
						<?php endif; ?>
					<?php endif; ?>
				</div>

				<div class="topranker-editable-field">
					<label for="topranker-keyphrase-custom"><?php esc_html_e( 'Focus keyphrase:', 'topranker-ai' ); ?></label>
					<input type="text" id="topranker-keyphrase-custom" name="topranker_keyphrase_custom" value="<?php echo esc_attr( $current['focus_keyphrase'] ); ?>" />
				</div>
			</div>

			<!-- Open Graph Section -->
			<div class="topranker-section" id="topranker-social-section">
				<div class="topranker-section-title">
					<span class="dashicons dashicons-share" aria-hidden="true"></span>
					<?php esc_html_e( 'Social Meta (Open Graph)', 'topranker-ai' ); ?>
				</div>

				<div class="topranker-social-preview" id="topranker-social-preview">
					<div class="topranker-social-card topranker-social-facebook">
						<div class="topranker-social-card-header">
							<span class="dashicons dashicons-facebook-alt" aria-hidden="true"></span>
							<?php esc_html_e( 'Facebook / Open Graph', 'topranker-ai' ); ?>
						</div>
						<div class="topranker-social-card-content">
							<div class="topranker-editable-field">
								<label for="topranker-og-title"><?php esc_html_e( 'OG Title:', 'topranker-ai' ); ?></label>
								<input type="text" id="topranker-og-title" name="topranker_og_title" value="<?php echo esc_attr( $current['og_title'] ); ?>" maxlength="70" />
							</div>
							<div class="topranker-editable-field">
								<label for="topranker-og-description"><?php esc_html_e( 'OG Description:', 'topranker-ai' ); ?></label>
								<textarea id="topranker-og-description" name="topranker_og_description" maxlength="250" rows="2"><?php echo esc_textarea( $current['og_description'] ); ?></textarea>
							</div>
						</div>
					</div>

					<div class="topranker-social-card topranker-social-twitter">
						<div class="topranker-social-card-header">
							<span class="dashicons dashicons-twitter" aria-hidden="true"></span>
							<?php esc_html_e( 'Twitter Card', 'topranker-ai' ); ?>
						</div>
						<div class="topranker-social-card-content">
							<div class="topranker-editable-field">
								<label for="topranker-twitter-title"><?php esc_html_e( 'Twitter Title:', 'topranker-ai' ); ?></label>
								<input type="text" id="topranker-twitter-title" name="topranker_twitter_title" value="<?php echo esc_attr( $current['twitter_title'] ); ?>" maxlength="70" />
							</div>
							<div class="topranker-editable-field">
								<label for="topranker-twitter-description"><?php esc_html_e( 'Twitter Description:', 'topranker-ai' ); ?></label>
								<textarea id="topranker-twitter-description" name="topranker_twitter_description" maxlength="250" rows="2"><?php echo esc_textarea( $current['twitter_description'] ); ?></textarea>
							</div>
						</div>
					</div>
				</div>
			</div>

			<!-- Alt Tags Section (Pro) -->
			<?php if ( $is_pro ) : ?>
				<div class="topranker-section" id="topranker-alt-tags-section">
					<div class="topranker-section-title">
						<span class="dashicons dashicons-images-alt2" aria-hidden="true"></span>
						<?php esc_html_e( 'Image Alt Tags', 'topranker-ai' ); ?>
					</div>

					<div id="topranker-alt-tags-results" class="topranker-alt-tags-container">
						<p class="topranker-alt-tags-placeholder">
							<?php esc_html_e( 'Alt tags will be generated when you optimize this post.', 'topranker-ai' ); ?>
						</p>
					</div>
				</div>
			<?php endif; ?>

			<!-- SERP Preview Section -->
			<div class="topranker-section" id="topranker-serp-section">
				<div class="topranker-section-title">
					<span class="dashicons dashicons-google" aria-hidden="true"></span>
					<?php esc_html_e( 'SERP Preview', 'topranker-ai' ); ?>
				</div>

				<div class="topranker-serp-preview" id="topranker-serp-preview">
					<div class="topranker-serp-preview-label"><?php esc_html_e( 'Google Search Preview', 'topranker-ai' ); ?></div>
					<a href="#" class="topranker-serp-title" id="topranker-serp-title">
						<?php echo esc_html( ! empty( $current['meta_title'] ) ? $current['meta_title'] : $post->post_title ); ?>
					</a>
					<div class="topranker-serp-url">
						<span class="topranker-serp-favicon"></span>
						<cite id="topranker-serp-url"><?php echo esc_html( get_permalink( $post->ID ) ); ?></cite>
					</div>
					<div class="topranker-serp-description" id="topranker-serp-description">
						<?php
						$description = ! empty( $current['meta_description'] ) ? $current['meta_description'] : wp_trim_words( $post->post_content, 25, '...' );
						echo esc_html( $description );
						?>
					</div>
					<div class="topranker-serp-counts">
						<span id="topranker-serp-title-count" class="topranker-char-count is-good">
							<?php esc_html_e( 'Title:', 'topranker-ai' ); ?>
							<span class="count"><?php echo esc_html( mb_strlen( ! empty( $current['meta_title'] ) ? $current['meta_title'] : $post->post_title ) ); ?></span>/60
						</span>
						<span id="topranker-serp-description-count" class="topranker-char-count is-good">
							<?php esc_html_e( 'Description:', 'topranker-ai' ); ?>
							<span class="count"><?php echo esc_html( mb_strlen( $current['meta_description'] ) ); ?></span>/155
						</span>
					</div>
				</div>
			</div>

			<!-- Optimization History Section (Pro) -->
			<?php if ( $is_pro ) : ?>
				<?php
				$history       = class_exists( 'TopRanker_History' ) ? new TopRanker_History() : null;
				$history_count = $history ? $history->get_count( $post->ID ) : 0;
				$history_data  = $history ? $history->get_history( $post->ID ) : array();
				?>
				<div class="topranker-section topranker-history-section" id="topranker-history-section">
					<div class="topranker-section-title">
						<span class="dashicons dashicons-backup" aria-hidden="true"></span>
						<?php esc_html_e( 'Optimization History', 'topranker-ai' ); ?>
						<?php if ( $history_count > 0 ) : ?>
							<span class="topranker-history-count"><?php echo esc_html( $history_count ); ?></span>
						<?php endif; ?>
					</div>

					<div id="topranker-history-container" class="topranker-history-container">
						<?php if ( $history_count > 0 ) : ?>
							<div class="topranker-history-list">
								<?php foreach ( $history_data as $index => $entry ) : ?>
									<div class="topranker-history-entry" data-index="<?php echo esc_attr( $index ); ?>">
										<div class="topranker-history-entry-header">
											<span class="topranker-history-date">
												<?php echo esc_html( wp_date( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $entry['timestamp'] ) ); ?>
											</span>
											<span class="topranker-history-user">
												<?php echo esc_html( isset( $entry['user_name'] ) ? $entry['user_name'] : __( 'Unknown', 'topranker-ai' ) ); ?>
											</span>
										</div>
										<div class="topranker-history-entry-summary">
											<?php
											$fields_count = 0;
											if ( isset( $entry['values'] ) && is_array( $entry['values'] ) ) {
												foreach ( $entry['values'] as $value ) {
													if ( ! empty( $value ) ) {
														$fields_count++;
													}
												}
											}
											printf(
												/* translators: %d: Number of fields */
												esc_html( _n( '%d field', '%d fields', $fields_count, 'topranker-ai' ) ),
												$fields_count
											);
											?>
										</div>
										<div class="topranker-history-entry-actions">
											<button type="button" class="button button-small topranker-history-preview-btn" data-index="<?php echo esc_attr( $index ); ?>">
												<span class="dashicons dashicons-visibility" aria-hidden="true"></span>
												<?php esc_html_e( 'Preview', 'topranker-ai' ); ?>
											</button>
											<button type="button" class="button button-small topranker-history-revert-btn" data-index="<?php echo esc_attr( $index ); ?>">
												<span class="dashicons dashicons-undo" aria-hidden="true"></span>
												<?php esc_html_e( 'Revert', 'topranker-ai' ); ?>
											</button>
										</div>
									</div>
								<?php endforeach; ?>
							</div>
						<?php else : ?>
							<div class="topranker-history-empty">
								<span class="dashicons dashicons-clock" aria-hidden="true"></span>
								<p><?php esc_html_e( 'No optimization history yet. History will be saved when you apply changes.', 'topranker-ai' ); ?></p>
							</div>
						<?php endif; ?>
					</div>

					<!-- History Preview Modal -->
					<div id="topranker-history-preview-modal" class="topranker-modal" style="display: none;">
						<div class="topranker-modal-content">
							<div class="topranker-modal-header">
								<h3><?php esc_html_e( 'History Entry Preview', 'topranker-ai' ); ?></h3>
								<button type="button" class="topranker-modal-close" aria-label="<?php esc_attr_e( 'Close', 'topranker-ai' ); ?>">
									<span class="dashicons dashicons-no-alt" aria-hidden="true"></span>
								</button>
							</div>
							<div class="topranker-modal-body" id="topranker-history-preview-content">
								<!-- Content loaded via AJAX -->
							</div>
							<div class="topranker-modal-footer">
								<button type="button" class="button topranker-modal-close-btn"><?php esc_html_e( 'Close', 'topranker-ai' ); ?></button>
								<button type="button" class="button button-primary topranker-history-revert-confirm-btn"><?php esc_html_e( 'Revert to This Version', 'topranker-ai' ); ?></button>
							</div>
						</div>
					</div>
				</div>
			<?php endif; ?>

			<!-- Pro Features Section -->
			<?php if ( ! $is_pro ) : ?>
				<div class="topranker-section topranker-pro-section">
					<div class="topranker-section-title">
						<span class="dashicons dashicons-star-filled" aria-hidden="true"></span>
						<?php esc_html_e( 'Pro Features', 'topranker-ai' ); ?>
					</div>

					<div class="topranker-pro-lock">
						<div class="topranker-pro-lock-icon">
							<span class="dashicons dashicons-lock" aria-hidden="true"></span>
						</div>
						<div class="topranker-pro-lock-content">
							<div class="topranker-pro-lock-title"><?php esc_html_e( 'Unlock More Features', 'topranker-ai' ); ?></div>
							<div class="topranker-pro-lock-description">
								<?php esc_html_e( 'Image alt tags, internal links, schema markup, SEO audit, optimization history, and more.', 'topranker-ai' ); ?>
							</div>
							<?php if ( function_exists( 'topranker_fs' ) ) : ?>
								<a href="<?php echo esc_url( topranker_fs()->get_upgrade_url() ); ?>" class="topranker-pro-lock-link">
									<?php esc_html_e( 'Upgrade to Pro', 'topranker-ai' ); ?>
									<span class="dashicons dashicons-arrow-right-alt2" aria-hidden="true"></span>
								</a>
							<?php endif; ?>
						</div>
					</div>
				</div>
			<?php endif; ?>

			<!-- Apply Button Section -->
			<div class="topranker-apply-section">
				<span class="topranker-apply-info">
					<?php
					if ( ! empty( $suggestions_data['last_optimized'] ) ) {
						printf(
							/* translators: %s: Time ago */
							esc_html__( 'Last optimized: %s', 'topranker-ai' ),
							esc_html( human_time_diff( $suggestions_data['last_optimized'], time() ) . ' ' . __( 'ago', 'topranker-ai' ) )
						);
					}
					?>
				</span>
				<div class="topranker-apply-buttons">
					<button type="button" id="topranker-apply-btn" class="button button-primary topranker-apply-btn">
						<span class="dashicons dashicons-yes" aria-hidden="true"></span>
						<?php
						if ( $is_sync_mode && $has_seo_plugin ) {
							printf(
								/* translators: %s: SEO plugin name */
								esc_html__( 'Apply & Sync to %s', 'topranker-ai' ),
								esc_html( $seo_plugin_name )
							);
						} else {
							esc_html_e( 'Apply Selected', 'topranker-ai' );
						}
						?>
					</button>
					<?php if ( $is_suggest_mode && $has_seo_plugin ) : ?>
						<button type="button" id="topranker-copy-to-seo-btn" class="button topranker-copy-btn" data-seo-plugin="<?php echo esc_attr( $seo_plugin_name ); ?>">
							<span class="dashicons dashicons-migrate" aria-hidden="true"></span>
							<?php
							printf(
								/* translators: %s: SEO plugin name */
								esc_html__( 'Copy to %s', 'topranker-ai' ),
								esc_html( $seo_plugin_name )
							);
							?>
						</button>
					<?php endif; ?>
				</div>
			</div>

			<?php if ( $is_sync_mode && $has_seo_plugin ) : ?>
				<!-- Sync Mode Notice -->
				<div class="topranker-notice is-info topranker-sync-notice">
					<div class="topranker-notice-icon">
						<span class="dashicons dashicons-update" aria-hidden="true"></span>
					</div>
					<div class="topranker-notice-content">
						<div class="topranker-notice-message">
							<?php
							printf(
								/* translators: %s: SEO plugin name */
								esc_html__( 'Sync mode enabled. Applying changes will also update %s meta fields.', 'topranker-ai' ),
								esc_html( $seo_plugin_name )
							);
							?>
						</div>
					</div>
				</div>
			<?php elseif ( $is_suggest_mode && $has_seo_plugin ) : ?>
				<!-- Suggest Mode Notice -->
				<div class="topranker-notice is-info topranker-suggest-notice">
					<div class="topranker-notice-icon">
						<span class="dashicons dashicons-info" aria-hidden="true"></span>
					</div>
					<div class="topranker-notice-content">
						<div class="topranker-notice-message">
							<?php
							printf(
								/* translators: %s: SEO plugin name */
								esc_html__( 'Suggest mode enabled. Use "Copy to %s" to transfer data to your SEO plugin.', 'topranker-ai' ),
								esc_html( $seo_plugin_name )
							);
							?>
						</div>
					</div>
				</div>
			<?php endif; ?>
		</div>

		<!-- Empty State (no suggestions yet) -->
		<div id="topranker-empty-state" class="topranker-empty-state" <?php echo ( $has_suggestions ) ? 'style="display: none;"' : ''; ?>>
			<div class="topranker-empty-state-icon">
				<span class="dashicons dashicons-chart-line" aria-hidden="true"></span>
			</div>
			<div class="topranker-empty-state-title"><?php esc_html_e( 'Ready to Optimize', 'topranker-ai' ); ?></div>
			<div class="topranker-empty-state-message">
				<?php esc_html_e( 'Click the "Optimize This Post" button above to generate SEO suggestions for your content.', 'topranker-ai' ); ?>
			</div>
		</div>

	<?php endif; ?>

</div>
