<?php
/**
 * Admin functionality.
 *
 * Handles admin pages, settings registration, metaboxes, and asset enqueueing.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * TopRanker Admin class.
 *
 * @since 1.0.0
 */
class TopRanker_Admin {

	/**
	 * Settings page slug.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	const MENU_SLUG = 'topranker-ai';

	/**
	 * Settings option group.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	const OPTION_GROUP = 'topranker_settings';

	/**
	 * Current settings tab.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	private $current_tab = 'general';

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		$this->init_hooks();
	}

	/**
	 * Initialize hooks.
	 *
	 * @since 1.0.0
	 */
	private function init_hooks() {
		// Admin menu.
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );

		// Register settings.
		add_action( 'admin_init', array( $this, 'register_settings' ) );

		// Enqueue admin assets.
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );

		// Add settings link to plugins page.
		add_filter( 'plugin_action_links_' . TOPRANKER_BASENAME, array( $this, 'add_settings_link' ) );

		// Register metabox for Classic Editor.
		add_action( 'add_meta_boxes', array( $this, 'register_metabox' ) );

		// Admin notices for onboarding.
		add_action( 'admin_notices', array( $this, 'display_admin_notices' ) );

		// AJAX handler to dismiss notices.
		add_action( 'wp_ajax_topranker_dismiss_notice', array( $this, 'ajax_dismiss_notice' ) );
	}

	/**
	 * Add admin menu pages.
	 *
	 * @since 1.0.0
	 */
	public function add_admin_menu() {
		// Main menu page.
		add_menu_page(
			__( 'TopRanker AI', 'topranker-ai' ),
			__( 'TopRanker AI', 'topranker-ai' ),
			'manage_options',
			self::MENU_SLUG,
			array( $this, 'render_settings_page' ),
			'dashicons-chart-line',
			80
		);

		// Settings submenu (same as main).
		add_submenu_page(
			self::MENU_SLUG,
			__( 'Settings', 'topranker-ai' ),
			__( 'Settings', 'topranker-ai' ),
			'manage_options',
			self::MENU_SLUG,
			array( $this, 'render_settings_page' )
		);

		// SEO Dashboard submenu.
		add_submenu_page(
			self::MENU_SLUG,
			__( 'SEO Dashboard', 'topranker-ai' ),
			__( 'SEO Dashboard', 'topranker-ai' ),
			'edit_posts',
			self::MENU_SLUG . '-dashboard',
			array( $this, 'render_dashboard_page' )
		);

		// Bulk Optimize submenu.
		add_submenu_page(
			self::MENU_SLUG,
			__( 'Bulk Optimize', 'topranker-ai' ),
			__( 'Bulk Optimize', 'topranker-ai' ),
			'edit_posts',
			self::MENU_SLUG . '-bulk',
			array( $this, 'render_bulk_page' )
		);
	}

	/**
	 * Register settings using the Settings API.
	 *
	 * @since 1.0.0
	 */
	public function register_settings() {
		// Register settings.
		register_setting(
			self::OPTION_GROUP,
			'topranker_api_key',
			array(
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
				'default'           => '',
			)
		);

		register_setting(
			self::OPTION_GROUP,
			'topranker_model',
			array(
				'type'              => 'string',
				'sanitize_callback' => array( $this, 'sanitize_model' ),
				'default'           => 'gpt-5.2',
			)
		);

		register_setting(
			self::OPTION_GROUP,
			'topranker_tone',
			array(
				'type'              => 'string',
				'sanitize_callback' => array( $this, 'sanitize_tone' ),
				'default'           => 'professional',
			)
		);

		register_setting(
			self::OPTION_GROUP,
			'topranker_post_types',
			array(
				'type'              => 'array',
				'sanitize_callback' => array( $this, 'sanitize_post_types' ),
				'default'           => array( 'post', 'page' ),
			)
		);

		register_setting(
			self::OPTION_GROUP,
			'topranker_seo_mode',
			array(
				'type'              => 'string',
				'sanitize_callback' => array( $this, 'sanitize_seo_mode' ),
				'default'           => 'standalone',
			)
		);

		register_setting(
			self::OPTION_GROUP,
			'topranker_auto_optimize',
			array(
				'type'              => 'boolean',
				'sanitize_callback' => 'rest_sanitize_boolean',
				'default'           => true,
			)
		);

		register_setting(
			self::OPTION_GROUP,
			'topranker_auto_alt_tags',
			array(
				'type'              => 'boolean',
				'sanitize_callback' => 'rest_sanitize_boolean',
				'default'           => true,
			)
		);

		register_setting(
			self::OPTION_GROUP,
			'topranker_auto_schema',
			array(
				'type'              => 'boolean',
				'sanitize_callback' => 'rest_sanitize_boolean',
				'default'           => true,
			)
		);

		register_setting(
			self::OPTION_GROUP,
			'topranker_auto_internal_links',
			array(
				'type'              => 'boolean',
				'sanitize_callback' => 'rest_sanitize_boolean',
				'default'           => true,
			)
		);

		register_setting(
			self::OPTION_GROUP,
			'topranker_acf_enabled',
			array(
				'type'              => 'boolean',
				'sanitize_callback' => 'rest_sanitize_boolean',
				'default'           => true,
			)
		);

		register_setting(
			self::OPTION_GROUP,
			'topranker_acf_field_types',
			array(
				'type'              => 'array',
				'sanitize_callback' => array( $this, 'sanitize_acf_field_types' ),
				'default'           => array( 'text', 'textarea', 'wysiwyg' ),
			)
		);

		// General section.
		add_settings_section(
			'topranker_general_section',
			__( 'API Configuration', 'topranker-ai' ),
			array( $this, 'render_general_section' ),
			self::MENU_SLUG
		);

		// API Key field.
		add_settings_field(
			'topranker_api_key',
			__( 'OpenAI API Key', 'topranker-ai' ),
			array( $this, 'render_api_key_field' ),
			self::MENU_SLUG,
			'topranker_general_section'
		);

		// Model field.
		add_settings_field(
			'topranker_model',
			__( 'AI Model', 'topranker-ai' ),
			array( $this, 'render_model_field' ),
			self::MENU_SLUG,
			'topranker_general_section'
		);

		// Tone field.
		add_settings_field(
			'topranker_tone',
			__( 'Tone of Voice', 'topranker-ai' ),
			array( $this, 'render_tone_field' ),
			self::MENU_SLUG,
			'topranker_general_section'
		);

		// Post types section.
		add_settings_section(
			'topranker_post_types_section',
			__( 'Content Types', 'topranker-ai' ),
			array( $this, 'render_post_types_section' ),
			self::MENU_SLUG
		);

		// Post types field.
		add_settings_field(
			'topranker_post_types',
			__( 'Enable TopRanker on', 'topranker-ai' ),
			array( $this, 'render_post_types_field' ),
			self::MENU_SLUG,
			'topranker_post_types_section'
		);

		// SEO compatibility section.
		add_settings_section(
			'topranker_seo_section',
			__( 'SEO Plugin Compatibility', 'topranker-ai' ),
			array( $this, 'render_seo_section' ),
			self::MENU_SLUG
		);

		// SEO mode field.
		add_settings_field(
			'topranker_seo_mode',
			__( 'Meta Tag Output', 'topranker-ai' ),
			array( $this, 'render_seo_mode_field' ),
			self::MENU_SLUG,
			'topranker_seo_section'
		);

		// ACF integration section (only shown when ACF is active).
		if ( function_exists( 'get_field_objects' ) ) {
			add_settings_section(
				'topranker_acf_section',
				__( 'Advanced Custom Fields', 'topranker-ai' ),
				array( $this, 'render_acf_section' ),
				self::MENU_SLUG
			);

			add_settings_field(
				'topranker_acf_enabled',
				__( 'Include ACF Content', 'topranker-ai' ),
				array( $this, 'render_acf_enabled_field' ),
				self::MENU_SLUG,
				'topranker_acf_section'
			);

			add_settings_field(
				'topranker_acf_field_types',
				__( 'Field Types to Include', 'topranker-ai' ),
				array( $this, 'render_acf_field_types_field' ),
				self::MENU_SLUG,
				'topranker_acf_section'
			);
		}

		// Pro features section (always shown, fields show lock for non-Pro).
		add_settings_section(
			'topranker_pro_section',
			__( 'Pro Features', 'topranker-ai' ),
			array( $this, 'render_pro_section' ),
			self::MENU_SLUG
		);

		// Auto-optimize on publish field.
		add_settings_field(
			'topranker_auto_optimize',
			__( 'Auto-Optimize on Publish', 'topranker-ai' ),
			array( $this, 'render_auto_optimize_field' ),
			self::MENU_SLUG,
			'topranker_pro_section'
		);

		// Auto alt tags field.
		add_settings_field(
			'topranker_auto_alt_tags',
			__( 'Auto Alt Tags', 'topranker-ai' ),
			array( $this, 'render_auto_alt_tags_field' ),
			self::MENU_SLUG,
			'topranker_pro_section'
		);

		// Auto schema field.
		add_settings_field(
			'topranker_auto_schema',
			__( 'Auto Schema Markup', 'topranker-ai' ),
			array( $this, 'render_auto_schema_field' ),
			self::MENU_SLUG,
			'topranker_pro_section'
		);

		// Auto internal links field.
		add_settings_field(
			'topranker_auto_internal_links',
			__( 'Auto Internal Links', 'topranker-ai' ),
			array( $this, 'render_auto_internal_links_field' ),
			self::MENU_SLUG,
			'topranker_pro_section'
		);
	}

	/**
	 * Enqueue admin assets.
	 *
	 * @since 1.0.0
	 * @param string $hook_suffix The current admin page hook suffix.
	 */
	public function enqueue_admin_assets( $hook_suffix ) {
		// Plugin settings pages.
		$plugin_pages = array(
			'toplevel_page_' . self::MENU_SLUG,
			'topranker-ai_page_' . self::MENU_SLUG . '-dashboard',
			'topranker-ai_page_' . self::MENU_SLUG . '-bulk',
		);

		$is_plugin_page    = in_array( $hook_suffix, $plugin_pages, true );
		$is_editor_page    = $this->is_post_edit_screen( $hook_suffix );
		$is_gutenberg_mode = $is_editor_page && $this->is_gutenberg_active();
		$is_classic_mode   = $is_editor_page && ! $this->is_gutenberg_active();

		// Return if not a relevant page.
		if ( ! $is_plugin_page && ! $is_editor_page ) {
			return;
		}

		// Always load admin CSS on relevant pages.
		wp_enqueue_style(
			'topranker-admin',
			TOPRANKER_URL . 'admin/css/topranker-admin.css',
			array(),
			TOPRANKER_VERSION
		);

		// Load settings page JS only on plugin pages.
		if ( $is_plugin_page ) {
			wp_enqueue_script(
				'topranker-admin',
				TOPRANKER_URL . 'admin/js/topranker-admin.js',
				array( 'jquery' ),
				TOPRANKER_VERSION,
				true
			);

			wp_localize_script(
				'topranker-admin',
				'toprankerAdmin',
				array(
					'ajaxUrl'     => admin_url( 'admin-ajax.php' ),
					'restUrl'     => rest_url( 'topranker/v1/' ),
					'nonce'       => wp_create_nonce( 'topranker_admin' ),
					'restNonce'   => wp_create_nonce( 'wp_rest' ),
					'isPro'       => topranker_is_pro(),
					'upgradeUrl'  => topranker_get_upgrade_url(),
					'i18n'        => array(
						'testing'          => __( 'Testing connection...', 'topranker-ai' ),
						'success'          => __( 'Connection successful!', 'topranker-ai' ),
						'error'            => __( 'Connection failed', 'topranker-ai' ),
						'invalidKey'       => __( 'Invalid API key. Please check your key.', 'topranker-ai' ),
						'networkError'     => __( 'Network error. Please try again.', 'topranker-ai' ),
						'speed'            => __( 'Speed', 'topranker-ai' ),
						'cost'             => __( 'Cost', 'topranker-ai' ),
						'usageText'        => __( '%1$d of %2$d optimizations used this month', 'topranker-ai' ),
						'resetsOn'         => __( 'Resets on %s', 'topranker-ai' ),
						'remainingWarning' => __( 'You have %d optimizations remaining this month.', 'topranker-ai' ),
						'refreshing'       => __( 'Refreshing...', 'topranker-ai' ),
					),
				)
			);
		}

		// Load Classic Editor JS on post edit screens (when not using Gutenberg).
		if ( $is_classic_mode ) {
			global $post;

			wp_enqueue_script(
				'topranker-editor',
				TOPRANKER_URL . 'admin/js/topranker-editor.js',
				array( 'jquery' ),
				TOPRANKER_VERSION,
				true
			);

			wp_localize_script(
				'topranker-editor',
				'toprankerEditor',
				TopRanker_AJAX::get_localized_data()
			);
		}

		// Load Gutenberg sidebar on block editor screens.
		if ( $is_gutenberg_mode ) {
			$this->enqueue_gutenberg_assets();
		}
	}

	/**
	 * Enqueue Gutenberg sidebar assets.
	 *
	 * @since 1.0.0
	 */
	private function enqueue_gutenberg_assets() {
		global $post;

		// Get the asset file for dependencies.
		$asset_file = TOPRANKER_PATH . 'admin/js/topranker-gutenberg.asset.php';
		$asset      = file_exists( $asset_file )
			? require $asset_file
			: array(
				'dependencies' => array(
					'wp-plugins',
					'wp-edit-post',
					'wp-element',
					'wp-components',
					'wp-data',
					'wp-i18n',
					'wp-wordcount',
				),
				'version'      => TOPRANKER_VERSION,
			);

		// Enqueue the Gutenberg sidebar script.
		wp_enqueue_script(
			'topranker-gutenberg',
			TOPRANKER_URL . 'admin/js/topranker-gutenberg.js',
			$asset['dependencies'],
			$asset['version'],
			true
		);

		// Get usage data.
		$usage = array(
			'used'      => 0,
			'limit'     => 25,
			'remaining' => 25,
			'resetDate' => '',
		);

		if ( class_exists( 'TopRanker_Usage' ) ) {
			$usage_tracker = new TopRanker_Usage();
			$usage_data    = $usage_tracker->get_usage();
			$usage         = array(
				'used'      => $usage_data['count'],
				'limit'     => $usage_tracker->get_limit(),
				'remaining' => $usage_tracker->get_remaining(),
				'resetDate' => $usage_data['reset_date'],
			);
		}

		// Get SEO compatibility data.
		$seo_compat_data = array(
			'hasPlugin'    => false,
			'pluginName'   => '',
			'pluginSlug'   => '',
			'mode'         => 'standalone',
			'isSync'       => false,
			'isSuggest'    => false,
			'isStandalone' => true,
		);

		if ( class_exists( 'TopRanker_SEO_Compat' ) ) {
			$seo_compat      = new TopRanker_SEO_Compat();
			$seo_compat_data = array(
				'hasPlugin'    => $seo_compat->has_seo_plugin(),
				'pluginName'   => $seo_compat->get_detected_plugin_name() ?: '',
				'pluginSlug'   => $seo_compat->get_detected_plugin() ?: '',
				'mode'         => $seo_compat->get_seo_mode(),
				'isSync'       => $seo_compat->is_sync_mode(),
				'isSuggest'    => $seo_compat->is_suggest_mode(),
				'isStandalone' => $seo_compat->is_standalone_mode(),
			);
		}

		// Localize script data.
		wp_localize_script(
			'topranker-gutenberg',
			'toprankerGutenberg',
			array(
				'apiKey'      => get_option( 'topranker_api_key', '' ) ? true : false,
				'isPro'       => topranker_is_pro(),
				'upgradeUrl'  => topranker_get_upgrade_url(),
				'settingsUrl' => admin_url( 'admin.php?page=' . self::MENU_SLUG ),
				'restUrl'     => rest_url( 'topranker/v1/' ),
				'nonce'       => wp_create_nonce( 'wp_rest' ),
				'postId'      => $post ? $post->ID : 0,
				'usage'       => $usage,
				'seoCompat'   => $seo_compat_data,
				'i18n'        => array(
					'optimizing'      => __( 'Optimizing...', 'topranker-ai' ),
					'generating'      => __( 'Generating suggestions...', 'topranker-ai' ),
					'error'           => __( 'An error occurred', 'topranker-ai' ),
					'retry'           => __( 'Retry', 'topranker-ai' ),
					'apply'           => __( 'Apply Selected', 'topranker-ai' ),
					'applied'         => __( 'Applied!', 'topranker-ai' ),
					'metaTitle'       => __( 'Meta Title', 'topranker-ai' ),
					'metaDescription' => __( 'Meta Description', 'topranker-ai' ),
					'excerpt'         => __( 'Excerpt', 'topranker-ai' ),
					'keyphrase'       => __( 'Focus Keyphrase', 'topranker-ai' ),
					'socialMeta'      => __( 'Social Meta', 'topranker-ai' ),
				),
			)
		);

		// Set script translations.
		wp_set_script_translations( 'topranker-gutenberg', 'topranker-ai' );
	}

	/**
	 * Add settings link to plugin action links.
	 *
	 * @since 1.0.0
	 * @param array $links Existing plugin action links.
	 * @return array Modified plugin action links.
	 */
	public function add_settings_link( $links ) {
		$settings_link = sprintf(
			'<a href="%s">%s</a>',
			admin_url( 'admin.php?page=' . self::MENU_SLUG ),
			esc_html__( 'Settings', 'topranker-ai' )
		);
		array_unshift( $links, $settings_link );
		return $links;
	}

	/**
	 * Render the settings page.
	 *
	 * @since 1.0.0
	 */
	public function render_settings_page() {
		// Check user capabilities.
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$view_file = TOPRANKER_PATH . 'admin/views/settings-page.php';
		if ( file_exists( $view_file ) ) {
			include $view_file;
		}
	}

	/**
	 * Render the dashboard page.
	 *
	 * @since 1.0.0
	 */
	public function render_dashboard_page() {
		if ( ! current_user_can( 'edit_posts' ) ) {
			return;
		}

		$view_file = TOPRANKER_PATH . 'admin/views/dashboard-page.php';
		if ( file_exists( $view_file ) ) {
			include $view_file;
		}
	}

	/**
	 * Render the bulk optimize page.
	 *
	 * @since 1.0.0
	 */
	public function render_bulk_page() {
		if ( ! current_user_can( 'edit_posts' ) || ! topranker_is_pro() ) {
			return;
		}

		$view_file = TOPRANKER_PATH . 'admin/views/bulk-page.php';
		if ( file_exists( $view_file ) ) {
			include $view_file;
		}
	}

	/**
	 * Render general section description.
	 *
	 * @since 1.0.0
	 */
	public function render_general_section() {
		echo '<p>' . esc_html__( 'Configure your OpenAI API settings to enable AI-powered SEO optimization.', 'topranker-ai' ) . '</p>';
	}

	/**
	 * Render post types section description.
	 *
	 * @since 1.0.0
	 */
	public function render_post_types_section() {
		echo '<p>' . esc_html__( 'Select which content types should have TopRanker AI optimization enabled.', 'topranker-ai' ) . '</p>';
	}

	/**
	 * Render SEO section description.
	 *
	 * @since 1.0.0
	 */
	public function render_seo_section() {
		$detected = $this->get_detected_seo_plugin();
		if ( $detected ) {
			printf(
				'<p>' . esc_html__( 'Detected SEO plugin: %s. Choose how TopRanker AI should work alongside it.', 'topranker-ai' ) . '</p>',
				'<strong>' . esc_html( $detected ) . '</strong>'
			);
		} else {
			echo '<p>' . esc_html__( 'No SEO plugin detected. TopRanker AI will output meta tags directly.', 'topranker-ai' ) . '</p>';
		}
	}

	/**
	 * Render ACF section description.
	 *
	 * @since 1.0.0
	 */
	public function render_acf_section() {
		echo '<p>' . esc_html__( 'TopRanker AI can include content from Advanced Custom Fields when generating SEO suggestions.', 'topranker-ai' ) . '</p>';
	}

	/**
	 * Render ACF enabled toggle field.
	 *
	 * @since 1.0.0
	 */
	public function render_acf_enabled_field() {
		$enabled = get_option( 'topranker_acf_enabled', true );
		?>
		<label>
			<input
				type="checkbox"
				id="topranker_acf_enabled"
				name="topranker_acf_enabled"
				value="1"
				<?php checked( $enabled ); ?>
				aria-describedby="topranker-acf-enabled-description"
			/>
			<?php esc_html_e( 'Include ACF field content in AI analysis', 'topranker-ai' ); ?>
		</label>
		<p class="description" id="topranker-acf-enabled-description">
			<?php esc_html_e( 'When enabled, text-based ACF fields are appended to post content before sending to the AI for SEO optimization.', 'topranker-ai' ); ?>
		</p>
		<?php
	}

	/**
	 * Render ACF field types checkboxes.
	 *
	 * @since 1.0.0
	 */
	public function render_acf_field_types_field() {
		$selected = get_option( 'topranker_acf_field_types', array( 'text', 'textarea', 'wysiwyg' ) );
		if ( ! is_array( $selected ) ) {
			$selected = array( 'text', 'textarea', 'wysiwyg' );
		}

		$field_types = array(
			'text'     => __( 'Text', 'topranker-ai' ),
			'textarea' => __( 'Textarea', 'topranker-ai' ),
			'wysiwyg'  => __( 'WYSIWYG Editor', 'topranker-ai' ),
			'url'      => __( 'URL', 'topranker-ai' ),
			'email'    => __( 'Email', 'topranker-ai' ),
		);
		?>
		<fieldset>
			<legend class="screen-reader-text"><?php esc_html_e( 'Field Types to Include', 'topranker-ai' ); ?></legend>
			<?php foreach ( $field_types as $value => $label ) : ?>
				<label>
					<input
						type="checkbox"
						name="topranker_acf_field_types[]"
						value="<?php echo esc_attr( $value ); ?>"
						<?php checked( in_array( $value, $selected, true ) ); ?>
					/>
					<?php echo esc_html( $label ); ?>
				</label><br>
			<?php endforeach; ?>
		</fieldset>
		<p class="description">
			<?php esc_html_e( 'Select which ACF field types to include in the content sent to the AI.', 'topranker-ai' ); ?>
		</p>
		<?php
	}

	/**
	 * Render Pro features section description.
	 *
	 * @since 1.0.0
	 */
	public function render_pro_section() {
		if ( topranker_is_pro() ) {
			echo '<p>' . esc_html__( 'Configure your Pro features below.', 'topranker-ai' ) . '</p>';
		} else {
			printf(
				'<p>' . esc_html__( 'Unlock these features with %s.', 'topranker-ai' ) . '</p>',
				'<a href="' . esc_url( topranker_get_upgrade_url() ) . '">' . esc_html__( 'TopRanker AI Pro', 'topranker-ai' ) . '</a>'
			);
		}
	}

	/**
	 * Render auto-optimize on publish field.
	 *
	 * @since 1.0.0
	 */
	public function render_auto_optimize_field() {
		$enabled = get_option( 'topranker_auto_optimize', true );
		$is_pro  = topranker_is_pro();
		?>
		<label class="topranker-toggle-wrapper<?php echo ! $is_pro ? ' topranker-pro-locked' : ''; ?>">
			<input
				type="checkbox"
				id="topranker_auto_optimize"
				name="topranker_auto_optimize"
				value="1"
				<?php checked( $enabled ); ?>
				<?php disabled( ! $is_pro ); ?>
				aria-describedby="topranker-auto-optimize-description"
			/>
			<?php esc_html_e( 'Automatically optimize new posts when published', 'topranker-ai' ); ?>
			<?php if ( ! $is_pro ) : ?>
				<span class="topranker-pro-badge">
					<span class="dashicons dashicons-lock" aria-hidden="true"></span>
					<?php esc_html_e( 'Pro', 'topranker-ai' ); ?>
				</span>
			<?php endif; ?>
		</label>
		<p class="description" id="topranker-auto-optimize-description">
			<?php esc_html_e( 'When enabled, TopRanker AI will automatically generate SEO metadata for new posts immediately after publishing. The optimization runs in the background via WP Cron.', 'topranker-ai' ); ?>
		</p>
		<?php if ( $is_pro && class_exists( 'TopRanker_Cron' ) ) : ?>
			<?php
			$cron       = new TopRanker_Cron();
			$pending    = $cron->get_pending_count();
			?>
			<?php if ( $pending > 0 ) : ?>
				<p class="topranker-pending-jobs">
					<span class="dashicons dashicons-clock" aria-hidden="true"></span>
					<?php
					printf(
						/* translators: %d: number of pending jobs */
						esc_html( _n(
							'%d post is queued for auto-optimization.',
							'%d posts are queued for auto-optimization.',
							$pending,
							'topranker-ai'
						) ),
						$pending
					);
					?>
				</p>
			<?php endif; ?>
		<?php endif; ?>
		<?php
	}

	/**
	 * Render auto alt tags field.
	 *
	 * @since 1.1.0
	 */
	public function render_auto_alt_tags_field() {
		$enabled = get_option( 'topranker_auto_alt_tags', true );
		$is_pro  = topranker_is_pro();
		?>
		<label class="topranker-toggle-wrapper<?php echo ! $is_pro ? ' topranker-pro-locked' : ''; ?>">
			<input
				type="checkbox"
				id="topranker_auto_alt_tags"
				name="topranker_auto_alt_tags"
				value="1"
				<?php checked( $enabled ); ?>
				<?php disabled( ! $is_pro ); ?>
				aria-describedby="topranker-auto-alt-tags-description"
			/>
			<?php esc_html_e( 'Generate alt tags for images during optimization', 'topranker-ai' ); ?>
			<?php if ( ! $is_pro ) : ?>
				<span class="topranker-pro-badge">
					<span class="dashicons dashicons-lock" aria-hidden="true"></span>
					<?php esc_html_e( 'Pro', 'topranker-ai' ); ?>
				</span>
			<?php endif; ?>
		</label>
		<p class="description" id="topranker-auto-alt-tags-description">
			<?php esc_html_e( 'When enabled, TopRanker AI will automatically generate SEO-optimized alt text for images when you optimize a post.', 'topranker-ai' ); ?>
		</p>
		<?php
	}

	/**
	 * Render auto schema field.
	 *
	 * @since 1.1.0
	 */
	public function render_auto_schema_field() {
		$enabled = get_option( 'topranker_auto_schema', true );
		$is_pro  = topranker_is_pro();
		?>
		<label class="topranker-toggle-wrapper<?php echo ! $is_pro ? ' topranker-pro-locked' : ''; ?>">
			<input
				type="checkbox"
				id="topranker_auto_schema"
				name="topranker_auto_schema"
				value="1"
				<?php checked( $enabled ); ?>
				<?php disabled( ! $is_pro ); ?>
				aria-describedby="topranker-auto-schema-description"
			/>
			<?php esc_html_e( 'Generate schema markup during optimization', 'topranker-ai' ); ?>
			<?php if ( ! $is_pro ) : ?>
				<span class="topranker-pro-badge">
					<span class="dashicons dashicons-lock" aria-hidden="true"></span>
					<?php esc_html_e( 'Pro', 'topranker-ai' ); ?>
				</span>
			<?php endif; ?>
		</label>
		<p class="description" id="topranker-auto-schema-description">
			<?php esc_html_e( 'When enabled, TopRanker AI will automatically generate JSON-LD schema markup (Article, FAQ, HowTo) when you optimize a post.', 'topranker-ai' ); ?>
		</p>
		<?php
	}

	/**
	 * Render auto internal links field.
	 *
	 * @since 1.1.0
	 */
	public function render_auto_internal_links_field() {
		$enabled = get_option( 'topranker_auto_internal_links', true );
		$is_pro  = topranker_is_pro();
		?>
		<label class="topranker-toggle-wrapper<?php echo ! $is_pro ? ' topranker-pro-locked' : ''; ?>">
			<input
				type="checkbox"
				id="topranker_auto_internal_links"
				name="topranker_auto_internal_links"
				value="1"
				<?php checked( $enabled ); ?>
				<?php disabled( ! $is_pro ); ?>
				aria-describedby="topranker-auto-internal-links-description"
			/>
			<?php esc_html_e( 'Generate internal linking suggestions during optimization', 'topranker-ai' ); ?>
			<?php if ( ! $is_pro ) : ?>
				<span class="topranker-pro-badge">
					<span class="dashicons dashicons-lock" aria-hidden="true"></span>
					<?php esc_html_e( 'Pro', 'topranker-ai' ); ?>
				</span>
			<?php endif; ?>
		</label>
		<p class="description" id="topranker-auto-internal-links-description">
			<?php esc_html_e( 'When enabled, TopRanker AI will suggest and auto-insert relevant internal links between your posts during optimization.', 'topranker-ai' ); ?>
		</p>
		<?php
	}

	/**
	 * Render API key field.
	 *
	 * @since 1.0.0
	 */
	public function render_api_key_field() {
		$api_key = get_option( 'topranker_api_key', '' );
		?>
		<div class="topranker-api-key-wrapper">
			<input
				type="password"
				id="topranker_api_key"
				name="topranker_api_key"
				value="<?php echo esc_attr( $api_key ); ?>"
				class="regular-text"
				autocomplete="off"
				aria-describedby="topranker-api-key-description"
			/>
			<button
				type="button"
				id="topranker-test-connection"
				class="button button-secondary"
				aria-label="<?php esc_attr_e( 'Test API connection', 'topranker-ai' ); ?>"
			>
				<?php esc_html_e( 'Test Connection', 'topranker-ai' ); ?>
			</button>
			<span id="topranker-connection-status" class="topranker-status" aria-live="polite"></span>
		</div>
		<p class="description" id="topranker-api-key-description">
			<?php
			printf(
				/* translators: %s: OpenAI platform URL */
				esc_html__( 'Enter your OpenAI API key. Get one at %s', 'topranker-ai' ),
				'<a href="https://platform.openai.com/api-keys" target="_blank" rel="noopener noreferrer">platform.openai.com</a>'
			);
			?>
		</p>
		<?php
	}

	/**
	 * Render model selection field.
	 *
	 * @since 1.0.0
	 */
	public function render_model_field() {
		$model   = get_option( 'topranker_model', 'gpt-5.2' );
		$models  = $this->get_available_models();
		?>
		<select id="topranker_model" name="topranker_model" aria-describedby="topranker-model-description">
			<?php foreach ( $models as $value => $label ) : ?>
				<option value="<?php echo esc_attr( $value ); ?>" <?php selected( $model, $value ); ?>>
					<?php echo esc_html( $label ); ?>
				</option>
			<?php endforeach; ?>
		</select>
		<p class="description" id="topranker-model-description">
			<?php esc_html_e( 'Select the AI model. GPT-5.2 delivers the best quality SEO content.', 'topranker-ai' ); ?>
		</p>
		<?php
	}

	/**
	 * Render tone selection field.
	 *
	 * @since 1.0.0
	 */
	public function render_tone_field() {
		$tone  = get_option( 'topranker_tone', 'professional' );
		$tones = $this->get_available_tones();
		?>
		<select id="topranker_tone" name="topranker_tone" aria-describedby="topranker-tone-description">
			<?php foreach ( $tones as $value => $label ) : ?>
				<option value="<?php echo esc_attr( $value ); ?>" <?php selected( $tone, $value ); ?>>
					<?php echo esc_html( $label ); ?>
				</option>
			<?php endforeach; ?>
		</select>
		<p class="description" id="topranker-tone-description">
			<?php esc_html_e( 'Choose the tone of voice for generated content.', 'topranker-ai' ); ?>
		</p>
		<?php
	}

	/**
	 * Render post types checkboxes field.
	 *
	 * @since 1.0.0
	 */
	public function render_post_types_field() {
		$enabled_types = get_option( 'topranker_post_types', array( 'post', 'page' ) );
		if ( ! is_array( $enabled_types ) ) {
			$enabled_types = array( 'post', 'page' );
		}

		$post_types = get_post_types(
			array(
				'public' => true,
			),
			'objects'
		);

		// Remove attachment.
		unset( $post_types['attachment'] );

		?>
		<fieldset>
			<legend class="screen-reader-text"><?php esc_html_e( 'Enable TopRanker on', 'topranker-ai' ); ?></legend>
			<?php foreach ( $post_types as $post_type ) : ?>
				<label>
					<input
						type="checkbox"
						name="topranker_post_types[]"
						value="<?php echo esc_attr( $post_type->name ); ?>"
						<?php checked( in_array( $post_type->name, $enabled_types, true ) ); ?>
					/>
					<?php echo esc_html( $post_type->labels->name ); ?>
				</label><br>
			<?php endforeach; ?>
		</fieldset>
		<?php
	}

	/**
	 * Render SEO mode selection field.
	 *
	 * @since 1.0.0
	 */
	public function render_seo_mode_field() {
		$mode  = get_option( 'topranker_seo_mode', 'standalone' );
		$modes = $this->get_seo_modes();
		?>
		<fieldset>
			<legend class="screen-reader-text"><?php esc_html_e( 'Meta Tag Output', 'topranker-ai' ); ?></legend>
			<?php foreach ( $modes as $value => $data ) : ?>
				<label>
					<input
						type="radio"
						name="topranker_seo_mode"
						value="<?php echo esc_attr( $value ); ?>"
						<?php checked( $mode, $value ); ?>
					/>
					<strong><?php echo esc_html( $data['label'] ); ?></strong>
					<span class="description"><?php echo esc_html( $data['description'] ); ?></span>
				</label><br>
			<?php endforeach; ?>
		</fieldset>
		<p class="topranker-privacy-notice">
			<span class="dashicons dashicons-info" aria-hidden="true"></span>
			<?php esc_html_e( 'TopRanker AI sends your post content to OpenAI\'s API for analysis. No data is stored on our servers.', 'topranker-ai' ); ?>
		</p>
		<?php
	}

	/**
	 * Get available AI models.
	 *
	 * @since  1.0.0
	 * @return array Model value => label pairs.
	 */
	public function get_available_models() {
		return array(
			'gpt-5.2'      => __( 'GPT-5.2 (Recommended)', 'topranker-ai' ),
			'gpt-4.1-mini' => __( 'GPT-4.1 Mini (Fast & Cheap)', 'topranker-ai' ),
			'gpt-4.1-nano' => __( 'GPT-4.1 Nano (Cheapest)', 'topranker-ai' ),
		);
	}

	/**
	 * Get available tones.
	 *
	 * @since  1.0.0
	 * @return array Tone value => label pairs.
	 */
	public function get_available_tones() {
		return array(
			'professional' => __( 'Professional', 'topranker-ai' ),
			'casual'       => __( 'Casual', 'topranker-ai' ),
			'technical'    => __( 'Technical', 'topranker-ai' ),
			'friendly'     => __( 'Friendly', 'topranker-ai' ),
		);
	}

	/**
	 * Get SEO mode options.
	 *
	 * @since  1.0.0
	 * @return array Mode value => data pairs.
	 */
	public function get_seo_modes() {
		return array(
			'standalone' => array(
				'label'       => __( 'Standalone', 'topranker-ai' ),
				'description' => __( 'TopRanker outputs meta tags directly in your site\'s HTML.', 'topranker-ai' ),
			),
			'suggest'    => array(
				'label'       => __( 'Suggest Only', 'topranker-ai' ),
				'description' => __( 'Generate suggestions only. Copy them to your SEO plugin manually.', 'topranker-ai' ),
			),
			'sync'       => array(
				'label'       => __( 'Sync', 'topranker-ai' ),
				'description' => __( 'Apply suggestions directly to your SEO plugin\'s fields.', 'topranker-ai' ),
			),
		);
	}

	/**
	 * Detect installed SEO plugin.
	 *
	 * @since  1.0.0
	 * @return string|false Plugin name or false if none detected.
	 */
	public function get_detected_seo_plugin() {
		if ( defined( 'WPSEO_VERSION' ) ) {
			return 'Yoast SEO';
		}

		if ( class_exists( 'RankMath' ) ) {
			return 'Rank Math';
		}

		if ( function_exists( 'seopress_init' ) ) {
			return 'SEOPress';
		}

		return false;
	}

	/**
	 * Sanitize model setting.
	 *
	 * @since  1.0.0
	 * @param  string $value The value to sanitize.
	 * @return string Sanitized value.
	 */
	public function sanitize_model( $value ) {
		$valid = array_keys( $this->get_available_models() );
		return in_array( $value, $valid, true ) ? $value : 'gpt-5.2';
	}

	/**
	 * Sanitize tone setting.
	 *
	 * @since  1.0.0
	 * @param  string $value The value to sanitize.
	 * @return string Sanitized value.
	 */
	public function sanitize_tone( $value ) {
		$valid = array_keys( $this->get_available_tones() );
		return in_array( $value, $valid, true ) ? $value : 'professional';
	}

	/**
	 * Sanitize post types setting.
	 *
	 * @since  1.0.0
	 * @param  array $value The value to sanitize.
	 * @return array Sanitized value.
	 */
	public function sanitize_post_types( $value ) {
		if ( ! is_array( $value ) ) {
			return array( 'post', 'page' );
		}

		$valid_types = get_post_types( array( 'public' => true ) );
		return array_values( array_intersect( $value, array_keys( $valid_types ) ) );
	}

	/**
	 * Sanitize SEO mode setting.
	 *
	 * @since  1.0.0
	 * @param  string $value The value to sanitize.
	 * @return string Sanitized value.
	 */
	public function sanitize_seo_mode( $value ) {
		$valid = array_keys( $this->get_seo_modes() );
		return in_array( $value, $valid, true ) ? $value : 'standalone';
	}

	/**
	 * Sanitize ACF field types setting.
	 *
	 * @since  1.0.0
	 * @param  array $value The value to sanitize.
	 * @return array Sanitized value.
	 */
	public function sanitize_acf_field_types( $value ) {
		if ( ! is_array( $value ) ) {
			return array( 'text', 'textarea', 'wysiwyg' );
		}

		$valid = array( 'text', 'textarea', 'wysiwyg', 'url', 'email' );
		return array_values( array_intersect( $value, $valid ) );
	}

	/**
	 * Get settings tabs.
	 *
	 * @since  1.0.0
	 * @return array Tab slug => label pairs.
	 */
	public function get_settings_tabs() {
		return array(
			'general' => __( 'General', 'topranker-ai' ),
			'usage'   => __( 'Usage', 'topranker-ai' ),
		);
	}

	/**
	 * Get current settings tab.
	 *
	 * @since  1.0.0
	 * @return string Current tab slug.
	 */
	public function get_current_tab() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Tab display only.
		$tab = isset( $_GET['tab'] ) ? sanitize_key( $_GET['tab'] ) : 'general';
		$tabs = array_keys( $this->get_settings_tabs() );
		return in_array( $tab, $tabs, true ) ? $tab : 'general';
	}

	/**
	 * Register the TopRanker AI metabox for Classic Editor.
	 *
	 * @since 1.0.0
	 */
	public function register_metabox() {
		$enabled_post_types = get_option( 'topranker_post_types', array( 'post', 'page' ) );

		if ( empty( $enabled_post_types ) ) {
			$enabled_post_types = array( 'post', 'page' );
		}

		foreach ( $enabled_post_types as $post_type ) {
			add_meta_box(
				'topranker-ai-metabox',
				__( 'TopRanker AI', 'topranker-ai' ),
				array( $this, 'render_metabox' ),
				$post_type,
				'normal',
				'high'
			);
		}
	}

	/**
	 * Render the TopRanker AI metabox content.
	 *
	 * @since 1.0.0
	 * @param WP_Post $post The current post object.
	 */
	public function render_metabox( $post ) {
		// Check user capabilities.
		if ( ! current_user_can( 'edit_post', $post->ID ) ) {
			return;
		}

		$view_file = TOPRANKER_PATH . 'admin/views/metabox.php';
		if ( file_exists( $view_file ) ) {
			include $view_file;
		}
	}

	/**
	 * Check if we're on a post edit screen.
	 *
	 * @since  1.0.0
	 * @param  string $hook_suffix The current admin page hook suffix.
	 * @return bool True if on a post edit screen.
	 */
	public function is_post_edit_screen( $hook_suffix ) {
		if ( 'post.php' !== $hook_suffix && 'post-new.php' !== $hook_suffix ) {
			return false;
		}

		global $post;

		if ( ! $post ) {
			return false;
		}

		$enabled_post_types = get_option( 'topranker_post_types', array( 'post', 'page' ) );

		if ( empty( $enabled_post_types ) ) {
			$enabled_post_types = array( 'post', 'page' );
		}

		return in_array( $post->post_type, $enabled_post_types, true );
	}

	/**
	 * Check if Gutenberg (Block Editor) is active for the current post.
	 *
	 * @since  1.0.0
	 * @return bool True if Gutenberg is active.
	 */
	public function is_gutenberg_active() {
		global $post;

		if ( ! $post ) {
			return false;
		}

		// Check if the Classic Editor plugin is active and forces classic mode.
		if ( class_exists( 'Classic_Editor' ) ) {
			$classic_editor = get_option( 'classic-editor-replace' );
			if ( 'classic' === $classic_editor ) {
				return false;
			}

			// Per-post setting.
			$post_option = get_post_meta( $post->ID, 'classic-editor-remember', true );
			if ( 'classic-editor' === $post_option ) {
				return false;
			}
		}

		// Check if the post type supports the block editor.
		return use_block_editor_for_post( $post );
	}

	/**
	 * Display admin notices for onboarding.
	 *
	 * @since 1.0.0
	 */
	public function display_admin_notices() {
		// Only show to users who can manage options.
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$api_key          = get_option( 'topranker_api_key', '' );
		$show_welcome     = get_option( 'topranker_show_welcome', false );
		$dismissed        = get_option( 'topranker_dismissed_notices', array() );
		$current_screen   = get_current_screen();
		$is_settings_page = $current_screen && 'toplevel_page_topranker-ai' === $current_screen->id;

		// Welcome notice on first activation (show on all admin pages except our settings).
		if ( $show_welcome && empty( $api_key ) && ! $is_settings_page ) {
			$this->render_welcome_notice();
			return;
		}

		// API key reminder on post edit screens.
		if ( empty( $api_key ) && $this->is_edit_screen_for_enabled_post_type() ) {
			if ( ! isset( $dismissed['api_key_reminder'] ) ) {
				$this->render_api_key_reminder();
			}
		}

		// Success notice after saving API key (shown on settings page).
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Display only.
		if ( $is_settings_page && isset( $_GET['settings-updated'] ) && ! empty( $api_key ) ) {
			$this->render_api_key_saved_notice();
		}
	}

	/**
	 * Render the welcome notice for first-time users.
	 *
	 * @since 1.0.0
	 */
	private function render_welcome_notice() {
		$settings_url = admin_url( 'admin.php?page=' . self::MENU_SLUG );
		?>
		<div class="notice notice-info is-dismissible topranker-welcome-notice" data-notice="welcome">
			<p>
				<strong><?php esc_html_e( 'Welcome to TopRanker AI!', 'topranker-ai' ); ?></strong>
				<?php esc_html_e( 'Enter your OpenAI API key to get started with AI-powered SEO optimization.', 'topranker-ai' ); ?>
			</p>
			<p>
				<a href="<?php echo esc_url( $settings_url ); ?>" class="button button-primary">
					<?php esc_html_e( 'Go to Settings', 'topranker-ai' ); ?>
				</a>
				<a href="https://platform.openai.com/api-keys" target="_blank" rel="noopener noreferrer" class="button button-secondary">
					<?php esc_html_e( 'Get an API Key', 'topranker-ai' ); ?>
				</a>
			</p>
		</div>
		<script>
		jQuery(document).ready(function($) {
			$('.topranker-welcome-notice').on('click', '.notice-dismiss', function() {
				$.post(ajaxurl, {
					action: 'topranker_dismiss_notice',
					notice: 'welcome',
					_wpnonce: '<?php echo esc_js( wp_create_nonce( 'topranker_dismiss_notice' ) ); ?>'
				});
			});
		});
		</script>
		<?php
	}

	/**
	 * Render the API key reminder notice on edit screens.
	 *
	 * @since 1.0.0
	 */
	private function render_api_key_reminder() {
		$settings_url = admin_url( 'admin.php?page=' . self::MENU_SLUG );
		?>
		<div class="notice notice-info is-dismissible topranker-api-reminder-notice" data-notice="api_key_reminder">
			<p>
				<span class="dashicons dashicons-chart-line" style="color: #2271b1;" aria-hidden="true"></span>
				<strong><?php esc_html_e( 'TopRanker AI:', 'topranker-ai' ); ?></strong>
				<?php
				printf(
					/* translators: %s: Settings page link */
					esc_html__( 'Set up your OpenAI API key in %s to enable AI-powered SEO optimization for your content.', 'topranker-ai' ),
					'<a href="' . esc_url( $settings_url ) . '">' . esc_html__( 'Settings', 'topranker-ai' ) . '</a>'
				);
				?>
			</p>
		</div>
		<script>
		jQuery(document).ready(function($) {
			$('.topranker-api-reminder-notice').on('click', '.notice-dismiss', function() {
				$.post(ajaxurl, {
					action: 'topranker_dismiss_notice',
					notice: 'api_key_reminder',
					_wpnonce: '<?php echo esc_js( wp_create_nonce( 'topranker_dismiss_notice' ) ); ?>'
				});
			});
		});
		</script>
		<?php
	}

	/**
	 * Render the API key saved success notice.
	 *
	 * @since 1.0.0
	 */
	private function render_api_key_saved_notice() {
		// Check if we just came from saving and have an API key.
		$test_success = get_transient( 'topranker_api_test_success' );

		if ( $test_success ) {
			delete_transient( 'topranker_api_test_success' );
			?>
			<div class="notice notice-success is-dismissible">
				<p>
					<span class="dashicons dashicons-yes-alt" style="color: #00a32a;" aria-hidden="true"></span>
					<strong><?php esc_html_e( 'API key saved and verified!', 'topranker-ai' ); ?></strong>
					<?php
					printf(
						/* translators: %s: Link to first post */
						esc_html__( 'You\'re all set! %s', 'topranker-ai' ),
						'<a href="' . esc_url( admin_url( 'edit.php' ) ) . '">' . esc_html__( 'Go optimize your first post!', 'topranker-ai' ) . '</a>'
					);
					?>
				</p>
			</div>
			<?php
		}
	}

	/**
	 * AJAX handler to dismiss notices.
	 *
	 * @since 1.0.0
	 */
	public function ajax_dismiss_notice() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'topranker_dismiss_notice', '_wpnonce', false ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid nonce.', 'topranker-ai' ) ) );
		}

		// Check capabilities.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'topranker-ai' ) ) );
		}

		// Get the notice type.
		$notice = isset( $_POST['notice'] ) ? sanitize_key( $_POST['notice'] ) : '';

		if ( empty( $notice ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid notice.', 'topranker-ai' ) ) );
		}

		// Handle welcome notice dismissal.
		if ( 'welcome' === $notice ) {
			delete_option( 'topranker_show_welcome' );
			wp_send_json_success();
		}

		// Handle other dismissible notices.
		$dismissed           = get_option( 'topranker_dismissed_notices', array() );
		$dismissed[ $notice ] = time();
		update_option( 'topranker_dismissed_notices', $dismissed );

		wp_send_json_success();
	}

	/**
	 * Check if current screen is an edit screen for an enabled post type.
	 *
	 * @since  1.0.0
	 * @return bool True if on edit screen for enabled post type.
	 */
	private function is_edit_screen_for_enabled_post_type() {
		$current_screen = get_current_screen();

		if ( ! $current_screen ) {
			return false;
		}

		// Check if we're on a post edit screen.
		if ( 'post' !== $current_screen->base ) {
			return false;
		}

		$enabled_post_types = get_option( 'topranker_post_types', array( 'post', 'page' ) );

		if ( empty( $enabled_post_types ) ) {
			$enabled_post_types = array( 'post', 'page' );
		}

		return in_array( $current_screen->post_type, $enabled_post_types, true );
	}
}
