<?php
/**
 * AJAX handlers for Classic Editor.
 *
 * Registers and handles all admin-ajax.php endpoints for the Classic Editor
 * metabox functionality. Mirrors the REST API endpoints for consistency.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * TopRanker AJAX class.
 *
 * @since 1.0.0
 */
class TopRanker_AJAX {

	/**
	 * Nonce action name for AJAX requests.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	const NONCE_ACTION = 'topranker_ajax_nonce';

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		$this->init_hooks();
	}

	/**
	 * Initialize AJAX hooks.
	 *
	 * @since 1.0.0
	 */
	private function init_hooks() {
		// Optimization action.
		add_action( 'wp_ajax_topranker_optimize', array( $this, 'handle_optimize' ) );

		// Apply suggestions action.
		add_action( 'wp_ajax_topranker_apply', array( $this, 'handle_apply' ) );

		// Get suggestions action.
		add_action( 'wp_ajax_topranker_get_suggestions', array( $this, 'handle_get_suggestions' ) );

		// Get usage stats action.
		add_action( 'wp_ajax_topranker_get_usage', array( $this, 'handle_get_usage' ) );

		// Bulk optimize action (Pro).
		add_action( 'wp_ajax_topranker_bulk_optimize', array( $this, 'handle_bulk_optimize' ) );

		// Copy to SEO plugin action.
		add_action( 'wp_ajax_topranker_copy_to_seo_plugin', array( $this, 'handle_copy_to_seo_plugin' ) );

		// Pro: Generate alt tags action.
		add_action( 'wp_ajax_topranker_generate_alt_tags', array( $this, 'handle_generate_alt_tags' ) );

		// Pro: Apply alt tags action.
		add_action( 'wp_ajax_topranker_apply_alt_tags', array( $this, 'handle_apply_alt_tags' ) );

		// Pro: Generate single attachment alt tag action.
		add_action( 'wp_ajax_topranker_generate_attachment_alt', array( $this, 'handle_generate_attachment_alt' ) );

		// Pro: Get SEO audit data action.
		add_action( 'wp_ajax_topranker_get_seo_audit', array( $this, 'handle_get_seo_audit' ) );

		// Pro: Get optimization history action.
		add_action( 'wp_ajax_topranker_get_history', array( $this, 'handle_get_history' ) );

		// Pro: Revert to history entry action.
		add_action( 'wp_ajax_topranker_revert_history', array( $this, 'handle_revert_history' ) );
	}

	/**
	 * Verify AJAX request.
	 *
	 * Checks nonce and user capabilities.
	 *
	 * @since  1.0.0
	 * @param  string $capability Required capability. Default 'edit_posts'.
	 * @return bool|WP_Error True if verified, WP_Error on failure.
	 */
	private function verify_request( $capability = 'edit_posts' ) {
		// Verify nonce.
		if ( ! check_ajax_referer( self::NONCE_ACTION, 'nonce', false ) ) {
			return new WP_Error(
				'invalid_nonce',
				__( 'Security check failed. Please refresh the page and try again.', 'topranker-ai' )
			);
		}

		// Check capability.
		if ( ! current_user_can( $capability ) ) {
			return new WP_Error(
				'insufficient_permissions',
				__( 'You do not have permission to perform this action.', 'topranker-ai' )
			);
		}

		return true;
	}

	/**
	 * Check if user can edit a specific post.
	 *
	 * @since  1.0.0
	 * @param  int $post_id Post ID.
	 * @return bool True if user can edit the post.
	 */
	private function can_edit_post( $post_id ) {
		$post = get_post( $post_id );

		if ( ! $post ) {
			return false;
		}

		return current_user_can( 'edit_post', $post_id );
	}

	/**
	 * Check if a post type is enabled for TopRanker.
	 *
	 * @since  1.0.0
	 * @param  string $post_type Post type slug.
	 * @return bool True if post type is enabled.
	 */
	private function is_post_type_enabled( $post_type ) {
		$enabled_post_types = get_option( 'topranker_post_types', array( 'post', 'page' ) );

		if ( empty( $enabled_post_types ) ) {
			$enabled_post_types = array( 'post', 'page' );
		}

		return in_array( $post_type, $enabled_post_types, true );
	}

	/**
	 * Send JSON error response.
	 *
	 * @since 1.0.0
	 * @param WP_Error|string $error   Error object or message.
	 * @param int             $status  HTTP status code. Default 400.
	 */
	private function send_error( $error, $status = 400 ) {
		$message = is_wp_error( $error ) ? $error->get_error_message() : $error;
		$code    = is_wp_error( $error ) ? $error->get_error_code() : 'error';

		wp_send_json_error(
			array(
				'code'    => $code,
				'message' => $message,
			),
			$status
		);
	}

	/**
	 * Send JSON success response.
	 *
	 * @since 1.0.0
	 * @param array $data Response data.
	 */
	private function send_success( $data ) {
		wp_send_json_success( $data );
	}

	/**
	 * Log debug message if debug mode is enabled.
	 *
	 * @since 1.0.0
	 * @param string $message Debug message.
	 */
	private function log_debug( $message ) {
		if ( defined( 'TOPRANKER_DEBUG' ) && TOPRANKER_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging.
			error_log( '[TopRanker AI AJAX] ' . $message );
		}
	}

	/**
	 * Handle optimize AJAX request.
	 *
	 * Triggers optimization for a single post.
	 *
	 * @since 1.0.0
	 */
	public function handle_optimize() {
		// Verify request.
		$verified = $this->verify_request( 'edit_posts' );
		if ( is_wp_error( $verified ) ) {
			$this->send_error( $verified, 403 );
		}

		// Get and sanitize post ID.
		$post_id = isset( $_POST['post_id'] ) ? absint( $_POST['post_id'] ) : 0;

		if ( $post_id <= 0 ) {
			$this->send_error(
				new WP_Error( 'invalid_post_id', __( 'Invalid post ID.', 'topranker-ai' ) ),
				400
			);
		}

		// Verify user can edit this specific post.
		if ( ! $this->can_edit_post( $post_id ) ) {
			$this->send_error(
				new WP_Error( 'forbidden', __( 'You do not have permission to edit this post.', 'topranker-ai' ) ),
				403
			);
		}

		$post = get_post( $post_id );

		if ( ! $post ) {
			$this->send_error(
				new WP_Error( 'not_found', __( 'Post not found.', 'topranker-ai' ) ),
				404
			);
		}

		// Check if post type is enabled.
		if ( ! $this->is_post_type_enabled( $post->post_type ) ) {
			$this->send_error(
				new WP_Error( 'post_type_disabled', __( 'TopRanker AI is not enabled for this post type.', 'topranker-ai' ) ),
				400
			);
		}

		// Get force refresh flag.
		$force_refresh = isset( $_POST['force_refresh'] ) && ( 'true' === $_POST['force_refresh'] || '1' === $_POST['force_refresh'] );

		// Run the optimization.
		$optimizer = new TopRanker_Optimizer();
		$result    = $optimizer->optimize( $post, $force_refresh );

		if ( is_wp_error( $result ) ) {
			$status = 400;

			// Map specific errors to status codes.
			$error_code = $result->get_error_code();
			if ( 'usage_limit_reached' === $error_code ) {
				$status = 429;
			} elseif ( 'invalid_post' === $error_code ) {
				$status = 404;
			}

			$this->send_error( $result, $status );
		}

		$this->log_debug( 'Optimization completed for post ' . $post_id );

		$this->send_success( $result );
	}

	/**
	 * Handle apply suggestions AJAX request.
	 *
	 * Saves selected optimization values to post meta.
	 *
	 * @since 1.0.0
	 */
	public function handle_apply() {
		// Verify request.
		$verified = $this->verify_request( 'edit_posts' );
		if ( is_wp_error( $verified ) ) {
			$this->send_error( $verified, 403 );
		}

		// Get and sanitize post ID.
		$post_id = isset( $_POST['post_id'] ) ? absint( $_POST['post_id'] ) : 0;

		if ( $post_id <= 0 ) {
			$this->send_error(
				new WP_Error( 'invalid_post_id', __( 'Invalid post ID.', 'topranker-ai' ) ),
				400
			);
		}

		// Verify user can edit this specific post.
		if ( ! $this->can_edit_post( $post_id ) ) {
			$this->send_error(
				new WP_Error( 'forbidden', __( 'You do not have permission to edit this post.', 'topranker-ai' ) ),
				403
			);
		}

		$post = get_post( $post_id );

		if ( ! $post ) {
			$this->send_error(
				new WP_Error( 'not_found', __( 'Post not found.', 'topranker-ai' ) ),
				404
			);
		}

		// Collect and sanitize selected values from request.
		$selected = array();

		$text_fields = array(
			'meta_title',
			'focus_keyphrase',
			'og_title',
			'twitter_title',
		);

		$textarea_fields = array(
			'meta_description',
			'excerpt',
			'og_description',
			'twitter_description',
		);

		foreach ( $text_fields as $field ) {
			if ( isset( $_POST[ $field ] ) && '' !== $_POST[ $field ] ) {
				$selected[ $field ] = sanitize_text_field( wp_unslash( $_POST[ $field ] ) );
			}
		}

		foreach ( $textarea_fields as $field ) {
			if ( isset( $_POST[ $field ] ) && '' !== $_POST[ $field ] ) {
				$selected[ $field ] = sanitize_textarea_field( wp_unslash( $_POST[ $field ] ) );
			}
		}

		// Handle secondary keyphrases (array).
		if ( isset( $_POST['secondary_keyphrases'] ) ) {
			$keyphrases = $_POST['secondary_keyphrases'];
			if ( is_string( $keyphrases ) ) {
				// Try to decode JSON.
				$keyphrases = json_decode( wp_unslash( $keyphrases ), true );
			}
			if ( is_array( $keyphrases ) ) {
				$selected['secondary_keyphrases'] = array_map( 'sanitize_text_field', $keyphrases );
			}
		}

		// Handle alt tags (array of objects).
		if ( isset( $_POST['alt_tags'] ) ) {
			$alt_tags_data = $_POST['alt_tags'];
			if ( is_string( $alt_tags_data ) ) {
				$alt_tags_data = json_decode( wp_unslash( $alt_tags_data ), true );
			}
			if ( is_array( $alt_tags_data ) && ! empty( $alt_tags_data ) ) {
				$sanitized_alt_tags = array();
				foreach ( $alt_tags_data as $item ) {
					if ( isset( $item['image_id'], $item['new_alt'] ) ) {
						$sanitized_alt_tags[] = array(
							'image_id' => absint( $item['image_id'] ),
							'new_alt'  => sanitize_text_field( $item['new_alt'] ),
						);
					}
				}
				if ( ! empty( $sanitized_alt_tags ) ) {
					$selected['alt_tags'] = $sanitized_alt_tags;
				}
			}
		}

		if ( empty( $selected ) ) {
			$this->send_error(
				new WP_Error( 'no_values', __( 'No values provided to apply.', 'topranker-ai' ) ),
				400
			);
		}

		// Apply the selected values.
		$optimizer = new TopRanker_Optimizer();
		$result    = $optimizer->apply( $post_id, $selected );

		if ( is_wp_error( $result ) ) {
			$this->send_error( $result, 400 );
		}

		$this->log_debug( 'Applied suggestions to post ' . $post_id );

		$this->send_success( $result );
	}

	/**
	 * Handle get suggestions AJAX request.
	 *
	 * Returns cached suggestions and current saved values for a post.
	 *
	 * @since 1.0.0
	 */
	public function handle_get_suggestions() {
		// Verify request.
		$verified = $this->verify_request( 'edit_posts' );
		if ( is_wp_error( $verified ) ) {
			$this->send_error( $verified, 403 );
		}

		// Get and sanitize post ID.
		$post_id = isset( $_POST['post_id'] ) ? absint( $_POST['post_id'] ) : 0;

		// Also support GET for convenience.
		if ( 0 === $post_id ) {
			$post_id = isset( $_GET['post_id'] ) ? absint( $_GET['post_id'] ) : 0;
		}

		if ( $post_id <= 0 ) {
			$this->send_error(
				new WP_Error( 'invalid_post_id', __( 'Invalid post ID.', 'topranker-ai' ) ),
				400
			);
		}

		// Verify user can edit this specific post.
		if ( ! $this->can_edit_post( $post_id ) ) {
			$this->send_error(
				new WP_Error( 'forbidden', __( 'You do not have permission to view this post.', 'topranker-ai' ) ),
				403
			);
		}

		$post = get_post( $post_id );

		if ( ! $post ) {
			$this->send_error(
				new WP_Error( 'not_found', __( 'Post not found.', 'topranker-ai' ) ),
				404
			);
		}

		$optimizer = new TopRanker_Optimizer();
		$data      = $optimizer->get_suggestions_for_editor( $post_id );

		// Add usage stats.
		$usage         = new TopRanker_Usage();
		$data['usage'] = $usage->get_api_response();

		$this->send_success( $data );
	}

	/**
	 * Handle get usage AJAX request.
	 *
	 * Returns current usage statistics.
	 *
	 * @since 1.0.0
	 */
	public function handle_get_usage() {
		// Verify request.
		$verified = $this->verify_request( 'edit_posts' );
		if ( is_wp_error( $verified ) ) {
			$this->send_error( $verified, 403 );
		}

		$usage = new TopRanker_Usage();

		$this->send_success( $usage->get_api_response() );
	}

	/**
	 * Handle bulk optimize AJAX request (Pro only).
	 *
	 * Processes a single post in a bulk optimization queue.
	 * Called sequentially by the browser to avoid PHP timeouts.
	 *
	 * @since 1.0.0
	 */
	public function handle_bulk_optimize() {
		// Verify request.
		$verified = $this->verify_request( 'edit_posts' );
		if ( is_wp_error( $verified ) ) {
			$this->send_error( $verified, 403 );
		}

		// Check Pro status.
		if ( ! topranker_is_pro() ) {
			$this->send_error(
				new WP_Error( 'pro_required', __( 'Bulk optimization requires TopRanker Pro.', 'topranker-ai' ) ),
				403
			);
		}

		// Check API key is configured.
		$api_key = get_option( 'topranker_api_key', '' );
		if ( empty( $api_key ) ) {
			$this->send_error(
				new WP_Error( 'no_api_key', __( 'No API key configured. Please add your OpenAI API key in Settings.', 'topranker-ai' ) ),
				400
			);
		}

		// Get and sanitize post ID.
		$post_id = isset( $_POST['post_id'] ) ? absint( $_POST['post_id'] ) : 0;

		if ( $post_id <= 0 ) {
			$this->send_error(
				new WP_Error( 'invalid_post_id', __( 'Invalid post ID.', 'topranker-ai' ) ),
				400
			);
		}

		// Verify user can edit this specific post.
		if ( ! $this->can_edit_post( $post_id ) ) {
			$this->send_error(
				new WP_Error( 'forbidden', __( 'You do not have permission to edit this post.', 'topranker-ai' ) ),
				403
			);
		}

		$post = get_post( $post_id );

		if ( ! $post ) {
			$this->send_error(
				new WP_Error( 'not_found', __( 'Post not found.', 'topranker-ai' ) ),
				404
			);
		}

		// Check if post type is enabled.
		if ( ! $this->is_post_type_enabled( $post->post_type ) ) {
			$this->send_error(
				new WP_Error( 'post_type_disabled', __( 'TopRanker AI is not enabled for this post type.', 'topranker-ai' ) ),
				400
			);
		}

		// Run the optimization (always force refresh for bulk).
		$optimizer = new TopRanker_Optimizer();
		$result    = $optimizer->optimize( $post, true );

		if ( is_wp_error( $result ) ) {
			// For bulk, return error but don't stop the entire batch.
			$this->send_success(
				array(
					'post_id' => $post_id,
					'success' => false,
					'error'   => $result->get_error_message(),
				)
			);
			return;
		}

		// Auto-apply the first suggestion for each field.
		$selected = array();

		if ( ! empty( $result['results']['meta_title']['suggestions'][0] ) ) {
			$first = $result['results']['meta_title']['suggestions'][0];
			$selected['meta_title'] = is_array( $first ) ? ( $first['title'] ?? '' ) : $first;
		}

		if ( ! empty( $result['results']['meta_description']['suggestions'][0] ) ) {
			$first = $result['results']['meta_description']['suggestions'][0];
			$selected['meta_description'] = is_array( $first ) ? ( $first['description'] ?? '' ) : $first;
		}

		if ( ! empty( $result['results']['excerpt']['excerpt'] ) ) {
			$selected['excerpt'] = $result['results']['excerpt']['excerpt'];
		}

		if ( ! empty( $result['results']['keyphrases']['primary'] ) ) {
			$selected['focus_keyphrase'] = $result['results']['keyphrases']['primary'];
		}

		if ( ! empty( $result['results']['keyphrases']['secondary'] ) ) {
			$selected['secondary_keyphrases'] = $result['results']['keyphrases']['secondary'];
		}

		if ( ! empty( $result['results']['social_meta']['og_title'] ) ) {
			$selected['og_title'] = $result['results']['social_meta']['og_title'];
		}

		if ( ! empty( $result['results']['social_meta']['og_description'] ) ) {
			$selected['og_description'] = $result['results']['social_meta']['og_description'];
		}

		if ( ! empty( $result['results']['social_meta']['twitter_title'] ) ) {
			$selected['twitter_title'] = $result['results']['social_meta']['twitter_title'];
		}

		if ( ! empty( $result['results']['social_meta']['twitter_description'] ) ) {
			$selected['twitter_description'] = $result['results']['social_meta']['twitter_description'];
		}

		// Alt tags (Pro).
		if ( ! empty( $result['results']['alt_tags']['results'] ) ) {
			$selected['alt_tags'] = $result['results']['alt_tags']['results'];
		}

		// Schema markup (Pro).
		if ( ! empty( $result['results']['schema']['schema'] ) ) {
			$selected['schema'] = $result['results']['schema']['schema'];
		}

		// Internal links (Pro).
		if ( ! empty( $result['results']['internal_links']['suggestions'] ) ) {
			$selected['internal_links'] = $result['results']['internal_links']['suggestions'];
		}

		// Apply selected values.
		if ( ! empty( $selected ) ) {
			$optimizer->apply( $post_id, $selected );
		}

		$this->log_debug( 'Bulk optimization completed for post ' . $post_id );

		$this->send_success(
			array(
				'post_id' => $post_id,
				'success' => true,
				'results' => $result['results'],
				'applied' => array_keys( $selected ),
			)
		);
	}

	/**
	 * Handle copy to SEO plugin AJAX request.
	 *
	 * Copies TopRanker meta data to the detected SEO plugin's fields.
	 *
	 * @since 1.0.0
	 */
	public function handle_copy_to_seo_plugin() {
		// Verify request.
		$verified = $this->verify_request( 'edit_posts' );
		if ( is_wp_error( $verified ) ) {
			$this->send_error( $verified, 403 );
		}

		// Get and sanitize post ID.
		$post_id = isset( $_POST['post_id'] ) ? absint( $_POST['post_id'] ) : 0;

		if ( $post_id <= 0 ) {
			$this->send_error(
				new WP_Error( 'invalid_post_id', __( 'Invalid post ID.', 'topranker-ai' ) ),
				400
			);
		}

		// Verify user can edit this specific post.
		if ( ! $this->can_edit_post( $post_id ) ) {
			$this->send_error(
				new WP_Error( 'forbidden', __( 'You do not have permission to edit this post.', 'topranker-ai' ) ),
				403
			);
		}

		$post = get_post( $post_id );

		if ( ! $post ) {
			$this->send_error(
				new WP_Error( 'not_found', __( 'Post not found.', 'topranker-ai' ) ),
				404
			);
		}

		// Copy to SEO plugin.
		$optimizer = new TopRanker_Optimizer();
		$result    = $optimizer->copy_to_seo_plugin( $post_id );

		if ( is_wp_error( $result ) ) {
			$this->send_error( $result, 400 );
		}

		$this->log_debug( 'Copied TopRanker data to SEO plugin for post ' . $post_id );

		$this->send_success( $result );
	}

	/**
	 * Get the nonce for AJAX requests.
	 *
	 * @since  1.0.0
	 * @return string The nonce value.
	 */
	public static function get_nonce() {
		return wp_create_nonce( self::NONCE_ACTION );
	}

	/**
	 * Handle generate alt tags AJAX request (Pro only).
	 *
	 * Generates SEO-optimized alt text for all images in a post.
	 *
	 * @since 1.0.0
	 */
	public function handle_generate_alt_tags() {
		// Verify request.
		$verified = $this->verify_request( 'edit_posts' );
		if ( is_wp_error( $verified ) ) {
			$this->send_error( $verified, 403 );
		}

		// Check Pro status.
		if ( ! topranker_is_pro() ) {
			$this->send_error(
				new WP_Error( 'pro_required', __( 'Alt tag generation is a Pro feature. Please upgrade to unlock.', 'topranker-ai' ) ),
				403
			);
		}

		// Check if Pro class exists.
		if ( ! class_exists( 'TopRanker_Alt_Tags' ) ) {
			$this->send_error(
				new WP_Error( 'feature_unavailable', __( 'Alt tag generation is not available.', 'topranker-ai' ) ),
				500
			);
		}

		// Get and sanitize post ID.
		$post_id = isset( $_POST['post_id'] ) ? absint( $_POST['post_id'] ) : 0;

		if ( $post_id <= 0 ) {
			$this->send_error(
				new WP_Error( 'invalid_post_id', __( 'Invalid post ID.', 'topranker-ai' ) ),
				400
			);
		}

		// Verify user can edit this specific post.
		if ( ! $this->can_edit_post( $post_id ) ) {
			$this->send_error(
				new WP_Error( 'forbidden', __( 'You do not have permission to edit this post.', 'topranker-ai' ) ),
				403
			);
		}

		$post = get_post( $post_id );

		if ( ! $post ) {
			$this->send_error(
				new WP_Error( 'not_found', __( 'Post not found.', 'topranker-ai' ) ),
				404
			);
		}

		// Get focus keyphrase if available.
		$focus_keyphrase = get_post_meta( $post_id, '_topranker_focus_keyphrase', true );

		// Generate alt tags.
		$alt_tags = new TopRanker_Alt_Tags();
		$result   = $alt_tags->generate_alt_tags( $post, $focus_keyphrase );

		if ( is_wp_error( $result ) ) {
			$this->send_error( $result, 400 );
		}

		$this->log_debug( 'Generated alt tags for post ' . $post_id . ': ' . $result['success'] . ' success, ' . $result['failed'] . ' failed' );

		$this->send_success( $result );
	}

	/**
	 * Handle apply alt tags AJAX request (Pro only).
	 *
	 * Saves generated alt text to attachment meta.
	 *
	 * @since 1.0.0
	 */
	public function handle_apply_alt_tags() {
		// Verify request.
		$verified = $this->verify_request( 'edit_posts' );
		if ( is_wp_error( $verified ) ) {
			$this->send_error( $verified, 403 );
		}

		// Check Pro status.
		if ( ! topranker_is_pro() ) {
			$this->send_error(
				new WP_Error( 'pro_required', __( 'Alt tag generation is a Pro feature. Please upgrade to unlock.', 'topranker-ai' ) ),
				403
			);
		}

		// Check if Pro class exists.
		if ( ! class_exists( 'TopRanker_Alt_Tags' ) ) {
			$this->send_error(
				new WP_Error( 'feature_unavailable', __( 'Alt tag generation is not available.', 'topranker-ai' ) ),
				500
			);
		}

		// Get results from request.
		$results = isset( $_POST['results'] ) ? $_POST['results'] : array();

		if ( is_string( $results ) ) {
			$results = json_decode( wp_unslash( $results ), true );
		}

		if ( ! is_array( $results ) || empty( $results ) ) {
			$this->send_error(
				new WP_Error( 'invalid_results', __( 'No valid alt tag results provided.', 'topranker-ai' ) ),
				400
			);
		}

		// Verify user can edit the attachments.
		foreach ( $results as $result ) {
			$image_id = isset( $result['image_id'] ) ? (int) $result['image_id'] : 0;
			if ( $image_id > 0 && ! current_user_can( 'edit_post', $image_id ) ) {
				$this->send_error(
					new WP_Error( 'forbidden', __( 'You do not have permission to edit one or more of these images.', 'topranker-ai' ) ),
					403
				);
			}
		}

		// Get post ID if provided (to update alt in post content HTML).
		$post_id = isset( $_POST['post_id'] ) ? absint( $_POST['post_id'] ) : 0;

		// Apply alt tags.
		$alt_tags     = new TopRanker_Alt_Tags();
		$apply_result = $alt_tags->apply_alt_tags( $results, $post_id );

		$this->log_debug( 'Applied alt tags: ' . $apply_result['applied'] . ' applied, ' . $apply_result['failed'] . ' failed' );

		$this->send_success( $apply_result );
	}

	/**
	 * Handle generate single attachment alt tag AJAX request (Pro only).
	 *
	 * Generates alt text for an image in the Media Library.
	 *
	 * @since 1.0.0
	 */
	public function handle_generate_attachment_alt() {
		// Verify request.
		$verified = $this->verify_request( 'edit_posts' );
		if ( is_wp_error( $verified ) ) {
			$this->send_error( $verified, 403 );
		}

		// Check Pro status.
		if ( ! topranker_is_pro() ) {
			$this->send_error(
				new WP_Error( 'pro_required', __( 'Alt tag generation is a Pro feature. Please upgrade to unlock.', 'topranker-ai' ) ),
				403
			);
		}

		// Check if Pro class exists.
		if ( ! class_exists( 'TopRanker_Alt_Tags' ) ) {
			$this->send_error(
				new WP_Error( 'feature_unavailable', __( 'Alt tag generation is not available.', 'topranker-ai' ) ),
				500
			);
		}

		// Get and sanitize attachment ID.
		$attachment_id = isset( $_POST['attachment_id'] ) ? absint( $_POST['attachment_id'] ) : 0;

		if ( $attachment_id <= 0 ) {
			$this->send_error(
				new WP_Error( 'invalid_attachment_id', __( 'Invalid attachment ID.', 'topranker-ai' ) ),
				400
			);
		}

		// Verify user can edit this attachment.
		if ( ! current_user_can( 'edit_post', $attachment_id ) ) {
			$this->send_error(
				new WP_Error( 'forbidden', __( 'You do not have permission to edit this image.', 'topranker-ai' ) ),
				403
			);
		}

		// Generate alt tag.
		$alt_tags = new TopRanker_Alt_Tags();
		$result   = $alt_tags->generate_for_attachment( $attachment_id );

		if ( is_wp_error( $result ) ) {
			$this->send_error( $result, 400 );
		}

		$this->log_debug( 'Generated alt tag for attachment ' . $attachment_id );

		$this->send_success( $result );
	}

	/**
	 * Handle get SEO audit AJAX request (Pro only).
	 *
	 * Returns SEO audit/score data for a post.
	 *
	 * @since 1.0.0
	 */
	public function handle_get_seo_audit() {
		// Verify request.
		$verified = $this->verify_request( 'edit_posts' );
		if ( is_wp_error( $verified ) ) {
			$this->send_error( $verified, 403 );
		}

		// Check Pro status.
		if ( ! topranker_is_pro() ) {
			$this->send_error(
				new WP_Error( 'pro_required', __( 'SEO audit is a Pro feature. Please upgrade to unlock.', 'topranker-ai' ) ),
				403
			);
		}

		// Check if Pro class exists.
		if ( ! class_exists( 'TopRanker_SEO_Audit' ) ) {
			$this->send_error(
				new WP_Error( 'feature_unavailable', __( 'SEO audit is not available.', 'topranker-ai' ) ),
				500
			);
		}

		// Get and sanitize post ID.
		$post_id = isset( $_POST['post_id'] ) ? absint( $_POST['post_id'] ) : 0;

		// Also support GET for convenience.
		if ( 0 === $post_id ) {
			$post_id = isset( $_GET['post_id'] ) ? absint( $_GET['post_id'] ) : 0;
		}

		if ( $post_id <= 0 ) {
			$this->send_error(
				new WP_Error( 'invalid_post_id', __( 'Invalid post ID.', 'topranker-ai' ) ),
				400
			);
		}

		// Verify user can edit this specific post.
		if ( ! $this->can_edit_post( $post_id ) ) {
			$this->send_error(
				new WP_Error( 'forbidden', __( 'You do not have permission to view this post.', 'topranker-ai' ) ),
				403
			);
		}

		$post = get_post( $post_id );

		if ( ! $post ) {
			$this->send_error(
				new WP_Error( 'not_found', __( 'Post not found.', 'topranker-ai' ) ),
				404
			);
		}

		// Get SEO audit data.
		$audit = new TopRanker_SEO_Audit();
		$data  = $audit->get_api_data( $post );

		$data['post_id'] = $post_id;

		$this->log_debug( 'Got SEO audit for post ' . $post_id . ': score ' . $data['score'] );

		$this->send_success( $data );
	}

	/**
	 * Handle get history AJAX request (Pro only).
	 *
	 * Returns optimization history for a post.
	 *
	 * @since 1.0.0
	 */
	public function handle_get_history() {
		// Verify request.
		$verified = $this->verify_request( 'edit_posts' );
		if ( is_wp_error( $verified ) ) {
			$this->send_error( $verified, 403 );
		}

		// Check Pro status.
		if ( ! topranker_is_pro() ) {
			$this->send_error(
				new WP_Error( 'pro_required', __( 'Optimization history is a Pro feature. Please upgrade to unlock.', 'topranker-ai' ) ),
				403
			);
		}

		// Check if Pro class exists.
		if ( ! class_exists( 'TopRanker_History' ) ) {
			$this->send_error(
				new WP_Error( 'feature_unavailable', __( 'Optimization history is not available.', 'topranker-ai' ) ),
				500
			);
		}

		// Get and sanitize post ID.
		$post_id = isset( $_POST['post_id'] ) ? absint( $_POST['post_id'] ) : 0;

		// Also support GET for convenience.
		if ( 0 === $post_id ) {
			$post_id = isset( $_GET['post_id'] ) ? absint( $_GET['post_id'] ) : 0;
		}

		if ( $post_id <= 0 ) {
			$this->send_error(
				new WP_Error( 'invalid_post_id', __( 'Invalid post ID.', 'topranker-ai' ) ),
				400
			);
		}

		// Verify user can edit this specific post.
		if ( ! $this->can_edit_post( $post_id ) ) {
			$this->send_error(
				new WP_Error( 'forbidden', __( 'You do not have permission to view this post.', 'topranker-ai' ) ),
				403
			);
		}

		$post = get_post( $post_id );

		if ( ! $post ) {
			$this->send_error(
				new WP_Error( 'not_found', __( 'Post not found.', 'topranker-ai' ) ),
				404
			);
		}

		// Get history data.
		$history = new TopRanker_History();
		$data    = $history->get_api_response( $post_id );

		$this->log_debug( 'Got history for post ' . $post_id . ': ' . $data['count'] . ' entries' );

		$this->send_success( $data );
	}

	/**
	 * Handle revert history AJAX request (Pro only).
	 *
	 * Reverts to a specific history entry.
	 *
	 * @since 1.0.0
	 */
	public function handle_revert_history() {
		// Verify request.
		$verified = $this->verify_request( 'edit_posts' );
		if ( is_wp_error( $verified ) ) {
			$this->send_error( $verified, 403 );
		}

		// Check Pro status.
		if ( ! topranker_is_pro() ) {
			$this->send_error(
				new WP_Error( 'pro_required', __( 'Optimization history is a Pro feature. Please upgrade to unlock.', 'topranker-ai' ) ),
				403
			);
		}

		// Check if Pro class exists.
		if ( ! class_exists( 'TopRanker_History' ) ) {
			$this->send_error(
				new WP_Error( 'feature_unavailable', __( 'Optimization history is not available.', 'topranker-ai' ) ),
				500
			);
		}

		// Get and sanitize post ID and index.
		$post_id = isset( $_POST['post_id'] ) ? absint( $_POST['post_id'] ) : 0;
		$index   = isset( $_POST['index'] ) ? absint( $_POST['index'] ) : 0;

		if ( $post_id <= 0 ) {
			$this->send_error(
				new WP_Error( 'invalid_post_id', __( 'Invalid post ID.', 'topranker-ai' ) ),
				400
			);
		}

		// Verify user can edit this specific post.
		if ( ! $this->can_edit_post( $post_id ) ) {
			$this->send_error(
				new WP_Error( 'forbidden', __( 'You do not have permission to edit this post.', 'topranker-ai' ) ),
				403
			);
		}

		$post = get_post( $post_id );

		if ( ! $post ) {
			$this->send_error(
				new WP_Error( 'not_found', __( 'Post not found.', 'topranker-ai' ) ),
				404
			);
		}

		// Perform revert.
		$history = new TopRanker_History();
		$result  = $history->revert_to_entry( $post_id, $index );

		if ( is_wp_error( $result ) ) {
			$this->send_error( $result, 400 );
		}

		$this->log_debug( 'Reverted post ' . $post_id . ' to history entry ' . $index );

		$this->send_success( $result );
	}

	/**
	 * Get localized data for JavaScript.
	 *
	 * Returns an array of data to be passed to JavaScript via wp_localize_script.
	 *
	 * @since  1.0.0
	 * @return array Localized data array.
	 */
	public static function get_localized_data() {
		$usage      = new TopRanker_Usage();
		$seo_compat = new TopRanker_SEO_Compat();

		return array(
			'ajax_url'        => admin_url( 'admin-ajax.php' ),
			'nonce'           => self::get_nonce(),
			'is_pro'          => topranker_is_pro(),
			'usage'           => $usage->get_api_response(),
			'upgrade_url'     => function_exists( 'topranker_fs' ) ? topranker_fs()->get_upgrade_url() : '',
			'seo_compat'      => array(
				'has_plugin'   => $seo_compat->has_seo_plugin(),
				'plugin_name'  => $seo_compat->get_detected_plugin_name(),
				'plugin_slug'  => $seo_compat->get_detected_plugin(),
				'mode'         => $seo_compat->get_seo_mode(),
				'is_sync'      => $seo_compat->is_sync_mode(),
				'is_suggest'   => $seo_compat->is_suggest_mode(),
				'is_standalone' => $seo_compat->is_standalone_mode(),
			),
			'strings'         => array(
				'optimizing'            => __( 'Generating suggestions...', 'topranker-ai' ),
				'applying'              => __( 'Applying changes...', 'topranker-ai' ),
				'copying'               => __( 'Copying to SEO plugin...', 'topranker-ai' ),
				'success'               => __( 'Optimization complete!', 'topranker-ai' ),
				'applied'               => __( 'Selected optimizations have been applied.', 'topranker-ai' ),
				'copied'                => __( 'Data copied successfully!', 'topranker-ai' ),
				'error'                 => __( 'An error occurred. Please try again.', 'topranker-ai' ),
				'retry'                 => __( 'Retry', 'topranker-ai' ),
				'no_api_key'            => __( 'Please configure your OpenAI API key in TopRanker Settings.', 'topranker-ai' ),
				'limit_reached'         => __( 'You have reached your monthly optimization limit.', 'topranker-ai' ),
				'upgrade_to_pro'        => __( 'Upgrade to Pro', 'topranker-ai' ),
				'confirm_optimize'      => __( 'Optimize this post?', 'topranker-ai' ),
				'select_at_least'       => __( 'Please select at least one suggestion to apply.', 'topranker-ai' ),
				'optimize_btn'          => __( 'Optimize This Post', 'topranker-ai' ),
				'apply_btn'             => __( 'Apply Selected', 'topranker-ai' ),
				/* translators: %s: SEO plugin name */
				'copy_to'               => __( 'Copy to %s', 'topranker-ai' ),
				'generating_alt_tags'   => __( 'Generating alt tags...', 'topranker-ai' ),
				'applying_alt_tags'     => __( 'Applying alt tags...', 'topranker-ai' ),
				'alt_tags_generated'    => __( 'Alt tags generated successfully!', 'topranker-ai' ),
				'alt_tags_applied'      => __( 'Alt tags have been applied to images.', 'topranker-ai' ),
				'no_images'             => __( 'No images found in this post.', 'topranker-ai' ),
				'generate_alt_tags_btn' => __( 'Generate Alt Tags', 'topranker-ai' ),
				'apply_alt_tags_btn'    => __( 'Apply Alt Tags', 'topranker-ai' ),
				'history'                 => __( 'History', 'topranker-ai' ),
				'view_history'            => __( 'View History', 'topranker-ai' ),
				'optimization_history'    => __( 'Optimization History', 'topranker-ai' ),
				'no_history'              => __( 'No optimization history yet.', 'topranker-ai' ),
				'revert'                  => __( 'Revert', 'topranker-ai' ),
				'reverting'               => __( 'Reverting...', 'topranker-ai' ),
				'reverted'                => __( 'Reverted successfully!', 'topranker-ai' ),
				'confirm_revert'          => __( 'Are you sure you want to revert to this version? Current values will be saved to history.', 'topranker-ai' ),
				'show_diff'               => __( 'Show Changes', 'topranker-ai' ),
				'hide_diff'               => __( 'Hide Changes', 'topranker-ai' ),
				'history_loading'         => __( 'Loading...', 'topranker-ai' ),
				'history_error'           => __( 'Error', 'topranker-ai' ),
				'history_load_error'      => __( 'Failed to load history entry.', 'topranker-ai' ),
				'history_preview_title'   => __( 'Optimization from', 'topranker-ai' ),
				'history_revert_confirm'  => __( 'Are you sure you want to revert to this optimization? Current values will be replaced.', 'topranker-ai' ),
				'history_reverted'        => __( 'Successfully reverted to previous optimization.', 'topranker-ai' ),
				'history_revert_error'    => __( 'Failed to revert optimization.', 'topranker-ai' ),
				'meta_title'              => __( 'Meta Title', 'topranker-ai' ),
				'meta_description'        => __( 'Meta Description', 'topranker-ai' ),
				'excerpt'                 => __( 'Excerpt', 'topranker-ai' ),
				'focus_keyphrase'         => __( 'Focus Keyphrase', 'topranker-ai' ),
				'og_title'                => __( 'OG Title', 'topranker-ai' ),
				'og_description'          => __( 'OG Description', 'topranker-ai' ),
				/* translators: %d: number of characters */
				'chars'                   => __( '%d chars', 'topranker-ai' ),
				'primary_keyphrase'       => __( 'Primary', 'topranker-ai' ),
				'secondary_keyphrases'    => __( 'Secondary', 'topranker-ai' ),
				'last_optimized_just_now' => __( 'Last optimized: just now', 'topranker-ai' ),
			),
		);
	}
}
