<?php
/**
 * OpenAI API wrapper.
 *
 * Handles all communication with the OpenAI API including chat completions
 * and vision capabilities. Provides comprehensive error handling for all
 * failure modes.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * TopRanker API class.
 *
 * @since 1.0.0
 */
class TopRanker_API {

	/**
	 * OpenAI API endpoint.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	const API_ENDPOINT = 'https://api.openai.com/v1/chat/completions';

	/**
	 * Request timeout in seconds.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const TIMEOUT = 30;

	/**
	 * Maximum number of retry attempts for failed requests.
	 *
	 * @since 1.1.0
	 * @var   int
	 */
	const MAX_RETRIES = 2;

	/**
	 * Delay between retries in seconds.
	 *
	 * @since 1.1.0
	 * @var   int
	 */
	const RETRY_DELAY = 2;

	/**
	 * Last error message.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	private $last_error = '';

	/**
	 * Last error code.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	private $last_error_code = '';

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		// Nothing to initialize.
	}

	/**
	 * Send a chat completion request to OpenAI.
	 *
	 * @since 1.0.0
	 * @param array  $messages Array of message objects with 'role' and 'content'.
	 * @param string $model    Optional. Model to use. Defaults to saved setting.
	 * @return array|WP_Error Response array with 'content' key or WP_Error on failure.
	 */
	public function chat_completion( $messages, $model = null ) {
		$api_key = $this->get_api_key();

		if ( empty( $api_key ) ) {
			$this->last_error      = __( 'No API key configured. Please add your OpenAI API key in Settings.', 'topranker-ai' );
			$this->last_error_code = 'no_api_key';
			return new WP_Error( 'no_api_key', $this->last_error );
		}

		if ( null === $model ) {
			$model = $this->get_model();
		}

		$body = $this->build_request_body( $model, $messages );

		$this->log_request( $body );

		return $this->send_request( $api_key, $body );
	}

	/**
	 * Send a chat completion request with vision (image analysis).
	 *
	 * @since 1.0.0
	 * @param string $text_prompt The text prompt describing what to analyze.
	 * @param string $image_url   The URL of the image to analyze.
	 * @param string $model       Optional. Model to use. Defaults to saved setting.
	 * @return array|WP_Error Response array with 'content' key or WP_Error on failure.
	 */
	public function chat_completion_with_vision( $text_prompt, $image_url, $model = null ) {
		$api_key = $this->get_api_key();

		if ( empty( $api_key ) ) {
			$this->last_error      = __( 'No API key configured. Please add your OpenAI API key in Settings.', 'topranker-ai' );
			$this->last_error_code = 'no_api_key';
			return new WP_Error( 'no_api_key', $this->last_error );
		}

		if ( null === $model ) {
			$model = $this->get_model();
		}

		// Build content array with both text and image.
		$content = array(
			array(
				'type' => 'text',
				'text' => $text_prompt,
			),
			array(
				'type'      => 'image_url',
				'image_url' => array(
					'url' => $image_url,
				),
			),
		);

		$messages = array(
			array(
				'role'    => 'user',
				'content' => $content,
			),
		);

		$body = $this->build_request_body( $model, $messages );

		$this->log_request( $body );

		return $this->send_request( $api_key, $body );
	}

	/**
	 * Send a request to the OpenAI API with automatic retry on transient failures.
	 *
	 * Retries on network errors and 429/5xx status codes. Does not retry on
	 * authentication errors (401/403) or other client errors (4xx).
	 *
	 * @since 1.1.0
	 * @param string $api_key The API key.
	 * @param array  $body    The request body.
	 * @return array|WP_Error Parsed response or WP_Error on failure.
	 */
	private function send_request( $api_key, $body ) {
		$last_result = null;

		for ( $attempt = 0; $attempt <= self::MAX_RETRIES; $attempt++ ) {
			if ( $attempt > 0 ) {
				sleep( self::RETRY_DELAY * $attempt );
			}

			$response = wp_remote_post(
				self::API_ENDPOINT,
				array(
					'timeout' => self::TIMEOUT,
					'headers' => array(
						'Authorization' => 'Bearer ' . $api_key,
						'Content-Type'  => 'application/json',
					),
					'body'    => wp_json_encode( $body ),
				)
			);

			// Network-level error — always retry.
			if ( is_wp_error( $response ) ) {
				$last_result = $this->handle_response( $response );
				continue;
			}

			$status_code = wp_remote_retrieve_response_code( $response );

			// Rate limited or server error — retry.
			if ( 429 === $status_code || $status_code >= 500 ) {
				$last_result = $this->handle_response( $response );
				continue;
			}

			// Success or non-retryable error (401, 403, 400, etc.) — return immediately.
			return $this->handle_response( $response );
		}

		return $last_result;
	}

	/**
	 * Test the API connection with a minimal request.
	 *
	 * @since 1.0.0
	 * @param string $api_key Optional. API key to test. Defaults to saved key.
	 * @return array|WP_Error Success array with 'success' => true or WP_Error.
	 */
	public function test_connection( $api_key = null ) {
		if ( null === $api_key ) {
			$api_key = $this->get_api_key();
		}

		if ( empty( $api_key ) ) {
			$this->last_error      = __( 'No API key provided.', 'topranker-ai' );
			$this->last_error_code = 'no_api_key';
			return new WP_Error( 'no_api_key', $this->last_error );
		}

		$messages = array(
			array(
				'role'    => 'user',
				'content' => 'Respond with JSON: {"status": "ok"}',
			),
		);

		$body = $this->build_request_body( $this->get_model(), $messages, 500 );

		$this->log_request( $body, 'test_connection' );

		$response = wp_remote_post(
			self::API_ENDPOINT,
			array(
				'timeout' => 15,
				'headers' => array(
					'Authorization' => 'Bearer ' . $api_key,
					'Content-Type'  => 'application/json',
				),
				'body'    => wp_json_encode( $body ),
			)
		);

		$result = $this->handle_response( $response );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		return array( 'success' => true );
	}

	/**
	 * Handle the API response.
	 *
	 * @since 1.0.0
	 * @param array|WP_Error $response The wp_remote_post response.
	 * @return array|WP_Error Parsed response or WP_Error on failure.
	 */
	private function handle_response( $response ) {
		// Network error.
		if ( is_wp_error( $response ) ) {
			$error_message = $response->get_error_message();

			// Check for timeout.
			if ( strpos( $error_message, 'timed out' ) !== false || strpos( $error_message, 'timeout' ) !== false ) {
				$this->last_error      = __( 'The request timed out. Try again or switch to a faster model.', 'topranker-ai' );
				$this->last_error_code = 'timeout';
				return new WP_Error( 'timeout', $this->last_error );
			}

			$this->last_error      = __( 'Could not connect to OpenAI. Check your server\'s internet connection.', 'topranker-ai' );
			$this->last_error_code = 'network_error';
			return new WP_Error( 'network_error', $this->last_error );
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		$body        = wp_remote_retrieve_body( $response );
		$data        = json_decode( $body, true );

		$this->log_response( $status_code, $data );

		// Handle HTTP errors.
		if ( $status_code >= 400 ) {
			return $this->handle_error_response( $status_code, $data );
		}

		// Check for valid response structure.
		if ( ! is_array( $data ) || ! isset( $data['choices'][0]['message']['content'] ) ) {
			$this->last_error      = __( 'Unexpected response from OpenAI. Please try again.', 'topranker-ai' );
			$this->last_error_code = 'malformed_response';
			return new WP_Error( 'malformed_response', $this->last_error );
		}

		return array(
			'content' => $data['choices'][0]['message']['content'],
			'usage'   => isset( $data['usage'] ) ? $data['usage'] : null,
		);
	}

	/**
	 * Handle error responses from the API.
	 *
	 * @since 1.0.0
	 * @param int   $status_code HTTP status code.
	 * @param array $data        Decoded response body.
	 * @return WP_Error Error object with appropriate message.
	 */
	private function handle_error_response( $status_code, $data ) {
		$error_message = isset( $data['error']['message'] ) ? $data['error']['message'] : '';
		$error_type    = isset( $data['error']['type'] ) ? $data['error']['type'] : '';
		$error_code    = isset( $data['error']['code'] ) ? $data['error']['code'] : '';

		// 401 Unauthorized - Invalid API key.
		if ( 401 === $status_code ) {
			$this->last_error      = __( 'Your API key is invalid. Please check it in Settings.', 'topranker-ai' );
			$this->last_error_code = 'invalid_api_key';
			return new WP_Error( 'invalid_api_key', $this->last_error );
		}

		// 429 Rate limit.
		if ( 429 === $status_code ) {
			// Check if it's a quota exceeded error.
			if ( 'insufficient_quota' === $error_code || strpos( $error_message, 'quota' ) !== false ) {
				$this->last_error      = __( 'Your OpenAI account may have no credits. Check your OpenAI billing.', 'topranker-ai' );
				$this->last_error_code = 'no_credits';
				return new WP_Error( 'no_credits', $this->last_error );
			}

			$this->last_error      = __( 'OpenAI rate limit reached. Please wait a minute and try again.', 'topranker-ai' );
			$this->last_error_code = 'rate_limit';
			return new WP_Error( 'rate_limit', $this->last_error );
		}

		// 400 Bad request.
		if ( 400 === $status_code ) {
			$this->last_error      = __( 'Invalid request to OpenAI. Please try again.', 'topranker-ai' );
			$this->last_error_code = 'bad_request';
			return new WP_Error( 'bad_request', $this->last_error );
		}

		// 500+ Server errors.
		if ( $status_code >= 500 ) {
			$this->last_error      = __( 'OpenAI server error. Please try again later.', 'topranker-ai' );
			$this->last_error_code = 'server_error';
			return new WP_Error( 'server_error', $this->last_error );
		}

		// Generic error.
		$this->last_error      = __( 'An error occurred while connecting to OpenAI. Please try again.', 'topranker-ai' );
		$this->last_error_code = 'api_error';
		return new WP_Error( 'api_error', $this->last_error );
	}

	/**
	 * Parse JSON from the API response content.
	 *
	 * Extracts JSON from the response, handling cases where the model
	 * might include markdown code blocks or extra text.
	 *
	 * @since 1.0.0
	 * @param string $content The raw content string from the API.
	 * @return array|WP_Error Parsed JSON array or WP_Error on failure.
	 */
	public function parse_json_response( $content ) {
		// Try direct parse first.
		$data = json_decode( $content, true );

		if ( JSON_ERROR_NONE === json_last_error() && is_array( $data ) ) {
			return $data;
		}

		// Try to extract JSON from markdown code blocks.
		if ( preg_match( '/```(?:json)?\s*\n?([\s\S]*?)\n?```/', $content, $matches ) ) {
			$data = json_decode( trim( $matches[1] ), true );

			if ( JSON_ERROR_NONE === json_last_error() && is_array( $data ) ) {
				return $data;
			}
		}

		// Try to find JSON object or array in the content.
		if ( preg_match( '/(\{[\s\S]*\}|\[[\s\S]*\])/', $content, $matches ) ) {
			$data = json_decode( $matches[1], true );

			if ( JSON_ERROR_NONE === json_last_error() && is_array( $data ) ) {
				return $data;
			}
		}

		return new WP_Error(
			'json_parse_error',
			__( 'Failed to parse JSON from API response.', 'topranker-ai' )
		);
	}

	/**
	 * Get the configured API key.
	 *
	 * @since  1.0.0
	 * @return string API key or empty string if not configured.
	 */
	public function get_api_key() {
		return get_option( 'topranker_api_key', '' );
	}

	/**
	 * Build the request body with correct parameters for the selected model.
	 *
	 * GPT-5.x models are reasoning models and need max_completion_tokens + reasoning_effort.
	 * GPT-4.1 models are non-reasoning and use max_tokens.
	 *
	 * @since  1.1.0
	 * @param  string $model    Model identifier.
	 * @param  array  $messages Array of message objects.
	 * @param  int    $tokens   Optional. Max tokens for the response. Default 2000.
	 * @return array Request body array.
	 */
	private function build_request_body( $model, $messages, $tokens = 2000 ) {
		$body = array(
			'model'           => $model,
			'messages'        => $messages,
			'response_format' => array( 'type' => 'json_object' ),
		);

		if ( 0 === strpos( $model, 'gpt-5' ) ) {
			$body['max_completion_tokens'] = $tokens;
			$body['reasoning_effort']      = 'none';
		} else {
			$body['max_tokens'] = $tokens;
		}

		return $body;
	}

	/**
	 * Get the configured AI model.
	 *
	 * @since  1.0.0
	 * @return string Model identifier.
	 */
	public function get_model() {
		return get_option( 'topranker_model', 'gpt-5.2' );
	}

	/**
	 * Get the last error message.
	 *
	 * @since  1.0.0
	 * @return string Last error message.
	 */
	public function get_last_error() {
		return $this->last_error;
	}

	/**
	 * Get the last error code.
	 *
	 * @since  1.0.0
	 * @return string Last error code.
	 */
	public function get_last_error_code() {
		return $this->last_error_code;
	}

	/**
	 * Check if the API is configured and ready.
	 *
	 * @since  1.0.0
	 * @return bool True if API key is configured.
	 */
	public function is_configured() {
		return ! empty( $this->get_api_key() );
	}

	/**
	 * Log API request for debugging.
	 *
	 * Only logs when TOPRANKER_DEBUG is defined and true.
	 *
	 * @since 1.0.0
	 * @param array  $body    Request body.
	 * @param string $context Optional. Context identifier for the request.
	 */
	private function log_request( $body, $context = 'chat_completion' ) {
		if ( ! $this->is_debug_enabled() ) {
			return;
		}

		// Sanitize the body for logging (don't log full content).
		$log_body = $body;
		if ( isset( $log_body['messages'] ) ) {
			foreach ( $log_body['messages'] as $key => $message ) {
				if ( isset( $message['content'] ) && is_string( $message['content'] ) && strlen( $message['content'] ) > 200 ) {
					$log_body['messages'][ $key ]['content'] = substr( $message['content'], 0, 200 ) . '... [truncated]';
				}
			}
		}

		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
		error_log(
			sprintf(
				'[TopRanker AI] API Request (%s): %s',
				$context,
				wp_json_encode( $log_body )
			)
		);
	}

	/**
	 * Log API response for debugging.
	 *
	 * Only logs when TOPRANKER_DEBUG is defined and true.
	 *
	 * @since 1.0.0
	 * @param int   $status_code HTTP status code.
	 * @param array $data        Response data.
	 */
	private function log_response( $status_code, $data ) {
		if ( ! $this->is_debug_enabled() ) {
			return;
		}

		// Sanitize response for logging.
		$log_data = $data;
		if ( isset( $log_data['choices'][0]['message']['content'] ) ) {
			$content = $log_data['choices'][0]['message']['content'];
			if ( strlen( $content ) > 200 ) {
				$log_data['choices'][0]['message']['content'] = substr( $content, 0, 200 ) . '... [truncated]';
			}
		}

		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
		error_log(
			sprintf(
				'[TopRanker AI] API Response (HTTP %d): %s',
				$status_code,
				wp_json_encode( $log_data )
			)
		);
	}

	/**
	 * Check if debug mode is enabled.
	 *
	 * @since  1.0.0
	 * @return bool True if debug mode is enabled.
	 */
	private function is_debug_enabled() {
		return defined( 'TOPRANKER_DEBUG' ) && TOPRANKER_DEBUG;
	}

	/**
	 * Build the system prompt prefix with context.
	 *
	 * @since 1.0.0
	 * @param string $locale   The locale/language code.
	 * @param string $tone     The tone of voice setting.
	 * @return string System prompt prefix.
	 */
	public function build_context_prefix( $locale = null, $tone = null ) {
		if ( null === $locale ) {
			$locale = get_locale();
		}

		if ( null === $tone ) {
			$tone = get_option( 'topranker_tone', 'professional' );
		}

		$site_name        = get_bloginfo( 'name' );
		$site_description = get_bloginfo( 'description' );
		$language_name    = $this->get_language_name( $locale );

		$tone_labels = array(
			'professional' => __( 'professional', 'topranker-ai' ),
			'casual'       => __( 'casual', 'topranker-ai' ),
			'technical'    => __( 'technical', 'topranker-ai' ),
			'friendly'     => __( 'friendly', 'topranker-ai' ),
		);

		$tone_label = isset( $tone_labels[ $tone ] ) ? $tone_labels[ $tone ] : $tone_labels['professional'];

		$context = sprintf(
			/* translators: 1: Site name, 2: Site description, 3: Tone, 4: Language */
			__(
				'You are an SEO optimization assistant for the website "%1$s" (%2$s). ' .
				'Use a %3$s tone. Generate all output in %4$s.',
				'topranker-ai'
			),
			$site_name,
			$site_description,
			$tone_label,
			$language_name
		);

		return $context;
	}

	/**
	 * Get the human-readable language name from a locale code.
	 *
	 * @since 1.0.0
	 * @param string $locale Locale code (e.g., 'en_US', 'sv_SE').
	 * @return string Language name.
	 */
	private function get_language_name( $locale ) {
		// Map common locales to language names.
		$languages = array(
			'en_US' => 'English',
			'en_GB' => 'English',
			'en_AU' => 'English',
			'sv_SE' => 'Swedish',
			'de_DE' => 'German',
			'de_AT' => 'German',
			'de_CH' => 'German',
			'fr_FR' => 'French',
			'fr_CA' => 'French',
			'es_ES' => 'Spanish',
			'es_MX' => 'Spanish',
			'it_IT' => 'Italian',
			'pt_BR' => 'Portuguese',
			'pt_PT' => 'Portuguese',
			'nl_NL' => 'Dutch',
			'nl_BE' => 'Dutch',
			'pl_PL' => 'Polish',
			'ru_RU' => 'Russian',
			'ja'    => 'Japanese',
			'ko_KR' => 'Korean',
			'zh_CN' => 'Chinese',
			'zh_TW' => 'Chinese',
			'ar'    => 'Arabic',
			'he_IL' => 'Hebrew',
			'fi'    => 'Finnish',
			'da_DK' => 'Danish',
			'nb_NO' => 'Norwegian',
			'nn_NO' => 'Norwegian',
		);

		// Check for exact match.
		if ( isset( $languages[ $locale ] ) ) {
			return $languages[ $locale ];
		}

		// Check for language code only (first 2 chars).
		$lang_code = substr( $locale, 0, 2 );
		foreach ( $languages as $loc => $name ) {
			if ( 0 === strpos( $loc, $lang_code ) ) {
				return $name;
			}
		}

		// Fallback to English.
		return 'English';
	}
}
