<?php
/**
 * Core plugin class.
 *
 * The main class that orchestrates the entire plugin. Uses singleton pattern
 * to ensure only one instance exists. Loads all other classes and defines hooks.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Main TopRanker_Core class.
 *
 * @since 1.0.0
 */
class TopRanker_Core {

	/**
	 * Single instance of the class.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_Core|null
	 */
	private static $instance = null;

	/**
	 * Admin class instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_Admin|null
	 */
	public $admin = null;

	/**
	 * API class instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_API|null
	 */
	public $api = null;

	/**
	 * Optimizer class instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_Optimizer|null
	 */
	public $optimizer = null;

	/**
	 * Meta class instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_Meta|null
	 */
	public $meta = null;

	/**
	 * Social class instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_Social|null
	 */
	public $social = null;

	/**
	 * Dashboard class instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_Dashboard|null
	 */
	public $dashboard = null;

	/**
	 * SEO Compat class instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_SEO_Compat|null
	 */
	public $seo_compat = null;

	/**
	 * REST API class instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_REST_API|null
	 */
	public $rest_api = null;

	/**
	 * AJAX class instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_AJAX|null
	 */
	public $ajax = null;

	/**
	 * Usage class instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_Usage|null
	 */
	public $usage = null;

	/**
	 * Cron class instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_Cron|null
	 */
	public $cron = null;

	/**
	 * Post List class instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_Post_List|null
	 */
	public $post_list = null;

	/**
	 * Media Library class instance (Pro).
	 *
	 * @since 1.0.0
	 * @var   TopRanker_Media_Library|null
	 */
	public $media_library = null;

	/**
	 * WooCommerce integration class instance (Pro).
	 *
	 * @since 1.0.0
	 * @var   TopRanker_WooCommerce|null
	 */
	public $woocommerce = null;

	/**
	 * Get the single instance of the class.
	 *
	 * @since  1.0.0
	 * @return TopRanker_Core
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * Private to enforce singleton pattern.
	 *
	 * @since 1.0.0
	 */
	private function __construct() {
		$this->load_dependencies();
		$this->init_hooks();
	}

	/**
	 * Prevent cloning.
	 *
	 * @since 1.0.0
	 */
	private function __clone() {}

	/**
	 * Prevent unserialization.
	 *
	 * @since 1.0.0
	 */
	public function __wakeup() {
		throw new \Exception( 'Cannot unserialize singleton' );
	}

	/**
	 * Load required dependencies.
	 *
	 * @since 1.0.0
	 */
	private function load_dependencies() {
		$includes_path = TOPRANKER_PATH . 'includes/';
		$pro_path      = $includes_path . 'pro/';

		// Core classes (always loaded).
		$this->load_file( $includes_path . 'class-topranker-usage.php' );
		$this->load_file( $includes_path . 'class-topranker-api.php' );
		$this->load_file( $includes_path . 'class-topranker-optimizer.php' );
		$this->load_file( $includes_path . 'class-topranker-meta.php' );
		$this->load_file( $includes_path . 'class-topranker-social.php' );
		$this->load_file( $includes_path . 'class-topranker-seo-compat.php' );
		$this->load_file( $includes_path . 'class-topranker-dashboard.php' );
		$this->load_file( $includes_path . 'class-topranker-cron.php' );

		// REST API (always loaded for Gutenberg).
		$this->load_file( $includes_path . 'class-topranker-rest-api.php' );

		// Admin-only classes.
		if ( is_admin() ) {
			$this->load_file( $includes_path . 'class-topranker-admin.php' );
			$this->load_file( $includes_path . 'class-topranker-ajax.php' );
			$this->load_file( $includes_path . 'class-topranker-post-list.php' );
		}

		// Pro classes (only when Pro is active AND files exist).
		if ( topranker_is_pro() ) {
			$this->load_pro_file( $pro_path . 'class-topranker-alt-tags.php' );
			$this->load_pro_file( $pro_path . 'class-topranker-internal-links.php' );
			$this->load_pro_file( $pro_path . 'class-topranker-schema.php' );
			$this->load_pro_file( $pro_path . 'class-topranker-bulk.php' );
			$this->load_pro_file( $pro_path . 'class-topranker-history.php' );
			$this->load_pro_file( $pro_path . 'class-topranker-seo-audit.php' );
			$this->load_pro_file( $pro_path . 'class-topranker-media-library.php' );

			// WooCommerce integration (only if WooCommerce is active).
			if ( class_exists( 'WooCommerce' ) ) {
				$this->load_pro_file( $pro_path . 'class-topranker-woocommerce.php' );
			}
		}
	}

	/**
	 * Load a file if it exists.
	 *
	 * @since 1.0.0
	 * @param string $file Full path to the file.
	 * @return bool True if file was loaded, false otherwise.
	 */
	private function load_file( $file ) {
		if ( file_exists( $file ) ) {
			require_once $file;
			return true;
		}
		return false;
	}

	/**
	 * Load a Pro file if it exists and Pro is active.
	 *
	 * @since 1.0.0
	 * @param string $file Full path to the file.
	 * @return bool True if file was loaded, false otherwise.
	 */
	private function load_pro_file( $file ) {
		if ( topranker_is_pro() && file_exists( $file ) ) {
			require_once $file;
			return true;
		}
		return false;
	}

	/**
	 * Initialize hooks.
	 *
	 * @since 1.0.0
	 */
	private function init_hooks() {
		// Plugin initialization.
		add_action( 'init', array( $this, 'init' ) );

		// Admin initialization.
		add_action( 'admin_init', array( $this, 'admin_init' ) );

		// Frontend meta tag output.
		add_action( 'wp_head', array( $this, 'wp_head' ), 1 );

		// Load text domain for translations.
		add_action( 'init', array( $this, 'load_textdomain' ) );

		// Plugin activation hook.
		register_activation_hook( TOPRANKER_PATH . 'topranker-ai.php', array( $this, 'activate' ) );

		// Plugin deactivation hook.
		register_deactivation_hook( TOPRANKER_PATH . 'topranker-ai.php', array( $this, 'deactivate' ) );
	}

	/**
	 * Initialize plugin on 'init' hook.
	 *
	 * @since 1.0.0
	 */
	public function init() {
		// Initialize usage tracking.
		if ( class_exists( 'TopRanker_Usage' ) ) {
			$this->usage = new TopRanker_Usage();
		}

		// Initialize API wrapper.
		if ( class_exists( 'TopRanker_API' ) ) {
			$this->api = new TopRanker_API();
		}

		// Initialize optimizer.
		if ( class_exists( 'TopRanker_Optimizer' ) ) {
			$this->optimizer = new TopRanker_Optimizer();
		}

		// Initialize meta handler.
		if ( class_exists( 'TopRanker_Meta' ) ) {
			$this->meta = new TopRanker_Meta();
		}

		// Initialize social meta handler.
		if ( class_exists( 'TopRanker_Social' ) ) {
			$this->social = new TopRanker_Social();
		}

		// Initialize SEO compatibility.
		if ( class_exists( 'TopRanker_SEO_Compat' ) ) {
			$this->seo_compat = new TopRanker_SEO_Compat();
		}

		// Initialize dashboard.
		if ( class_exists( 'TopRanker_Dashboard' ) ) {
			$this->dashboard = new TopRanker_Dashboard();
		}

		// Initialize cron.
		if ( class_exists( 'TopRanker_Cron' ) ) {
			$this->cron = new TopRanker_Cron();
		}

		// Initialize REST API.
		if ( class_exists( 'TopRanker_REST_API' ) ) {
			$this->rest_api = new TopRanker_REST_API();
		}

		// Initialize AJAX handlers (admin only).
		if ( is_admin() && class_exists( 'TopRanker_AJAX' ) ) {
			$this->ajax = new TopRanker_AJAX();
		}

		// Initialize admin (admin only).
		if ( is_admin() && class_exists( 'TopRanker_Admin' ) ) {
			$this->admin = new TopRanker_Admin();
		}

		// Initialize post list modifications (admin only).
		if ( is_admin() && class_exists( 'TopRanker_Post_List' ) ) {
			$this->post_list = new TopRanker_Post_List();
		}

		// Initialize Media Library integration (admin only, Pro only).
		if ( is_admin() && topranker_is_pro() && class_exists( 'TopRanker_Media_Library' ) ) {
			$this->media_library = new TopRanker_Media_Library();
		}

		// Initialize WooCommerce integration (Pro only, WooCommerce active).
		if ( topranker_is_pro() && class_exists( 'WooCommerce' ) && class_exists( 'TopRanker_WooCommerce' ) ) {
			$this->woocommerce = new TopRanker_WooCommerce();
		}

		/**
		 * Fires after TopRanker AI has initialized.
		 *
		 * @since 1.0.0
		 */
		do_action( 'topranker_init' );
	}

	/**
	 * Admin initialization on 'admin_init' hook.
	 *
	 * @since 1.0.0
	 */
	public function admin_init() {
		/**
		 * Fires after TopRanker AI admin has initialized.
		 *
		 * @since 1.0.0
		 */
		do_action( 'topranker_admin_init' );
	}

	/**
	 * Output meta tags in wp_head.
	 *
	 * @since 1.0.0
	 */
	public function wp_head() {
		// Meta tag output is handled by TopRanker_Meta and TopRanker_Social classes.
		// This method triggers the action for extensibility.

		/**
		 * Fires in wp_head for TopRanker meta output.
		 *
		 * @since 1.0.0
		 */
		do_action( 'topranker_wp_head' );
	}

	/**
	 * Load plugin text domain for translations.
	 *
	 * @since 1.0.0
	 */
	public function load_textdomain() {
		load_plugin_textdomain(
			'topranker-ai',
			false,
			dirname( TOPRANKER_BASENAME ) . '/languages'
		);
	}

	/**
	 * Plugin activation.
	 *
	 * @since 1.0.0
	 */
	public function activate() {
		// Set activation flag for first-run onboarding.
		if ( false === get_option( 'topranker_activated' ) ) {
			add_option( 'topranker_activated', time() );
			add_option( 'topranker_show_welcome', true );
		}

		// Initialize default settings.
		$this->init_default_settings();

		// Flush rewrite rules for any custom endpoints.
		flush_rewrite_rules();

		/**
		 * Fires after plugin activation.
		 *
		 * @since 1.0.0
		 */
		do_action( 'topranker_activate' );
	}

	/**
	 * Plugin deactivation.
	 *
	 * @since 1.0.0
	 */
	public function deactivate() {
		// Clear scheduled cron events.
		wp_clear_scheduled_hook( 'topranker_daily_stats_refresh' );
		wp_clear_scheduled_hook( 'topranker_auto_optimize' );

		// Flush rewrite rules.
		flush_rewrite_rules();

		/**
		 * Fires after plugin deactivation.
		 *
		 * @since 1.0.0
		 */
		do_action( 'topranker_deactivate' );
	}

	/**
	 * Initialize default settings.
	 *
	 * @since 1.0.0
	 */
	private function init_default_settings() {
		// Default model.
		if ( false === get_option( 'topranker_model' ) ) {
			add_option( 'topranker_model', 'gpt-5.2' );
		}

		// Default tone.
		if ( false === get_option( 'topranker_tone' ) ) {
			add_option( 'topranker_tone', 'professional' );
		}

		// Default enabled post types.
		if ( false === get_option( 'topranker_post_types' ) ) {
			add_option( 'topranker_post_types', array( 'post', 'page' ) );
		}

		// Default SEO mode (will be set based on detected plugins later).
		if ( false === get_option( 'topranker_seo_mode' ) ) {
			add_option( 'topranker_seo_mode', 'standalone' );
		}

		// Initialize usage tracking.
		if ( false === get_option( 'topranker_usage' ) ) {
			$current_month = gmdate( 'Y-m' );
			$next_month    = gmdate( 'Y-m-01', strtotime( 'first day of next month' ) );
			add_option(
				'topranker_usage',
				array(
					'count'      => 0,
					'month'      => $current_month,
					'reset_date' => $next_month,
				)
			);
		}
	}

	/**
	 * Get enabled post types.
	 *
	 * @since  1.0.0
	 * @return array Array of enabled post type slugs.
	 */
	public function get_enabled_post_types() {
		$post_types = get_option( 'topranker_post_types', array( 'post', 'page' ) );
		return is_array( $post_types ) ? $post_types : array( 'post', 'page' );
	}

	/**
	 * Check if a post type is enabled.
	 *
	 * @since  1.0.0
	 * @param  string $post_type Post type slug.
	 * @return bool True if enabled.
	 */
	public function is_post_type_enabled( $post_type ) {
		return in_array( $post_type, $this->get_enabled_post_types(), true );
	}

	/**
	 * Get the current SEO mode.
	 *
	 * @since  1.0.0
	 * @return string 'standalone', 'suggest', or 'sync'.
	 */
	public function get_seo_mode() {
		return get_option( 'topranker_seo_mode', 'standalone' );
	}

	/**
	 * Get the selected AI model.
	 *
	 * @since  1.0.0
	 * @return string Model identifier.
	 */
	public function get_model() {
		return get_option( 'topranker_model', 'gpt-5.2' );
	}

	/**
	 * Get the selected tone of voice.
	 *
	 * @since  1.0.0
	 * @return string Tone identifier.
	 */
	public function get_tone() {
		return get_option( 'topranker_tone', 'professional' );
	}

	/**
	 * Check if API key is configured.
	 *
	 * @since  1.0.0
	 * @return bool True if API key exists.
	 */
	public function has_api_key() {
		$api_key = get_option( 'topranker_api_key', '' );
		return ! empty( $api_key );
	}
}
