<?php
/**
 * WP Cron scheduled tasks for TopRanker AI.
 *
 * Handles background processing for auto-optimize on publish,
 * dashboard stats refresh, and other scheduled tasks.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * TopRanker Cron class.
 *
 * @since 1.0.0
 */
class TopRanker_Cron {

	/**
	 * Cron hook for auto-optimization.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	const HOOK_AUTO_OPTIMIZE = 'topranker_auto_optimize';

	/**
	 * Cron hook for daily stats refresh.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	const HOOK_DAILY_STATS = 'topranker_daily_stats_refresh';

	/**
	 * Post meta key for auto-optimizing flag.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	const META_AUTO_OPTIMIZING = '_topranker_auto_optimizing';

	/**
	 * Delay in seconds before auto-optimize runs after publish.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const AUTO_OPTIMIZE_DELAY = 10;

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		$this->init_hooks();
	}

	/**
	 * Initialize hooks.
	 *
	 * @since 1.0.0
	 */
	private function init_hooks() {
		// Register cron handlers.
		add_action( self::HOOK_AUTO_OPTIMIZE, array( $this, 'run_auto_optimize' ), 10, 1 );
		add_action( self::HOOK_DAILY_STATS, array( $this, 'run_daily_stats_refresh' ) );

		// Hook into post status transitions for auto-optimize (Pro only).
		if ( function_exists( 'topranker_is_pro' ) && topranker_is_pro() ) {
			add_action( 'transition_post_status', array( $this, 'schedule_auto_optimize' ), 10, 3 );
		}

		// Schedule daily stats refresh on plugin load.
		add_action( 'admin_init', array( $this, 'maybe_schedule_daily_stats' ) );

		// Clean up on post deletion.
		add_action( 'before_delete_post', array( $this, 'cleanup_on_delete' ) );
	}

	/**
	 * Schedule auto-optimize when a post transitions to 'publish'.
	 *
	 * Only runs for Pro users and enabled post types.
	 *
	 * @since 1.0.0
	 * @param string  $new_status New post status.
	 * @param string  $old_status Old post status.
	 * @param WP_Post $post       Post object.
	 */
	public function schedule_auto_optimize( $new_status, $old_status, $post ) {
		// Only trigger when transitioning TO publish (not when already published).
		if ( 'publish' !== $new_status || 'publish' === $old_status ) {
			return;
		}

		// Check if Pro is active.
		if ( ! function_exists( 'topranker_is_pro' ) || ! topranker_is_pro() ) {
			return;
		}

		// Check if auto-optimize is enabled in settings.
		if ( ! $this->is_auto_optimize_enabled() ) {
			return;
		}

		// Check if this post type is enabled.
		if ( ! $this->is_post_type_enabled( $post->post_type ) ) {
			return;
		}

		// Check if post already has TopRanker meta (don't auto-optimize if already done).
		$has_meta = get_post_meta( $post->ID, '_topranker_meta_title', true );
		if ( ! empty( $has_meta ) ) {
			return;
		}

		// Schedule the auto-optimize event.
		$this->schedule_single_optimize( $post->ID );
	}

	/**
	 * Schedule a single optimization event for a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True if scheduled, false if already scheduled or failed.
	 */
	public function schedule_single_optimize( $post_id ) {
		// Check if already scheduled.
		$existing = wp_next_scheduled( self::HOOK_AUTO_OPTIMIZE, array( $post_id ) );
		if ( $existing ) {
			return false;
		}

		// Set the auto-optimizing flag.
		update_post_meta( $post_id, self::META_AUTO_OPTIMIZING, true );

		// Schedule the event.
		$scheduled = wp_schedule_single_event(
			time() + self::AUTO_OPTIMIZE_DELAY,
			self::HOOK_AUTO_OPTIMIZE,
			array( $post_id )
		);

		if ( defined( 'TOPRANKER_DEBUG' ) && TOPRANKER_DEBUG ) {
			error_log( sprintf(
				'[TopRanker AI] Auto-optimize scheduled for post %d in %d seconds',
				$post_id,
				self::AUTO_OPTIMIZE_DELAY
			) );
		}

		return false !== $scheduled;
	}

	/**
	 * Run auto-optimization for a post.
	 *
	 * This runs in the background via WP Cron.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 */
	public function run_auto_optimize( $post_id ) {
		// Verify post exists.
		$post = get_post( $post_id );
		if ( ! $post ) {
			$this->clear_auto_optimizing_flag( $post_id );
			return;
		}

		// Verify post is still published.
		if ( 'publish' !== $post->post_status ) {
			$this->clear_auto_optimizing_flag( $post_id );
			return;
		}

		// Check Pro status again (in case license expired).
		if ( ! function_exists( 'topranker_is_pro' ) || ! topranker_is_pro() ) {
			$this->clear_auto_optimizing_flag( $post_id );
			return;
		}

		// Run the optimization.
		if ( class_exists( 'TopRanker_Optimizer' ) ) {
			$optimizer = new TopRanker_Optimizer();
			$result    = $optimizer->optimize( $post, true );

			// Apply the results automatically.
			if ( ! is_wp_error( $result ) && ! empty( $result['results'] ) ) {
				$selected = $this->prepare_auto_apply_data( $result['results'] );
				if ( ! empty( $selected ) ) {
					$optimizer->apply( $post_id, $selected );
				}
			}

			if ( defined( 'TOPRANKER_DEBUG' ) && TOPRANKER_DEBUG ) {
				if ( is_wp_error( $result ) ) {
					error_log( sprintf(
						'[TopRanker AI] Auto-optimize failed for post %d: %s',
						$post_id,
						$result->get_error_message()
					) );
				} else {
					error_log( sprintf(
						'[TopRanker AI] Auto-optimize completed for post %d',
						$post_id
					) );
				}
			}
		}

		// Clear the auto-optimizing flag.
		$this->clear_auto_optimizing_flag( $post_id );
	}

	/**
	 * Prepare data for auto-apply from optimization results.
	 *
	 * Selects the first (best) suggestion for each field.
	 *
	 * @since 1.0.0
	 * @param array $results Optimization results.
	 * @return array Data ready for apply method.
	 */
	private function prepare_auto_apply_data( $results ) {
		$data = array();

		// Meta title (take first suggestion).
		if ( ! empty( $results['meta_title']['suggestions'] ) && is_array( $results['meta_title']['suggestions'] ) ) {
			$first = reset( $results['meta_title']['suggestions'] );
			if ( ! empty( $first ) ) {
				$data['meta_title'] = is_array( $first ) ? ( $first['title'] ?? '' ) : $first;
			}
		}

		// Meta description (take first suggestion).
		if ( ! empty( $results['meta_description']['suggestions'] ) && is_array( $results['meta_description']['suggestions'] ) ) {
			$first = reset( $results['meta_description']['suggestions'] );
			if ( ! empty( $first ) ) {
				$data['meta_description'] = is_array( $first ) ? ( $first['description'] ?? '' ) : $first;
			}
		}

		// Excerpt.
		if ( ! empty( $results['excerpt'] ) ) {
			$excerpt = $results['excerpt'];
			if ( is_array( $excerpt ) ) {
				$excerpt = isset( $excerpt['excerpt'] ) ? $excerpt['excerpt'] : ( isset( $excerpt['text'] ) ? $excerpt['text'] : '' );
			}
			if ( ! empty( $excerpt ) ) {
				$data['excerpt'] = $excerpt;
			}
		}

		// Keyphrases.
		if ( ! empty( $results['keyphrases'] ) ) {
			if ( ! empty( $results['keyphrases']['primary'] ) ) {
				$data['focus_keyphrase'] = $results['keyphrases']['primary'];
			}
			if ( ! empty( $results['keyphrases']['secondary'] ) && is_array( $results['keyphrases']['secondary'] ) ) {
				$data['secondary_keyphrases'] = $results['keyphrases']['secondary'];
			}
		}

		// Social meta.
		if ( ! empty( $results['social_meta'] ) ) {
			if ( ! empty( $results['social_meta']['og_title'] ) ) {
				$data['og_title'] = $results['social_meta']['og_title'];
			}
			if ( ! empty( $results['social_meta']['og_description'] ) ) {
				$data['og_description'] = $results['social_meta']['og_description'];
			}
			if ( ! empty( $results['social_meta']['twitter_title'] ) ) {
				$data['twitter_title'] = $results['social_meta']['twitter_title'];
			}
			if ( ! empty( $results['social_meta']['twitter_description'] ) ) {
				$data['twitter_description'] = $results['social_meta']['twitter_description'];
			}
		}

		// Alt tags.
		if ( ! empty( $results['alt_tags']['results'] ) ) {
			$data['alt_tags'] = $results['alt_tags']['results'];
		}

		// Schema markup.
		if ( ! empty( $results['schema']['schema'] ) ) {
			$data['schema'] = $results['schema']['schema'];
		}

		// Internal links.
		if ( ! empty( $results['internal_links']['suggestions'] ) ) {
			$data['internal_links'] = $results['internal_links']['suggestions'];
		}

		return $data;
	}

	/**
	 * Clear the auto-optimizing flag for a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 */
	public function clear_auto_optimizing_flag( $post_id ) {
		delete_post_meta( $post_id, self::META_AUTO_OPTIMIZING );
	}

	/**
	 * Check if a post is currently being auto-optimized.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True if auto-optimizing is in progress.
	 */
	public function is_auto_optimizing( $post_id ) {
		return (bool) get_post_meta( $post_id, self::META_AUTO_OPTIMIZING, true );
	}

	/**
	 * Check if auto-optimize is enabled in settings.
	 *
	 * @since 1.0.0
	 * @return bool True if enabled.
	 */
	public function is_auto_optimize_enabled() {
		return (bool) get_option( 'topranker_auto_optimize', true );
	}

	/**
	 * Check if a post type is enabled for TopRanker.
	 *
	 * @since 1.0.0
	 * @param string $post_type Post type slug.
	 * @return bool True if enabled.
	 */
	private function is_post_type_enabled( $post_type ) {
		$enabled = get_option( 'topranker_post_types', array( 'post', 'page' ) );

		if ( ! is_array( $enabled ) ) {
			$enabled = array( 'post', 'page' );
		}

		return in_array( $post_type, $enabled, true );
	}

	/**
	 * Maybe schedule the daily stats refresh event.
	 *
	 * @since 1.0.0
	 */
	public function maybe_schedule_daily_stats() {
		if ( ! wp_next_scheduled( self::HOOK_DAILY_STATS ) ) {
			wp_schedule_event( time(), 'daily', self::HOOK_DAILY_STATS );
		}
	}

	/**
	 * Run the daily stats refresh.
	 *
	 * @since 1.0.0
	 */
	public function run_daily_stats_refresh() {
		if ( class_exists( 'TopRanker_Dashboard' ) ) {
			$dashboard = new TopRanker_Dashboard();
			$dashboard->refresh_stats();

			if ( defined( 'TOPRANKER_DEBUG' ) && TOPRANKER_DEBUG ) {
				error_log( '[TopRanker AI] Daily stats refresh completed' );
			}
		}
	}

	/**
	 * Clean up scheduled events when a post is deleted.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID being deleted.
	 */
	public function cleanup_on_delete( $post_id ) {
		// Unschedule any pending auto-optimize for this post.
		$timestamp = wp_next_scheduled( self::HOOK_AUTO_OPTIMIZE, array( $post_id ) );
		if ( $timestamp ) {
			wp_unschedule_event( $timestamp, self::HOOK_AUTO_OPTIMIZE, array( $post_id ) );
		}

		// Clear the auto-optimizing flag.
		$this->clear_auto_optimizing_flag( $post_id );
	}

	/**
	 * Unschedule all TopRanker cron events.
	 *
	 * Used during plugin deactivation.
	 *
	 * @since 1.0.0
	 */
	public static function unschedule_all_events() {
		// Unschedule daily stats refresh.
		$timestamp = wp_next_scheduled( self::HOOK_DAILY_STATS );
		if ( $timestamp ) {
			wp_unschedule_event( $timestamp, self::HOOK_DAILY_STATS );
		}

		// Note: Individual auto-optimize events are cleaned up via wp_clear_scheduled_hook
		// in the main plugin deactivation.
	}

	/**
	 * Get all pending auto-optimize jobs.
	 *
	 * @since 1.0.0
	 * @return array Array of pending jobs with post_id and scheduled_time.
	 */
	public function get_pending_jobs() {
		$crons = _get_cron_array();
		$jobs  = array();

		if ( empty( $crons ) ) {
			return $jobs;
		}

		foreach ( $crons as $timestamp => $cron ) {
			if ( isset( $cron[ self::HOOK_AUTO_OPTIMIZE ] ) ) {
				foreach ( $cron[ self::HOOK_AUTO_OPTIMIZE ] as $hook_data ) {
					if ( isset( $hook_data['args'][0] ) ) {
						$post_id = (int) $hook_data['args'][0];
						$jobs[] = array(
							'post_id'        => $post_id,
							'post_title'     => get_the_title( $post_id ),
							'scheduled_time' => $timestamp,
							'scheduled_date' => wp_date( 'Y-m-d H:i:s', $timestamp ),
							'runs_in'        => human_time_diff( time(), $timestamp ),
						);
					}
				}
			}
		}

		return $jobs;
	}

	/**
	 * Get the count of pending auto-optimize jobs.
	 *
	 * @since 1.0.0
	 * @return int Number of pending jobs.
	 */
	public function get_pending_count() {
		return count( $this->get_pending_jobs() );
	}

	/**
	 * Cancel a pending auto-optimize job for a specific post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True if cancelled, false if not found.
	 */
	public function cancel_auto_optimize( $post_id ) {
		$timestamp = wp_next_scheduled( self::HOOK_AUTO_OPTIMIZE, array( $post_id ) );

		if ( $timestamp ) {
			wp_unschedule_event( $timestamp, self::HOOK_AUTO_OPTIMIZE, array( $post_id ) );
			$this->clear_auto_optimizing_flag( $post_id );
			return true;
		}

		return false;
	}

	/**
	 * Get status information for display in admin.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return array Status information.
	 */
	public function get_post_auto_optimize_status( $post_id ) {
		$is_optimizing = $this->is_auto_optimizing( $post_id );
		$scheduled     = wp_next_scheduled( self::HOOK_AUTO_OPTIMIZE, array( $post_id ) );

		return array(
			'is_auto_optimizing' => $is_optimizing,
			'is_scheduled'       => (bool) $scheduled,
			'scheduled_time'     => $scheduled ? $scheduled : null,
			'scheduled_date'     => $scheduled ? wp_date( 'Y-m-d H:i:s', $scheduled ) : null,
			'runs_in'            => $scheduled ? human_time_diff( time(), $scheduled ) : null,
		);
	}
}
