<?php
/**
 * SEO Coverage Dashboard.
 *
 * Handles SEO coverage statistics calculation, caching, and dashboard display.
 * Provides methods to track which posts have SEO meta data and which are missing.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * TopRanker Dashboard class.
 *
 * @since 1.0.0
 */
class TopRanker_Dashboard {

	/**
	 * Transient key for cached stats.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	const CACHE_KEY = 'topranker_dashboard_stats';

	/**
	 * Cache TTL in seconds (1 hour).
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const CACHE_TTL = HOUR_IN_SECONDS;

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		$this->init_hooks();
	}

	/**
	 * Initialize hooks.
	 *
	 * @since 1.0.0
	 */
	private function init_hooks() {
		// Invalidate cache when post meta is updated.
		add_action( 'updated_post_meta', array( $this, 'maybe_invalidate_cache' ), 10, 4 );
		add_action( 'added_post_meta', array( $this, 'maybe_invalidate_cache' ), 10, 4 );
		add_action( 'deleted_post_meta', array( $this, 'maybe_invalidate_cache' ), 10, 4 );

		// Invalidate cache when posts are published/unpublished.
		add_action( 'transition_post_status', array( $this, 'invalidate_on_status_change' ), 10, 3 );

		// Register AJAX handler for refresh.
		add_action( 'wp_ajax_topranker_refresh_dashboard_stats', array( $this, 'ajax_refresh_stats' ) );

		// Register WordPress dashboard widget.
		add_action( 'wp_dashboard_setup', array( $this, 'register_dashboard_widget' ) );
	}

	/**
	 * Get dashboard statistics.
	 *
	 * Returns cached stats if available, otherwise calculates them.
	 *
	 * @since  1.0.0
	 * @param  bool $force_refresh Whether to force a cache refresh.
	 * @return array Dashboard statistics.
	 */
	public function get_stats( $force_refresh = false ) {
		$stats = get_transient( self::CACHE_KEY );

		if ( false === $stats || $force_refresh ) {
			$stats = $this->calculate_stats();
			set_transient( self::CACHE_KEY, $stats, self::CACHE_TTL );
		}

		return $stats;
	}

	/**
	 * Calculate dashboard statistics.
	 *
	 * Uses efficient COUNT queries to calculate SEO coverage.
	 * Never loads all posts into memory.
	 *
	 * @since  1.0.0
	 * @return array Dashboard statistics.
	 */
	public function calculate_stats() {
		global $wpdb;

		// Get enabled post types.
		$enabled_post_types = $this->get_enabled_post_types();

		if ( empty( $enabled_post_types ) ) {
			return $this->get_empty_stats();
		}

		// Prepare placeholders for post types.
		$placeholders = implode( ', ', array_fill( 0, count( $enabled_post_types ), '%s' ) );

		// Total published posts.
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $placeholders is safely generated.
		$total_posts = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_status = 'publish' AND post_type IN ({$placeholders})",
				...$enabled_post_types
			)
		);

		if ( 0 === $total_posts ) {
			return $this->get_empty_stats();
		}

		// Posts with meta titles.
		$with_meta_title = $this->count_posts_with_meta( '_topranker_meta_title', $enabled_post_types, $placeholders );

		// Posts with meta descriptions.
		$with_meta_description = $this->count_posts_with_meta( '_topranker_meta_description', $enabled_post_types, $placeholders );

		// Posts with focus keyphrase.
		$with_keyphrase = $this->count_posts_with_meta( '_topranker_focus_keyphrase', $enabled_post_types, $placeholders );

		// Posts with excerpts (standard WordPress excerpt).
		$with_excerpt = $this->count_posts_with_excerpt( $enabled_post_types, $placeholders );

		// Posts with social meta (OG title).
		$with_social_meta = $this->count_posts_with_meta( '_topranker_og_title', $enabled_post_types, $placeholders );

		// Count images missing alt tags (Pro feature - show teaser for free).
		$images_missing_alt = topranker_is_pro() ? $this->count_images_missing_alt() : null;

		// Calculate percentages.
		$meta_title_percent       = $this->calculate_percent( $with_meta_title, $total_posts );
		$meta_description_percent = $this->calculate_percent( $with_meta_description, $total_posts );
		$keyphrase_percent        = $this->calculate_percent( $with_keyphrase, $total_posts );
		$excerpt_percent          = $this->calculate_percent( $with_excerpt, $total_posts );
		$social_meta_percent      = $this->calculate_percent( $with_social_meta, $total_posts );

		// Overall coverage (weighted average).
		// Meta title and description are most important (30% each).
		// Keyphrase is important (20%).
		// Excerpt and social meta are secondary (10% each).
		$overall_coverage = (
			( $meta_title_percent * 0.30 ) +
			( $meta_description_percent * 0.30 ) +
			( $keyphrase_percent * 0.20 ) +
			( $excerpt_percent * 0.10 ) +
			( $social_meta_percent * 0.10 )
		);
		$overall_coverage = round( $overall_coverage );

		return array(
			'total_posts'       => $total_posts,
			'meta_title'        => array(
				'count'   => $with_meta_title,
				'missing' => $total_posts - $with_meta_title,
				'percent' => $meta_title_percent,
			),
			'meta_description'  => array(
				'count'   => $with_meta_description,
				'missing' => $total_posts - $with_meta_description,
				'percent' => $meta_description_percent,
			),
			'keyphrase'         => array(
				'count'   => $with_keyphrase,
				'missing' => $total_posts - $with_keyphrase,
				'percent' => $keyphrase_percent,
			),
			'excerpt'           => array(
				'count'   => $with_excerpt,
				'missing' => $total_posts - $with_excerpt,
				'percent' => $excerpt_percent,
			),
			'social_meta'       => array(
				'count'   => $with_social_meta,
				'missing' => $total_posts - $with_social_meta,
				'percent' => $social_meta_percent,
			),
			'images_missing_alt' => $images_missing_alt,
			'overall_coverage'  => $overall_coverage,
			'post_types'        => $enabled_post_types,
			'last_updated'      => time(),
			'is_pro'            => topranker_is_pro(),
		);
	}

	/**
	 * Count posts with a specific meta key.
	 *
	 * @since  1.0.0
	 * @param  string $meta_key         The meta key to check.
	 * @param  array  $post_types       Array of post types.
	 * @param  string $placeholders     SQL placeholders string.
	 * @return int Number of posts with the meta.
	 */
	private function count_posts_with_meta( $meta_key, $post_types, $placeholders ) {
		global $wpdb;

		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $placeholders is safely generated.
		return (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(DISTINCT p.ID) FROM {$wpdb->posts} p
				INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
				WHERE p.post_status = 'publish'
				AND p.post_type IN ({$placeholders})
				AND pm.meta_key = %s
				AND pm.meta_value != ''",
				...array_merge( $post_types, array( $meta_key ) )
			)
		);
	}

	/**
	 * Count posts with excerpts.
	 *
	 * @since  1.0.0
	 * @param  array  $post_types   Array of post types.
	 * @param  string $placeholders SQL placeholders string.
	 * @return int Number of posts with excerpts.
	 */
	private function count_posts_with_excerpt( $post_types, $placeholders ) {
		global $wpdb;

		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $placeholders is safely generated.
		return (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$wpdb->posts}
				WHERE post_status = 'publish'
				AND post_type IN ({$placeholders})
				AND post_excerpt != ''",
				...$post_types
			)
		);
	}

	/**
	 * Count images missing alt tags.
	 *
	 * Pro feature only.
	 *
	 * @since  1.0.0
	 * @return int Number of images missing alt tags.
	 */
	private function count_images_missing_alt() {
		global $wpdb;

		// Count attachments that are images and have empty or no alt text.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$total_images = (int) $wpdb->get_var(
			"SELECT COUNT(*) FROM {$wpdb->posts}
			WHERE post_type = 'attachment'
			AND post_mime_type LIKE 'image/%'
			AND post_status = 'inherit'"
		);

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$with_alt = (int) $wpdb->get_var(
			"SELECT COUNT(DISTINCT p.ID) FROM {$wpdb->posts} p
			INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
			WHERE p.post_type = 'attachment'
			AND p.post_mime_type LIKE 'image/%'
			AND p.post_status = 'inherit'
			AND pm.meta_key = '_wp_attachment_image_alt'
			AND pm.meta_value != ''"
		);

		return $total_images - $with_alt;
	}

	/**
	 * Calculate percentage.
	 *
	 * @since  1.0.0
	 * @param  int $count Count of items.
	 * @param  int $total Total items.
	 * @return int Percentage (0-100).
	 */
	private function calculate_percent( $count, $total ) {
		if ( 0 === $total ) {
			return 0;
		}
		return (int) round( ( $count / $total ) * 100 );
	}

	/**
	 * Get enabled post types.
	 *
	 * @since  1.0.0
	 * @return array Array of enabled post type slugs.
	 */
	private function get_enabled_post_types() {
		$post_types = get_option( 'topranker_post_types', array( 'post', 'page' ) );
		return is_array( $post_types ) ? $post_types : array( 'post', 'page' );
	}

	/**
	 * Get empty stats structure.
	 *
	 * @since  1.0.0
	 * @return array Empty stats.
	 */
	private function get_empty_stats() {
		return array(
			'total_posts'        => 0,
			'meta_title'         => array(
				'count'   => 0,
				'missing' => 0,
				'percent' => 0,
			),
			'meta_description'   => array(
				'count'   => 0,
				'missing' => 0,
				'percent' => 0,
			),
			'keyphrase'          => array(
				'count'   => 0,
				'missing' => 0,
				'percent' => 0,
			),
			'excerpt'            => array(
				'count'   => 0,
				'missing' => 0,
				'percent' => 0,
			),
			'social_meta'        => array(
				'count'   => 0,
				'missing' => 0,
				'percent' => 0,
			),
			'images_missing_alt' => null,
			'overall_coverage'   => 0,
			'post_types'         => $this->get_enabled_post_types(),
			'last_updated'       => time(),
			'is_pro'             => topranker_is_pro(),
		);
	}

	/**
	 * Maybe invalidate cache when post meta changes.
	 *
	 * Only invalidates for TopRanker meta keys.
	 *
	 * @since 1.0.0
	 * @param int    $meta_id    Meta ID.
	 * @param int    $object_id  Post ID.
	 * @param string $meta_key   Meta key.
	 * @param mixed  $meta_value Meta value.
	 */
	public function maybe_invalidate_cache( $meta_id, $object_id, $meta_key, $meta_value ) {
		// Only invalidate for TopRanker meta keys.
		$topranker_keys = array(
			'_topranker_meta_title',
			'_topranker_meta_description',
			'_topranker_focus_keyphrase',
			'_topranker_og_title',
			'_wp_attachment_image_alt',
		);

		if ( in_array( $meta_key, $topranker_keys, true ) ) {
			$this->invalidate_cache();
		}
	}

	/**
	 * Invalidate cache when post status changes.
	 *
	 * @since 1.0.0
	 * @param string  $new_status New post status.
	 * @param string  $old_status Old post status.
	 * @param WP_Post $post       Post object.
	 */
	public function invalidate_on_status_change( $new_status, $old_status, $post ) {
		// Only care about transitions to/from publish.
		if ( 'publish' === $new_status || 'publish' === $old_status ) {
			$enabled_types = $this->get_enabled_post_types();
			if ( in_array( $post->post_type, $enabled_types, true ) ) {
				$this->invalidate_cache();
			}
		}
	}

	/**
	 * Invalidate the dashboard stats cache.
	 *
	 * @since 1.0.0
	 */
	public function invalidate_cache() {
		delete_transient( self::CACHE_KEY );
	}

	/**
	 * Refresh stats.
	 *
	 * Calculates fresh stats and updates the cache.
	 *
	 * @since  1.0.0
	 * @return array Fresh stats.
	 */
	public function refresh_stats() {
		$this->invalidate_cache();
		return $this->get_stats( true );
	}

	/**
	 * AJAX handler to refresh dashboard stats.
	 *
	 * @since 1.0.0
	 */
	public function ajax_refresh_stats() {
		// Verify nonce.
		check_ajax_referer( 'topranker_admin', 'nonce' );

		// Check capabilities.
		if ( ! current_user_can( 'edit_posts' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'topranker-ai' ) ) );
		}

		$stats = $this->refresh_stats();

		wp_send_json_success( $stats );
	}

	/**
	 * Get posts missing meta title.
	 *
	 * @since  1.0.0
	 * @param  int $limit Maximum number of posts to return.
	 * @return array Array of post objects.
	 */
	public function get_posts_missing_meta_title( $limit = 20 ) {
		return $this->get_posts_missing_meta( '_topranker_meta_title', $limit );
	}

	/**
	 * Get posts missing meta description.
	 *
	 * @since  1.0.0
	 * @param  int $limit Maximum number of posts to return.
	 * @return array Array of post objects.
	 */
	public function get_posts_missing_meta_description( $limit = 20 ) {
		return $this->get_posts_missing_meta( '_topranker_meta_description', $limit );
	}

	/**
	 * Get posts missing focus keyphrase.
	 *
	 * @since  1.0.0
	 * @param  int $limit Maximum number of posts to return.
	 * @return array Array of post objects.
	 */
	public function get_posts_missing_keyphrase( $limit = 20 ) {
		return $this->get_posts_missing_meta( '_topranker_focus_keyphrase', $limit );
	}

	/**
	 * Get posts missing a specific meta key.
	 *
	 * @since  1.0.0
	 * @param  string $meta_key The meta key to check.
	 * @param  int    $limit    Maximum number of posts to return.
	 * @return array Array of post objects with ID, title, and edit link.
	 */
	private function get_posts_missing_meta( $meta_key, $limit = 20 ) {
		global $wpdb;

		$enabled_post_types = $this->get_enabled_post_types();

		if ( empty( $enabled_post_types ) ) {
			return array();
		}

		$placeholders = implode( ', ', array_fill( 0, count( $enabled_post_types ), '%s' ) );

		// Get posts that don't have this meta key or have empty value.
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $placeholders is safely generated.
		$post_ids = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT p.ID FROM {$wpdb->posts} p
				LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = %s
				WHERE p.post_status = 'publish'
				AND p.post_type IN ({$placeholders})
				AND (pm.meta_value IS NULL OR pm.meta_value = '')
				ORDER BY p.post_date DESC
				LIMIT %d",
				...array_merge( array( $meta_key ), $enabled_post_types, array( $limit ) )
			)
		);

		if ( empty( $post_ids ) ) {
			return array();
		}

		$posts = array();
		foreach ( $post_ids as $post_id ) {
			$post = get_post( $post_id );
			if ( $post ) {
				$posts[] = array(
					'ID'        => $post->ID,
					'title'     => get_the_title( $post ),
					'edit_link' => get_edit_post_link( $post->ID, 'raw' ),
					'post_type' => $post->post_type,
				);
			}
		}

		return $posts;
	}

	/**
	 * Get coverage grade based on percentage.
	 *
	 * @since  1.0.0
	 * @param  int $percent Coverage percentage.
	 * @return array Grade data with letter, label, and color.
	 */
	public function get_coverage_grade( $percent ) {
		if ( $percent >= 90 ) {
			return array(
				'letter' => 'A',
				'label'  => __( 'Excellent', 'topranker-ai' ),
				'color'  => 'green',
				'icon'   => 'yes-alt',
			);
		} elseif ( $percent >= 70 ) {
			return array(
				'letter' => 'B',
				'label'  => __( 'Good', 'topranker-ai' ),
				'color'  => 'blue',
				'icon'   => 'yes',
			);
		} elseif ( $percent >= 50 ) {
			return array(
				'letter' => 'C',
				'label'  => __( 'Fair', 'topranker-ai' ),
				'color'  => 'yellow',
				'icon'   => 'warning',
			);
		} elseif ( $percent >= 30 ) {
			return array(
				'letter' => 'D',
				'label'  => __( 'Needs Work', 'topranker-ai' ),
				'color'  => 'orange',
				'icon'   => 'warning',
			);
		} else {
			return array(
				'letter' => 'F',
				'label'  => __( 'Poor', 'topranker-ai' ),
				'color'  => 'red',
				'icon'   => 'dismiss',
			);
		}
	}

	/**
	 * Get the formatted "last updated" time.
	 *
	 * @since  1.0.0
	 * @param  int $timestamp Unix timestamp.
	 * @return string Human-readable time difference.
	 */
	public function get_last_updated_text( $timestamp ) {
		if ( ! $timestamp ) {
			return __( 'Never', 'topranker-ai' );
		}

		return sprintf(
			/* translators: %s: Human-readable time difference */
			__( '%s ago', 'topranker-ai' ),
			human_time_diff( $timestamp, time() )
		);
	}

	/**
	 * Get data for the dashboard view.
	 *
	 * Convenience method that combines stats with additional context.
	 *
	 * @since  1.0.0
	 * @return array Dashboard view data.
	 */
	public function get_view_data() {
		$stats = $this->get_stats();
		$grade = $this->get_coverage_grade( $stats['overall_coverage'] );

		return array(
			'stats'            => $stats,
			'grade'            => $grade,
			'last_updated'     => $this->get_last_updated_text( $stats['last_updated'] ),
			'is_pro'           => topranker_is_pro(),
			'upgrade_url'      => topranker_get_upgrade_url(),
			'bulk_url'         => admin_url( 'admin.php?page=topranker-ai-bulk' ),
			'missing_titles'   => $this->get_posts_missing_meta_title( 5 ),
			'missing_desc'     => $this->get_posts_missing_meta_description( 5 ),
		);
	}

	/**
	 * Register the WordPress dashboard widget.
	 *
	 * @since 1.0.0
	 */
	public function register_dashboard_widget() {
		// Only show widget if user can edit posts.
		if ( ! current_user_can( 'edit_posts' ) ) {
			return;
		}

		wp_add_dashboard_widget(
			'topranker_seo_coverage_widget',
			__( 'SEO Coverage - TopRanker AI', 'topranker-ai' ),
			array( $this, 'render_dashboard_widget' )
		);
	}

	/**
	 * Render the WordPress dashboard widget content.
	 *
	 * Shows a lightweight summary of SEO coverage stats with a link to the full dashboard.
	 *
	 * @since 1.0.0
	 */
	public function render_dashboard_widget() {
		// Use cached stats only for performance.
		$stats = $this->get_stats();
		$grade = $this->get_coverage_grade( $stats['overall_coverage'] );

		// Build the progress bar color class.
		$bar_class = 'is-good';
		if ( $stats['overall_coverage'] < 70 ) {
			$bar_class = $stats['overall_coverage'] >= 40 ? 'is-warning' : 'is-bad';
		}
		?>
		<div class="topranker-widget">
			<?php if ( 0 === $stats['total_posts'] ) : ?>
				<p class="topranker-widget-empty">
					<?php esc_html_e( 'No published content found. Publish some posts or pages to see your SEO coverage.', 'topranker-ai' ); ?>
				</p>
			<?php else : ?>
				<!-- Overall Coverage -->
				<div class="topranker-widget-score">
					<div class="topranker-widget-score-value">
						<?php echo esc_html( $stats['overall_coverage'] ); ?>%
					</div>
					<div class="topranker-widget-score-label">
						<?php esc_html_e( 'Overall SEO Coverage', 'topranker-ai' ); ?>
					</div>
					<div class="topranker-widget-bar">
						<div class="topranker-widget-bar-track">
							<div
								class="topranker-widget-bar-fill <?php echo esc_attr( $bar_class ); ?>"
								style="width: <?php echo esc_attr( $stats['overall_coverage'] ); ?>%;"
							></div>
						</div>
					</div>
					<div class="topranker-widget-grade <?php echo esc_attr( 'grade-' . strtolower( $grade['letter'] ) ); ?>">
						<span class="dashicons dashicons-<?php echo esc_attr( $grade['icon'] ); ?>" aria-hidden="true"></span>
						<?php echo esc_html( $grade['label'] ); ?>
					</div>
				</div>

				<!-- Quick Stats -->
				<div class="topranker-widget-stats">
					<div class="topranker-widget-stat">
						<span class="topranker-widget-stat-label"><?php esc_html_e( 'Total Content', 'topranker-ai' ); ?></span>
						<span class="topranker-widget-stat-value"><?php echo esc_html( number_format_i18n( $stats['total_posts'] ) ); ?></span>
					</div>
					<div class="topranker-widget-stat">
						<span class="topranker-widget-stat-label"><?php esc_html_e( 'Meta Titles', 'topranker-ai' ); ?></span>
						<span class="topranker-widget-stat-value">
							<?php echo esc_html( $stats['meta_title']['percent'] ); ?>%
							<small>(<?php echo esc_html( $stats['meta_title']['missing'] ); ?> <?php esc_html_e( 'missing', 'topranker-ai' ); ?>)</small>
						</span>
					</div>
					<div class="topranker-widget-stat">
						<span class="topranker-widget-stat-label"><?php esc_html_e( 'Meta Descriptions', 'topranker-ai' ); ?></span>
						<span class="topranker-widget-stat-value">
							<?php echo esc_html( $stats['meta_description']['percent'] ); ?>%
							<small>(<?php echo esc_html( $stats['meta_description']['missing'] ); ?> <?php esc_html_e( 'missing', 'topranker-ai' ); ?>)</small>
						</span>
					</div>
				</div>
			<?php endif; ?>

			<!-- Footer Link -->
			<div class="topranker-widget-footer">
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=topranker-ai-dashboard' ) ); ?>" class="button button-primary">
					<?php esc_html_e( 'View Full Dashboard', 'topranker-ai' ); ?>
				</a>
				<?php if ( ! topranker_is_pro() ) : ?>
					<a href="<?php echo esc_url( topranker_get_upgrade_url() ); ?>" class="topranker-widget-upgrade">
						<?php esc_html_e( 'Upgrade to Pro', 'topranker-ai' ); ?>
						<span class="dashicons dashicons-arrow-right-alt2" aria-hidden="true"></span>
					</a>
				<?php endif; ?>
			</div>
		</div>

		<style>
			.topranker-widget {
				margin: -12px;
			}
			.topranker-widget-empty {
				padding: 20px;
				text-align: center;
				color: #646970;
			}
			.topranker-widget-score {
				padding: 20px;
				text-align: center;
				border-bottom: 1px solid #dcdcde;
			}
			.topranker-widget-score-value {
				font-size: 36px;
				font-weight: 700;
				color: #1d2327;
				line-height: 1;
			}
			.topranker-widget-score-label {
				font-size: 12px;
				color: #646970;
				margin-top: 4px;
			}
			.topranker-widget-bar {
				margin-top: 12px;
				max-width: 200px;
				margin-left: auto;
				margin-right: auto;
			}
			.topranker-widget-bar-track {
				height: 8px;
				background: #dcdcde;
				border-radius: 4px;
				overflow: hidden;
			}
			.topranker-widget-bar-fill {
				height: 100%;
				border-radius: 4px;
				transition: width 0.3s ease;
			}
			.topranker-widget-bar-fill.is-good {
				background: #00a32a;
			}
			.topranker-widget-bar-fill.is-warning {
				background: #dba617;
			}
			.topranker-widget-bar-fill.is-bad {
				background: #d63638;
			}
			.topranker-widget-grade {
				display: inline-flex;
				align-items: center;
				gap: 4px;
				margin-top: 8px;
				font-size: 12px;
				font-weight: 600;
			}
			.topranker-widget-grade .dashicons {
				font-size: 14px;
				width: 14px;
				height: 14px;
			}
			.topranker-widget-grade.grade-a {
				color: #00a32a;
			}
			.topranker-widget-grade.grade-b {
				color: #2271b1;
			}
			.topranker-widget-grade.grade-c {
				color: #dba617;
			}
			.topranker-widget-grade.grade-d {
				color: #d63638;
			}
			.topranker-widget-grade.grade-f {
				color: #d63638;
			}
			.topranker-widget-stats {
				padding: 12px 20px;
				border-bottom: 1px solid #dcdcde;
			}
			.topranker-widget-stat {
				display: flex;
				justify-content: space-between;
				align-items: center;
				padding: 6px 0;
			}
			.topranker-widget-stat:not(:last-child) {
				border-bottom: 1px solid #f0f0f1;
			}
			.topranker-widget-stat-label {
				font-size: 13px;
				color: #646970;
			}
			.topranker-widget-stat-value {
				font-size: 13px;
				font-weight: 600;
				color: #1d2327;
			}
			.topranker-widget-stat-value small {
				font-weight: normal;
				color: #646970;
			}
			.topranker-widget-footer {
				display: flex;
				justify-content: space-between;
				align-items: center;
				padding: 12px 20px;
				background: #f6f7f7;
			}
			.topranker-widget-upgrade {
				display: inline-flex;
				align-items: center;
				gap: 2px;
				font-size: 12px;
				color: #2271b1;
				text-decoration: none;
			}
			.topranker-widget-upgrade:hover {
				color: #135e96;
			}
			.topranker-widget-upgrade .dashicons {
				font-size: 14px;
				width: 14px;
				height: 14px;
			}
		</style>
		<?php
	}
}
