<?php

/**
 * Freemius SDK integration.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */
// Exit if accessed directly.
if ( !defined( 'ABSPATH' ) ) {
    exit;
}
/**
 * Initialize Freemius SDK.
 *
 * @since  1.0.0
 * @return Freemius The Freemius SDK instance.
 */
function topranker_fs() {
    global $topranker_fs;
    if ( !isset( $topranker_fs ) ) {
        // Include Freemius SDK.
        $freemius_path = TOPRANKER_PATH . 'vendor/freemius/start.php';
        if ( !file_exists( $freemius_path ) ) {
            return null;
        }
        require_once $freemius_path;
        $topranker_fs = fs_dynamic_init( array(
            'id'               => '23987',
            'slug'             => 'topranker-ai',
            'premium_slug'     => 'topranker-ai-pro',
            'type'             => 'plugin',
            'public_key'       => 'pk_e832fb50869274242ea3602b35cb6',
            'is_premium'       => false,
            'is_org_compliant' => true,
            'has_addons'       => false,
            'has_paid_plans'   => true,
            'trial'            => array(
                'days'               => 7,
                'is_require_payment' => false,
            ),
            'menu'             => array(
                'slug'    => 'topranker-ai',
                'support' => false,
            ),
            'is_live'          => true,
        ) );
    }
    return $topranker_fs;
}

// Initialize Freemius.
topranker_fs();
// Hook uninstall cleanup into Freemius.
topranker_fs()->add_action( 'after_uninstall', 'topranker_fs_uninstall_cleanup' );
// Signal that SDK was initiated.
do_action( 'topranker_fs_loaded' );
/**
 * Clean up all plugin data on uninstall.
 *
 * Hooked to Freemius after_uninstall action so uninstall tracking works.
 *
 * @since 1.1.0
 */
function topranker_fs_uninstall_cleanup() {
    global $wpdb;
    // Delete all plugin options.
    $options_to_delete = array(
        'topranker_activated',
        'topranker_api_key',
        'topranker_model',
        'topranker_tone',
        'topranker_post_types',
        'topranker_seo_mode',
        'topranker_seo_mode_set',
        'topranker_usage',
        'topranker_auto_optimize',
        'topranker_auto_alt_tags',
        'topranker_auto_schema',
        'topranker_auto_internal_links',
        'topranker_show_welcome',
        'topranker_dismissed_notices',
        'topranker_acf_enabled',
        'topranker_acf_field_types'
    );
    foreach ( $options_to_delete as $option ) {
        delete_option( $option );
    }
    // Delete all plugin post meta.
    $meta_keys_to_delete = array(
        '_topranker_meta_title',
        '_topranker_meta_description',
        '_topranker_focus_keyphrase',
        '_topranker_secondary_keyphrases',
        '_topranker_og_title',
        '_topranker_og_description',
        '_topranker_twitter_title',
        '_topranker_twitter_description',
        '_topranker_schema',
        '_topranker_seo_score',
        '_topranker_last_optimized',
        '_topranker_content_hash',
        '_topranker_optimization_history',
        '_topranker_cached_suggestions',
        '_topranker_auto_optimizing',
        '_topranker_internal_links_cache'
    );
    foreach ( $meta_keys_to_delete as $meta_key ) {
        $wpdb->delete( $wpdb->postmeta, array(
            'meta_key' => $meta_key,
        ), array('%s') );
    }
    // Delete all plugin transients.
    $transients_to_delete = array('topranker_dashboard_stats', 'topranker_api_test_success');
    foreach ( $transients_to_delete as $transient ) {
        delete_transient( $transient );
    }
    // Delete dynamic transients.
    $wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s", $wpdb->esc_like( '_transient_topranker_' ) . '%', $wpdb->esc_like( '_transient_timeout_topranker_' ) . '%' ) );
    // Unschedule all plugin cron events.
    $cron_hooks_to_clear = array('topranker_auto_optimize', 'topranker_daily_stats_refresh');
    foreach ( $cron_hooks_to_clear as $hook ) {
        wp_clear_scheduled_hook( $hook );
        $timestamp = wp_next_scheduled( $hook );
        while ( $timestamp ) {
            wp_unschedule_event( $timestamp, $hook );
            $timestamp = wp_next_scheduled( $hook );
        }
    }
    // Clear auto-optimize cron events with post_id arguments.
    $cron_array = _get_cron_array();
    if ( is_array( $cron_array ) ) {
        $updated = false;
        foreach ( $cron_array as $timestamp => $crons ) {
            if ( isset( $crons['topranker_auto_optimize'] ) ) {
                unset($cron_array[$timestamp]['topranker_auto_optimize']);
                $updated = true;
                if ( empty( $cron_array[$timestamp] ) ) {
                    unset($cron_array[$timestamp]);
                }
            }
        }
        if ( $updated ) {
            _set_cron_array( $cron_array );
        }
    }
    // Delete user meta.
    $wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->usermeta} WHERE meta_key LIKE %s", $wpdb->esc_like( 'topranker_' ) . '%' ) );
    // Clear object cache.
    if ( function_exists( 'wp_cache_flush_group' ) ) {
        wp_cache_flush_group( 'topranker' );
    }
    wp_cache_delete( 'topranker_dashboard_stats', 'transient' );
    wp_cache_delete( 'topranker_api_test_success', 'transient' );
}

/**
 * Check if the Pro version is active.
 *
 * Uses Freemius can_use_premium_code() which checks both that this is the
 * premium codebase and the user has an active paid license or trial.
 *
 * @since  1.0.0
 * @return bool True if Pro features should be enabled.
 */
function topranker_is_pro() {
    $fs = topranker_fs();
    if ( !$fs ) {
        return false;
    }
    return $fs->can_use_premium_code();
}

/**
 * Get the Freemius upgrade URL.
 *
 * @since  1.0.0
 * @return string The upgrade URL or empty string if Freemius not available.
 */
function topranker_get_upgrade_url() {
    $fs = topranker_fs();
    if ( !$fs ) {
        return '';
    }
    return $fs->get_upgrade_url();
}

/**
 * Check if user is in trial period.
 *
 * @since  1.0.0
 * @return bool True if user is in trial.
 */
function topranker_is_trial() {
    $fs = topranker_fs();
    if ( !$fs ) {
        return false;
    }
    return $fs->is_trial();
}

/**
 * Get trial days remaining.
 *
 * @since  1.0.0
 * @return int|false Days remaining or false if not in trial.
 */
function topranker_trial_days_remaining() {
    $fs = topranker_fs();
    if ( !$fs || !$fs->is_trial() ) {
        return false;
    }
    $trial_plan = $fs->get_trial_plan();
    if ( !$trial_plan ) {
        return false;
    }
    $license = $fs->_get_license();
    if ( !$license || empty( $license->expiration ) ) {
        return false;
    }
    $expiration = $license->expiration;
    $now = time();
    $expiration = strtotime( $expiration );
    $diff = $expiration - $now;
    if ( $diff <= 0 ) {
        return 0;
    }
    return ceil( $diff / DAY_IN_SECONDS );
}
