<?php
/**
 * Meta title and description generation.
 *
 * Handles generation of SEO meta titles, descriptions, excerpts, and focus
 * keyphrases using OpenAI. Includes prompt engineering and response validation.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * TopRanker Meta class.
 *
 * @since 1.0.0
 */
class TopRanker_Meta {

	/**
	 * Maximum meta title length.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const META_TITLE_MAX_LENGTH = 60;

	/**
	 * Maximum meta description length.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const META_DESCRIPTION_MAX_LENGTH = 155;

	/**
	 * Maximum excerpt length.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const EXCERPT_MAX_LENGTH = 300;

	/**
	 * API instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_API|null
	 */
	private $api = null;

	/**
	 * Optimizer instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_Optimizer|null
	 */
	private $optimizer = null;

	/**
	 * SEO Compat instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_SEO_Compat|null
	 */
	private $seo_compat = null;

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		// Classes will be initialized lazily when needed.
		$this->init_hooks();
	}

	/**
	 * Initialize hooks for meta tag output.
	 *
	 * @since 1.0.0
	 */
	private function init_hooks() {
		// Hook into TopRanker's wp_head action for meta tag output.
		add_action( 'topranker_wp_head', array( $this, 'output_meta_tags' ) );
	}

	/**
	 * Get the API instance.
	 *
	 * @since  1.0.0
	 * @return TopRanker_API
	 */
	private function get_api() {
		if ( null === $this->api ) {
			$this->api = new TopRanker_API();
		}
		return $this->api;
	}

	/**
	 * Get the optimizer instance.
	 *
	 * @since  1.0.0
	 * @return TopRanker_Optimizer
	 */
	private function get_optimizer() {
		if ( null === $this->optimizer ) {
			$this->optimizer = new TopRanker_Optimizer();
		}
		return $this->optimizer;
	}

	/**
	 * Get the SEO Compat instance.
	 *
	 * @since  1.0.0
	 * @return TopRanker_SEO_Compat
	 */
	private function get_seo_compat() {
		if ( null === $this->seo_compat ) {
			$this->seo_compat = new TopRanker_SEO_Compat();
		}
		return $this->seo_compat;
	}

	/**
	 * Generate meta title suggestions for a post.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post           Post ID or WP_Post object.
	 * @param string      $focus_keyphrase Optional. Focus keyphrase to include.
	 * @return array|WP_Error Array with 'suggestions' key or WP_Error on failure.
	 */
	public function generate_meta_title( $post, $focus_keyphrase = '' ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return new WP_Error(
				'invalid_post',
				__( 'Invalid post.', 'topranker-ai' )
			);
		}

		$optimizer    = $this->get_optimizer();
		$api          = $this->get_api();
		$site_name    = get_bloginfo( 'name' );
		$post_title   = $post->post_title;
		$content      = $optimizer->prepare_content( $post, 16000 );
		$context      = $optimizer->build_context_prefix( $post );

		// Build the prompt.
		$prompt = $this->build_meta_title_prompt(
			$post_title,
			$content,
			$focus_keyphrase,
			$site_name,
			$context
		);

		// Call the API.
		$messages = array(
			array(
				'role'    => 'system',
				'content' => $context,
			),
			array(
				'role'    => 'user',
				'content' => $prompt,
			),
		);

		$response = $api->chat_completion( $messages );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		// Parse the JSON response.
		$parsed = $api->parse_json_response( $response['content'] );

		if ( is_wp_error( $parsed ) ) {
			// Retry once.
			$response = $api->chat_completion( $messages );

			if ( is_wp_error( $response ) ) {
				return $response;
			}

			$parsed = $api->parse_json_response( $response['content'] );

			if ( is_wp_error( $parsed ) ) {
				return $parsed;
			}
		}

		// Validate and format suggestions.
		$suggestions = $this->validate_meta_title_response( $parsed );

		if ( is_wp_error( $suggestions ) ) {
			return $suggestions;
		}

		return array(
			'suggestions' => $suggestions,
			'post_id'     => $post->ID,
		);
	}

	/**
	 * Build the meta title prompt.
	 *
	 * @since 1.0.0
	 * @param string $post_title      Post title.
	 * @param string $content         Prepared post content.
	 * @param string $focus_keyphrase Focus keyphrase if provided.
	 * @param string $site_name       Site name for optional suffix.
	 * @param string $context         Context prefix (unused in prompt body, used in system).
	 * @return string The formatted prompt.
	 */
	private function build_meta_title_prompt( $post_title, $content, $focus_keyphrase, $site_name, $context ) {
		$keyphrase_instruction = '';
		if ( ! empty( $focus_keyphrase ) ) {
			$keyphrase_instruction = sprintf(
				/* translators: %s: focus keyphrase */
				__( 'Include the focus keyphrase "%s" near the beginning of the title.', 'topranker-ai' ),
				$focus_keyphrase
			);
		}

		$prompt = sprintf(
			/* translators: 1: Post title, 2: Content excerpt, 3: Keyphrase instruction, 4: Site name, 5: Max length */
			__(
				'Generate 3 SEO-optimized meta title suggestions for the following content.

Post Title: %1$s

Content:
%2$s

Requirements:
- Maximum %5$d characters per title (strict limit)
- Be compelling and click-worthy without being clickbait
- Summarize the main value or topic clearly
%3$s
- Optionally append " | %4$s" if it fits within the character limit
- Each title should be unique and offer a different angle

Respond with valid JSON only, in this exact format:
{
  "suggestions": [
    {"title": "First title suggestion", "rank": 1},
    {"title": "Second title suggestion", "rank": 2},
    {"title": "Third title suggestion", "rank": 3}
  ]
}

Rank them by quality, with 1 being the best.',
				'topranker-ai'
			),
			$post_title,
			mb_substr( $content, 0, 4000 ),
			$keyphrase_instruction,
			$site_name,
			self::META_TITLE_MAX_LENGTH
		);

		return $prompt;
	}

	/**
	 * Validate and format meta title response.
	 *
	 * @since 1.0.0
	 * @param array $parsed Parsed JSON response.
	 * @return array|WP_Error Formatted suggestions or error.
	 */
	private function validate_meta_title_response( $parsed ) {
		if ( ! isset( $parsed['suggestions'] ) || ! is_array( $parsed['suggestions'] ) ) {
			return new WP_Error(
				'invalid_response',
				__( 'Invalid response format from AI.', 'topranker-ai' )
			);
		}

		$suggestions = array();

		foreach ( $parsed['suggestions'] as $suggestion ) {
			if ( ! isset( $suggestion['title'] ) ) {
				continue;
			}

			$title = sanitize_text_field( $suggestion['title'] );

			// Trim if too long.
			if ( mb_strlen( $title ) > self::META_TITLE_MAX_LENGTH ) {
				$title = $this->smart_truncate( $title, self::META_TITLE_MAX_LENGTH );
			}

			$suggestions[] = array(
				'title'  => $title,
				'length' => mb_strlen( $title ),
				'rank'   => isset( $suggestion['rank'] ) ? (int) $suggestion['rank'] : count( $suggestions ) + 1,
			);
		}

		if ( empty( $suggestions ) ) {
			return new WP_Error(
				'no_suggestions',
				__( 'No valid suggestions generated.', 'topranker-ai' )
			);
		}

		// Sort by rank.
		usort(
			$suggestions,
			function( $a, $b ) {
				return $a['rank'] - $b['rank'];
			}
		);

		return $suggestions;
	}

	/**
	 * Generate meta description suggestions for a post.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post           Post ID or WP_Post object.
	 * @param string      $focus_keyphrase Optional. Focus keyphrase to include.
	 * @return array|WP_Error Array with 'suggestions' key or WP_Error on failure.
	 */
	public function generate_meta_description( $post, $focus_keyphrase = '' ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return new WP_Error(
				'invalid_post',
				__( 'Invalid post.', 'topranker-ai' )
			);
		}

		$optimizer  = $this->get_optimizer();
		$api        = $this->get_api();
		$post_title = $post->post_title;
		$content    = $optimizer->prepare_content( $post, 16000 );
		$context    = $optimizer->build_context_prefix( $post );

		// Build the prompt.
		$prompt = $this->build_meta_description_prompt(
			$post_title,
			$content,
			$focus_keyphrase,
			$context
		);

		// Call the API.
		$messages = array(
			array(
				'role'    => 'system',
				'content' => $context,
			),
			array(
				'role'    => 'user',
				'content' => $prompt,
			),
		);

		$response = $api->chat_completion( $messages );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		// Parse the JSON response.
		$parsed = $api->parse_json_response( $response['content'] );

		if ( is_wp_error( $parsed ) ) {
			// Retry once.
			$response = $api->chat_completion( $messages );

			if ( is_wp_error( $response ) ) {
				return $response;
			}

			$parsed = $api->parse_json_response( $response['content'] );

			if ( is_wp_error( $parsed ) ) {
				return $parsed;
			}
		}

		// Validate and format suggestions.
		$suggestions = $this->validate_meta_description_response( $parsed );

		if ( is_wp_error( $suggestions ) ) {
			return $suggestions;
		}

		return array(
			'suggestions' => $suggestions,
			'post_id'     => $post->ID,
		);
	}

	/**
	 * Build the meta description prompt.
	 *
	 * @since 1.0.0
	 * @param string $post_title      Post title.
	 * @param string $content         Prepared post content.
	 * @param string $focus_keyphrase Focus keyphrase if provided.
	 * @param string $context         Context prefix.
	 * @return string The formatted prompt.
	 */
	private function build_meta_description_prompt( $post_title, $content, $focus_keyphrase, $context ) {
		$keyphrase_instruction = '';
		if ( ! empty( $focus_keyphrase ) ) {
			$keyphrase_instruction = sprintf(
				/* translators: %s: focus keyphrase */
				__( 'Include the focus keyphrase "%s" naturally in the description.', 'topranker-ai' ),
				$focus_keyphrase
			);
		}

		$prompt = sprintf(
			/* translators: 1: Post title, 2: Content excerpt, 3: Keyphrase instruction, 4: Max length */
			__(
				'Generate 3 SEO-optimized meta description suggestions for the following content.

Post Title: %1$s

Content:
%2$s

Requirements:
- Maximum %4$d characters per description (strict limit)
- Summarize the main value proposition clearly
- Include a subtle call-to-action (e.g., "Learn how...", "Discover...", "Find out...")
%3$s
- Make it compelling to encourage clicks from search results
- Each description should be unique and offer a different angle

Respond with valid JSON only, in this exact format:
{
  "suggestions": [
    {"description": "First description suggestion", "rank": 1},
    {"description": "Second description suggestion", "rank": 2},
    {"description": "Third description suggestion", "rank": 3}
  ]
}

Rank them by quality, with 1 being the best.',
				'topranker-ai'
			),
			$post_title,
			mb_substr( $content, 0, 4000 ),
			$keyphrase_instruction,
			self::META_DESCRIPTION_MAX_LENGTH
		);

		return $prompt;
	}

	/**
	 * Validate and format meta description response.
	 *
	 * @since 1.0.0
	 * @param array $parsed Parsed JSON response.
	 * @return array|WP_Error Formatted suggestions or error.
	 */
	private function validate_meta_description_response( $parsed ) {
		if ( ! isset( $parsed['suggestions'] ) || ! is_array( $parsed['suggestions'] ) ) {
			return new WP_Error(
				'invalid_response',
				__( 'Invalid response format from AI.', 'topranker-ai' )
			);
		}

		$suggestions = array();

		foreach ( $parsed['suggestions'] as $suggestion ) {
			if ( ! isset( $suggestion['description'] ) ) {
				continue;
			}

			$description = sanitize_text_field( $suggestion['description'] );

			// Trim if too long.
			if ( mb_strlen( $description ) > self::META_DESCRIPTION_MAX_LENGTH ) {
				$description = $this->smart_truncate( $description, self::META_DESCRIPTION_MAX_LENGTH );
			}

			$suggestions[] = array(
				'description' => $description,
				'length'      => mb_strlen( $description ),
				'rank'        => isset( $suggestion['rank'] ) ? (int) $suggestion['rank'] : count( $suggestions ) + 1,
			);
		}

		if ( empty( $suggestions ) ) {
			return new WP_Error(
				'no_suggestions',
				__( 'No valid suggestions generated.', 'topranker-ai' )
			);
		}

		// Sort by rank.
		usort(
			$suggestions,
			function( $a, $b ) {
				return $a['rank'] - $b['rank'];
			}
		);

		return $suggestions;
	}

	/**
	 * Generate excerpt for a post.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post Post ID or WP_Post object.
	 * @return array|WP_Error Array with 'excerpt' key or WP_Error on failure.
	 */
	public function generate_excerpt( $post ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return new WP_Error(
				'invalid_post',
				__( 'Invalid post.', 'topranker-ai' )
			);
		}

		$optimizer = $this->get_optimizer();
		$api       = $this->get_api();
		$content   = $optimizer->prepare_content( $post, 16000 );
		$context   = $optimizer->build_context_prefix( $post );

		// Build the prompt.
		$prompt = $this->build_excerpt_prompt( $content, $context );

		// Call the API.
		$messages = array(
			array(
				'role'    => 'system',
				'content' => $context,
			),
			array(
				'role'    => 'user',
				'content' => $prompt,
			),
		);

		$response = $api->chat_completion( $messages );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		// Parse the JSON response.
		$parsed = $api->parse_json_response( $response['content'] );

		if ( is_wp_error( $parsed ) ) {
			// Retry once.
			$response = $api->chat_completion( $messages );

			if ( is_wp_error( $response ) ) {
				return $response;
			}

			$parsed = $api->parse_json_response( $response['content'] );

			if ( is_wp_error( $parsed ) ) {
				return $parsed;
			}
		}

		// Validate and format excerpt.
		$excerpt = $this->validate_excerpt_response( $parsed );

		if ( is_wp_error( $excerpt ) ) {
			return $excerpt;
		}

		return array(
			'excerpt' => $excerpt,
			'post_id' => $post->ID,
		);
	}

	/**
	 * Build the excerpt prompt.
	 *
	 * @since 1.0.0
	 * @param string $content Prepared post content.
	 * @param string $context Context prefix.
	 * @return string The formatted prompt.
	 */
	private function build_excerpt_prompt( $content, $context ) {
		$prompt = sprintf(
			/* translators: 1: Content excerpt, 2: Max length */
			__(
				'Generate a compelling excerpt for the following content.

Content:
%1$s

Requirements:
- Maximum %2$d characters (strict limit)
- Summarize the key takeaway or main point
- Make it engaging and standalone-readable
- Do not start with "This article..." or similar phrases
- Write in first person or direct address when appropriate

Respond with valid JSON only, in this exact format:
{
  "excerpt": "The generated excerpt text here"
}',
				'topranker-ai'
			),
			mb_substr( $content, 0, 4000 ),
			self::EXCERPT_MAX_LENGTH
		);

		return $prompt;
	}

	/**
	 * Validate and format excerpt response.
	 *
	 * @since 1.0.0
	 * @param array $parsed Parsed JSON response.
	 * @return string|WP_Error Formatted excerpt or error.
	 */
	private function validate_excerpt_response( $parsed ) {
		if ( ! isset( $parsed['excerpt'] ) || ! is_string( $parsed['excerpt'] ) ) {
			return new WP_Error(
				'invalid_response',
				__( 'Invalid response format from AI.', 'topranker-ai' )
			);
		}

		$excerpt = sanitize_textarea_field( $parsed['excerpt'] );

		// Trim if too long.
		if ( mb_strlen( $excerpt ) > self::EXCERPT_MAX_LENGTH ) {
			$excerpt = $this->smart_truncate( $excerpt, self::EXCERPT_MAX_LENGTH );
		}

		if ( empty( $excerpt ) ) {
			return new WP_Error(
				'empty_excerpt',
				__( 'Generated excerpt was empty.', 'topranker-ai' )
			);
		}

		return $excerpt;
	}

	/**
	 * Generate focus keyphrase suggestions for a post.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post Post ID or WP_Post object.
	 * @return array|WP_Error Array with 'primary' and 'secondary' keys or WP_Error.
	 */
	public function generate_keyphrases( $post ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return new WP_Error(
				'invalid_post',
				__( 'Invalid post.', 'topranker-ai' )
			);
		}

		$optimizer  = $this->get_optimizer();
		$api        = $this->get_api();
		$post_title = $post->post_title;
		$content    = $optimizer->prepare_content( $post, 16000 );
		$context    = $optimizer->build_context_prefix( $post );

		// Build the prompt.
		$prompt = $this->build_keyphrase_prompt( $post_title, $content, $context );

		// Call the API.
		$messages = array(
			array(
				'role'    => 'system',
				'content' => $context,
			),
			array(
				'role'    => 'user',
				'content' => $prompt,
			),
		);

		$response = $api->chat_completion( $messages );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		// Parse the JSON response.
		$parsed = $api->parse_json_response( $response['content'] );

		if ( is_wp_error( $parsed ) ) {
			// Retry once.
			$response = $api->chat_completion( $messages );

			if ( is_wp_error( $response ) ) {
				return $response;
			}

			$parsed = $api->parse_json_response( $response['content'] );

			if ( is_wp_error( $parsed ) ) {
				return $parsed;
			}
		}

		// Validate and format keyphrases.
		$keyphrases = $this->validate_keyphrase_response( $parsed );

		if ( is_wp_error( $keyphrases ) ) {
			return $keyphrases;
		}

		return array(
			'primary'   => $keyphrases['primary'],
			'secondary' => $keyphrases['secondary'],
			'post_id'   => $post->ID,
		);
	}

	/**
	 * Build the keyphrase prompt.
	 *
	 * @since 1.0.0
	 * @param string $post_title Post title.
	 * @param string $content    Prepared post content.
	 * @param string $context    Context prefix.
	 * @return string The formatted prompt.
	 */
	private function build_keyphrase_prompt( $post_title, $content, $context ) {
		$prompt = sprintf(
			/* translators: 1: Post title, 2: Content excerpt */
			__(
				'Analyze the following content and suggest focus keyphrases for SEO.

Post Title: %1$s

Content:
%2$s

Requirements:
- Suggest ONE primary keyphrase (2-4 words) that best represents the main topic
- Suggest THREE secondary keyphrases (2-4 words each) for related topics
- Keyphrases should be terms people actually search for
- Avoid generic phrases; be specific to the content
- Consider search intent (informational, navigational, transactional)

Respond with valid JSON only, in this exact format:
{
  "primary": "main focus keyphrase",
  "secondary": ["first secondary", "second secondary", "third secondary"]
}',
				'topranker-ai'
			),
			$post_title,
			mb_substr( $content, 0, 4000 )
		);

		return $prompt;
	}

	/**
	 * Validate and format keyphrase response.
	 *
	 * @since 1.0.0
	 * @param array $parsed Parsed JSON response.
	 * @return array|WP_Error Formatted keyphrases or error.
	 */
	private function validate_keyphrase_response( $parsed ) {
		if ( ! isset( $parsed['primary'] ) || ! is_string( $parsed['primary'] ) ) {
			return new WP_Error(
				'invalid_response',
				__( 'Invalid response format from AI.', 'topranker-ai' )
			);
		}

		$primary   = sanitize_text_field( $parsed['primary'] );
		$secondary = array();

		if ( isset( $parsed['secondary'] ) && is_array( $parsed['secondary'] ) ) {
			foreach ( $parsed['secondary'] as $keyphrase ) {
				if ( is_string( $keyphrase ) ) {
					$secondary[] = sanitize_text_field( $keyphrase );
				}
			}
		}

		// Limit to 3 secondary keyphrases.
		$secondary = array_slice( $secondary, 0, 3 );

		if ( empty( $primary ) ) {
			return new WP_Error(
				'empty_keyphrase',
				__( 'Generated keyphrase was empty.', 'topranker-ai' )
			);
		}

		return array(
			'primary'   => $primary,
			'secondary' => $secondary,
		);
	}

	/**
	 * Smart truncate text at word boundary.
	 *
	 * @since 1.0.0
	 * @param string $text   Text to truncate.
	 * @param int    $length Maximum length.
	 * @return string Truncated text.
	 */
	private function smart_truncate( $text, $length ) {
		if ( mb_strlen( $text ) <= $length ) {
			return $text;
		}

		// Truncate to length.
		$truncated = mb_substr( $text, 0, $length );

		// Try to break at word boundary.
		$last_space = mb_strrpos( $truncated, ' ' );

		if ( false !== $last_space && $last_space > $length * 0.8 ) {
			$truncated = mb_substr( $truncated, 0, $last_space );
		}

		return trim( $truncated );
	}

	/**
	 * Save meta title for a post.
	 *
	 * @since 1.0.0
	 * @param int    $post_id    Post ID.
	 * @param string $meta_title Meta title to save.
	 * @return bool True on success.
	 */
	public function save_meta_title( $post_id, $meta_title ) {
		$meta_title = sanitize_text_field( $meta_title );

		if ( mb_strlen( $meta_title ) > self::META_TITLE_MAX_LENGTH ) {
			$meta_title = $this->smart_truncate( $meta_title, self::META_TITLE_MAX_LENGTH );
		}

		return (bool) update_post_meta( $post_id, '_topranker_meta_title', $meta_title );
	}

	/**
	 * Save meta description for a post.
	 *
	 * @since 1.0.0
	 * @param int    $post_id          Post ID.
	 * @param string $meta_description Meta description to save.
	 * @return bool True on success.
	 */
	public function save_meta_description( $post_id, $meta_description ) {
		$meta_description = sanitize_textarea_field( $meta_description );

		if ( mb_strlen( $meta_description ) > self::META_DESCRIPTION_MAX_LENGTH ) {
			$meta_description = $this->smart_truncate( $meta_description, self::META_DESCRIPTION_MAX_LENGTH );
		}

		return (bool) update_post_meta( $post_id, '_topranker_meta_description', $meta_description );
	}

	/**
	 * Save excerpt for a post.
	 *
	 * @since 1.0.0
	 * @param int    $post_id Post ID.
	 * @param string $excerpt Excerpt to save.
	 * @return bool True on success.
	 */
	public function save_excerpt( $post_id, $excerpt ) {
		$excerpt = sanitize_textarea_field( $excerpt );

		if ( mb_strlen( $excerpt ) > self::EXCERPT_MAX_LENGTH ) {
			$excerpt = $this->smart_truncate( $excerpt, self::EXCERPT_MAX_LENGTH );
		}

		// Update the WordPress post excerpt.
		$result = wp_update_post(
			array(
				'ID'           => $post_id,
				'post_excerpt' => $excerpt,
			)
		);

		return ! is_wp_error( $result );
	}

	/**
	 * Save focus keyphrase for a post.
	 *
	 * @since 1.0.0
	 * @param int    $post_id  Post ID.
	 * @param string $keyphrase Focus keyphrase to save.
	 * @return bool True on success.
	 */
	public function save_focus_keyphrase( $post_id, $keyphrase ) {
		$keyphrase = sanitize_text_field( $keyphrase );

		return (bool) update_post_meta( $post_id, '_topranker_focus_keyphrase', $keyphrase );
	}

	/**
	 * Save secondary keyphrases for a post.
	 *
	 * @since 1.0.0
	 * @param int   $post_id    Post ID.
	 * @param array $keyphrases Array of secondary keyphrases.
	 * @return bool True on success.
	 */
	public function save_secondary_keyphrases( $post_id, $keyphrases ) {
		if ( ! is_array( $keyphrases ) ) {
			$keyphrases = array();
		}

		$sanitized = array_map( 'sanitize_text_field', $keyphrases );
		$sanitized = array_slice( $sanitized, 0, 3 );

		return (bool) update_post_meta( $post_id, '_topranker_secondary_keyphrases', $sanitized );
	}

	/**
	 * Get saved meta title for a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return string Meta title or empty string.
	 */
	public function get_meta_title( $post_id ) {
		return get_post_meta( $post_id, '_topranker_meta_title', true );
	}

	/**
	 * Get saved meta description for a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return string Meta description or empty string.
	 */
	public function get_meta_description( $post_id ) {
		return get_post_meta( $post_id, '_topranker_meta_description', true );
	}

	/**
	 * Get saved focus keyphrase for a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return string Focus keyphrase or empty string.
	 */
	public function get_focus_keyphrase( $post_id ) {
		return get_post_meta( $post_id, '_topranker_focus_keyphrase', true );
	}

	/**
	 * Get saved secondary keyphrases for a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return array Array of secondary keyphrases.
	 */
	public function get_secondary_keyphrases( $post_id ) {
		$keyphrases = get_post_meta( $post_id, '_topranker_secondary_keyphrases', true );
		return is_array( $keyphrases ) ? $keyphrases : array();
	}

	/**
	 * Check if post has meta title.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True if meta title exists.
	 */
	public function has_meta_title( $post_id ) {
		return ! empty( $this->get_meta_title( $post_id ) );
	}

	/**
	 * Check if post has meta description.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True if meta description exists.
	 */
	public function has_meta_description( $post_id ) {
		return ! empty( $this->get_meta_description( $post_id ) );
	}

	/**
	 * Output meta tags in wp_head.
	 *
	 * Outputs meta title and description tags when:
	 * 1. We're on a singular post/page
	 * 2. The post has TopRanker meta values
	 * 3. SEO mode is 'standalone' (not 'suggest')
	 * 4. No duplicate tag from another SEO plugin
	 *
	 * @since 1.0.0
	 */
	public function output_meta_tags() {
		// Only output on singular posts/pages.
		if ( ! is_singular() ) {
			return;
		}

		$seo_compat = $this->get_seo_compat();

		// Use SEO Compat class to check if we should output meta description.
		if ( ! $seo_compat->should_output_meta_description() ) {
			return;
		}

		// Additional check: verify no SEO plugin has already output or will output meta description.
		if ( $this->is_meta_description_already_output() ) {
			return;
		}

		$post_id = get_queried_object_id();

		if ( ! $post_id ) {
			return;
		}

		// Check if post type is enabled.
		$enabled_post_types = get_option( 'topranker_post_types', array( 'post', 'page' ) );
		if ( ! is_array( $enabled_post_types ) ) {
			$enabled_post_types = array( 'post', 'page' );
		}

		$post_type = get_post_type( $post_id );
		if ( ! in_array( $post_type, $enabled_post_types, true ) ) {
			return;
		}

		// Output meta description.
		$this->output_meta_description( $post_id );
	}

	/**
	 * Output meta description tag.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 */
	private function output_meta_description( $post_id ) {
		$meta_description = $this->get_meta_description( $post_id );

		if ( empty( $meta_description ) ) {
			return;
		}

		printf(
			'<meta name="description" content="%s" />' . "\n",
			esc_attr( $meta_description )
		);
	}

	/**
	 * Check if meta description is already being output by another plugin.
	 *
	 * Uses multiple detection methods:
	 * 1. Check specific SEO plugin hooks that output meta description
	 * 2. Check if known SEO plugins are active and have meta output enabled
	 *
	 * @since  1.0.0
	 * @return bool True if meta description is handled by another plugin.
	 */
	private function is_meta_description_already_output() {
		// Check Yoast SEO specific hooks.
		if ( defined( 'WPSEO_VERSION' ) ) {
			// Yoast uses wpseo_head action for meta output.
			if ( has_action( 'wpseo_head' ) ) {
				return true;
			}
			// Also check if Yoast's frontend class is hooked.
			if ( class_exists( 'WPSEO_Frontend' ) ) {
				return true;
			}
		}

		// Check RankMath specific hooks.
		if ( class_exists( 'RankMath' ) ) {
			// RankMath uses rank_math/head action.
			if ( has_action( 'rank_math/head' ) ) {
				return true;
			}
			// Also check RankMath's head class.
			if ( class_exists( 'RankMath\\Head' ) ) {
				return true;
			}
		}

		// Check SEOPress specific hooks.
		if ( function_exists( 'seopress_init' ) ) {
			// Check if SEOPress titles module is active.
			$seopress_toggle = get_option( 'seopress_toggle', array() );
			if ( isset( $seopress_toggle['toggle-titles'] ) && '1' === $seopress_toggle['toggle-titles'] ) {
				return true;
			}
		}

		// Check All in One SEO Pack.
		if ( class_exists( 'AIOSEO\Plugin\AIOSEO' ) ) {
			// AIOSEO v4+ is active.
			return true;
		}
		if ( defined( 'AIOSEOP_VERSION' ) ) {
			// Legacy AIOSEO is active.
			return true;
		}

		// Check The SEO Framework.
		if ( function_exists( 'the_seo_framework' ) ) {
			$tsf = the_seo_framework();
			if ( is_object( $tsf ) && method_exists( $tsf, 'is_seo_active' ) ) {
				return true;
			}
			return true;
		}

		// Check Jetpack SEO module.
		if ( class_exists( 'Jetpack' ) && method_exists( 'Jetpack', 'is_module_active' ) ) {
			if ( Jetpack::is_module_active( 'seo-tools' ) ) {
				return true;
			}
		}

		// Check Squirrly SEO.
		if ( defined( 'SQ_VERSION' ) ) {
			return true;
		}

		// Check SmartCrawl.
		if ( defined( 'SMARTCRAWL_VERSION' ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Remove conflicting meta description hooks from other plugins.
	 *
	 * This method can be used to remove other plugin's meta output
	 * if TopRanker is set to standalone mode and the user explicitly
	 * wants TopRanker to handle all meta output.
	 *
	 * Note: This is an aggressive approach and should only be used
	 * when explicitly enabled by the user.
	 *
	 * @since 1.0.0
	 */
	public function remove_conflicting_meta_hooks() {
		// Remove Yoast SEO meta description.
		if ( class_exists( 'WPSEO_Frontend' ) ) {
			$wpseo_front = WPSEO_Frontend::get_instance();
			if ( $wpseo_front ) {
				remove_action( 'wpseo_head', array( $wpseo_front, 'metadesc' ), 10 );
			}
		}

		// Remove RankMath meta description.
		if ( class_exists( 'RankMath\\Head' ) ) {
			// RankMath's description is part of their head output.
			// Removing individual hooks requires deeper integration.
		}

		// Remove SEOPress meta description.
		if ( function_exists( 'seopress_titles_desc' ) ) {
			remove_action( 'wp_head', 'seopress_titles_desc', 1 );
		}
	}
}
