<?php
/**
 * Content optimization and preparation utility.
 *
 * Handles content preparation for AI processing including stripping HTML,
 * removing shortcodes, truncating to token limits, and generating content hashes.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * TopRanker Optimizer class.
 *
 * @since 1.0.0
 */
class TopRanker_Optimizer {

	/**
	 * Maximum characters for standard prompts (~4000 tokens).
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const MAX_CHARS_STANDARD = 16000;

	/**
	 * Meta handler instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_Meta|null
	 */
	private $meta = null;

	/**
	 * Social handler instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_Social|null
	 */
	private $social = null;

	/**
	 * Usage tracker instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_Usage|null
	 */
	private $usage = null;

	/**
	 * Maximum characters for long posts - first section.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const MAX_CHARS_FIRST = 12000;

	/**
	 * Maximum characters for long posts - last section.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const MAX_CHARS_LAST = 4000;

	/**
	 * Maximum characters for shorter prompts.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const MAX_CHARS_SHORT = 8000;

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		// Classes will be initialized lazily when needed.
	}

	/**
	 * Get the meta handler instance.
	 *
	 * @since  1.0.0
	 * @return TopRanker_Meta
	 */
	private function get_meta() {
		if ( null === $this->meta ) {
			$this->meta = new TopRanker_Meta();
		}
		return $this->meta;
	}

	/**
	 * Get the social handler instance.
	 *
	 * @since  1.0.0
	 * @return TopRanker_Social
	 */
	private function get_social() {
		if ( null === $this->social ) {
			$this->social = new TopRanker_Social();
		}
		return $this->social;
	}

	/**
	 * Get the usage tracker instance.
	 *
	 * @since  1.0.0
	 * @return TopRanker_Usage
	 */
	private function get_usage() {
		if ( null === $this->usage ) {
			$this->usage = new TopRanker_Usage();
		}
		return $this->usage;
	}

	/**
	 * Main optimization orchestrator.
	 *
	 * Runs all optimization tasks for a single post: meta title, description,
	 * excerpt, keyphrases, and social meta. Handles usage tracking, content
	 * hash checking, and caching.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post          Post ID or WP_Post object.
	 * @param bool        $force_refresh Optional. Force re-generation even if content unchanged. Default false.
	 * @return array|WP_Error Optimization results or WP_Error on failure.
	 */
	public function optimize( $post, $force_refresh = false ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return new WP_Error(
				'invalid_post',
				__( 'Invalid post.', 'topranker-ai' )
			);
		}

		// Check usage limits.
		$usage = $this->get_usage();

		if ( ! $usage->can_optimize() ) {
			return new WP_Error(
				'usage_limit_reached',
				$usage->get_limit_message()
			);
		}

		// Check if content has changed (use cached results if not).
		if ( ! $force_refresh && ! $this->has_content_changed( $post ) ) {
			$cached = $this->get_cached_suggestions( $post->ID );
			if ( ! empty( $cached ) ) {
				return array(
					'post_id'   => $post->ID,
					'cached'    => true,
					'results'   => $cached,
					'message'   => __( 'Using cached results. Content has not changed since last optimization.', 'topranker-ai' ),
				);
			}
		}

		$results = array(
			'meta_title'        => null,
			'meta_description'  => null,
			'excerpt'           => null,
			'keyphrases'        => null,
			'social_meta'       => null,
			'short_description' => null,
			'product_data'      => null,
			'alt_tags'          => null,
			'schema'            => null,
			'internal_links'    => null,
		);

		$errors = array();
		$meta   = $this->get_meta();
		$social = $this->get_social();

		// First, generate keyphrases as they're used by other prompts.
		$keyphrase_result = $meta->generate_keyphrases( $post );

		if ( is_wp_error( $keyphrase_result ) ) {
			$errors['keyphrases'] = $keyphrase_result->get_error_message();
		} else {
			$results['keyphrases'] = $keyphrase_result;
		}

		// Get focus keyphrase for other prompts.
		$focus_keyphrase = '';
		if ( ! empty( $results['keyphrases']['primary'] ) ) {
			$focus_keyphrase = $results['keyphrases']['primary'];
		}

		// Check if this is a WooCommerce product and use specialized optimization.
		$is_product = ( 'product' === $post->post_type && class_exists( 'WooCommerce' ) );
		$use_wc_optimization = $is_product
			&& function_exists( 'topranker_is_pro' )
			&& topranker_is_pro()
			&& class_exists( 'TopRanker_WooCommerce' );

		if ( $use_wc_optimization ) {
			// Use WooCommerce-specific optimization for products.
			$wc = new TopRanker_WooCommerce();
			$product_data = $wc->get_product_data( $post );

			if ( ! is_wp_error( $product_data ) ) {
				// Generate WooCommerce-specific meta title.
				$title_result = $wc->generate_product_meta_title( $post, $product_data, $focus_keyphrase );

				if ( is_wp_error( $title_result ) ) {
					$errors['meta_title'] = $title_result->get_error_message();
				} else {
					$results['meta_title'] = $title_result;
				}

				// Generate WooCommerce-specific meta description.
				$description_result = $wc->generate_product_meta_description( $post, $product_data, $focus_keyphrase );

				if ( is_wp_error( $description_result ) ) {
					$errors['meta_description'] = $description_result->get_error_message();
				} else {
					$results['meta_description'] = $description_result;
				}

				// Generate WooCommerce short description.
				$short_desc_result = $wc->generate_short_description( $post, $product_data );

				if ( is_wp_error( $short_desc_result ) ) {
					$errors['short_description'] = $short_desc_result->get_error_message();
				} else {
					$results['short_description'] = $short_desc_result;
				}

				// Add product data to results for reference.
				$results['product_data'] = $product_data;
			} else {
				// Fall back to standard optimization if product data can't be retrieved.
				$title_result = $meta->generate_meta_title( $post, $focus_keyphrase );

				if ( is_wp_error( $title_result ) ) {
					$errors['meta_title'] = $title_result->get_error_message();
				} else {
					$results['meta_title'] = $title_result;
				}

				$description_result = $meta->generate_meta_description( $post, $focus_keyphrase );

				if ( is_wp_error( $description_result ) ) {
					$errors['meta_description'] = $description_result->get_error_message();
				} else {
					$results['meta_description'] = $description_result;
				}
			}
		} else {
			// Generate standard meta title.
			$title_result = $meta->generate_meta_title( $post, $focus_keyphrase );

			if ( is_wp_error( $title_result ) ) {
				$errors['meta_title'] = $title_result->get_error_message();
			} else {
				$results['meta_title'] = $title_result;
			}

			// Generate standard meta description.
			$description_result = $meta->generate_meta_description( $post, $focus_keyphrase );

			if ( is_wp_error( $description_result ) ) {
				$errors['meta_description'] = $description_result->get_error_message();
			} else {
				$results['meta_description'] = $description_result;
			}
		}

		// Generate excerpt (skip for products since we generate short_description).
		if ( ! $use_wc_optimization ) {
			$excerpt_result = $meta->generate_excerpt( $post );

			if ( is_wp_error( $excerpt_result ) ) {
				$errors['excerpt'] = $excerpt_result->get_error_message();
			} else {
				$results['excerpt'] = $excerpt_result;
			}
		}

		// Generate social meta.
		$social_result = $social->generate_social_meta( $post, $focus_keyphrase );

		if ( is_wp_error( $social_result ) ) {
			$errors['social_meta'] = $social_result->get_error_message();
		} else {
			$results['social_meta'] = $social_result;
		}

		// Generate alt tags (Pro feature).
		if ( function_exists( 'topranker_is_pro' )
			&& topranker_is_pro()
			&& get_option( 'topranker_auto_alt_tags', true )
			&& class_exists( 'TopRanker_Alt_Tags' )
		) {
			$alt_tags    = new TopRanker_Alt_Tags();
			$alt_result  = $alt_tags->generate_alt_tags( $post, $focus_keyphrase );

			if ( is_wp_error( $alt_result ) ) {
				// no_images is not a failure — just means the post has no images.
				if ( 'no_images' !== $alt_result->get_error_code() ) {
					$errors['alt_tags'] = $alt_result->get_error_message();
				}
			} else {
				$results['alt_tags'] = $alt_result;
			}
		}

		// Generate schema markup (Pro feature).
		if ( function_exists( 'topranker_is_pro' )
			&& topranker_is_pro()
			&& get_option( 'topranker_auto_schema', true )
			&& class_exists( 'TopRanker_Schema' )
		) {
			$schema       = new TopRanker_Schema();
			$schema_result = $schema->generate_schema( $post, $focus_keyphrase );

			if ( is_wp_error( $schema_result ) ) {
				if ( 'no_content' !== $schema_result->get_error_code() ) {
					$errors['schema'] = $schema_result->get_error_message();
				}
			} else {
				$results['schema'] = $schema_result;
			}
		}

		// Generate internal linking suggestions (Pro feature).
		if ( function_exists( 'topranker_is_pro' )
			&& topranker_is_pro()
			&& get_option( 'topranker_auto_internal_links', true )
			&& class_exists( 'TopRanker_Internal_Links' )
		) {
			$internal_links = new TopRanker_Internal_Links();
			$links_result   = $internal_links->generate_suggestions( $post, $focus_keyphrase );

			if ( is_wp_error( $links_result ) ) {
				// no_candidates is not a failure — just means the site needs more content.
				if ( 'no_candidates' !== $links_result->get_error_code() ) {
					$errors['internal_links'] = $links_result->get_error_message();
				}
			} else {
				$results['internal_links'] = $links_result;
			}
		}

		// Check if we got at least some results.
		$has_results = false;
		foreach ( $results as $value ) {
			if ( null !== $value ) {
				$has_results = true;
				break;
			}
		}

		if ( ! $has_results ) {
			// All generation failed.
			$error_messages = implode( ' ', $errors );
			return new WP_Error(
				'optimization_failed',
				/* translators: %s: error messages */
				sprintf( __( 'Optimization failed: %s', 'topranker-ai' ), $error_messages )
			);
		}

		// Increment usage counter (only counted once per optimize click).
		$usage->increment();

		// Store content hash and timestamp.
		$this->store_content_hash( $post );
		$this->store_last_optimized( $post );

		// Cache the suggestions.
		$this->cache_suggestions( $post->ID, $results );

		// Update SEO score if Pro and audit class is available.
		$seo_score = null;
		if ( function_exists( 'topranker_is_pro' ) && topranker_is_pro() && class_exists( 'TopRanker_SEO_Audit' ) ) {
			$audit     = new TopRanker_SEO_Audit();
			$seo_score = $audit->update_score( $post );
		}

		$response = array(
			'post_id' => $post->ID,
			'cached'  => false,
			'results' => $results,
			'errors'  => $errors,
			'usage'   => $usage->get_stats(),
		);

		if ( null !== $seo_score ) {
			$response['seo_score'] = $seo_score;
		}

		return $response;
	}

	/**
	 * Apply selected optimization results to a post.
	 *
	 * Saves the selected suggestions to post meta, and in sync mode,
	 * also writes to the detected SEO plugin's meta fields.
	 *
	 * @since 1.0.0
	 * @param int   $post_id  Post ID.
	 * @param array $selected Array of selected values to apply.
	 * @return array|WP_Error Result array or WP_Error on failure.
	 */
	public function apply( $post_id, $selected ) {
		$post = get_post( $post_id );

		if ( ! $post ) {
			return new WP_Error(
				'invalid_post',
				__( 'Invalid post.', 'topranker-ai' )
			);
		}

		// Save to history before applying changes (Pro feature).
		if ( function_exists( 'topranker_is_pro' ) && topranker_is_pro() && class_exists( 'TopRanker_History' ) ) {
			$history = new TopRanker_History();
			$history->add_entry( $post_id, $selected );
		}

		$meta   = $this->get_meta();
		$social = $this->get_social();
		$saved  = array();

		// Apply meta title.
		if ( ! empty( $selected['meta_title'] ) ) {
			$meta->save_meta_title( $post_id, $selected['meta_title'] );
			$saved['meta_title'] = true;
		}

		// Apply meta description.
		if ( ! empty( $selected['meta_description'] ) ) {
			$meta->save_meta_description( $post_id, $selected['meta_description'] );
			$saved['meta_description'] = true;
		}

		// Apply excerpt.
		if ( ! empty( $selected['excerpt'] ) ) {
			$meta->save_excerpt( $post_id, $selected['excerpt'] );
			$saved['excerpt'] = true;
		}

		// Apply WooCommerce short description (Pro).
		if ( ! empty( $selected['short_description'] ) ) {
			$is_product = ( 'product' === $post->post_type && class_exists( 'WooCommerce' ) );
			if ( $is_product && function_exists( 'topranker_is_pro' ) && topranker_is_pro() && class_exists( 'TopRanker_WooCommerce' ) ) {
				$wc = new TopRanker_WooCommerce();
				if ( $wc->save_short_description( $post_id, $selected['short_description'] ) ) {
					$saved['short_description'] = true;
				}
			}
		}

		// Apply focus keyphrase.
		if ( ! empty( $selected['focus_keyphrase'] ) ) {
			$meta->save_focus_keyphrase( $post_id, $selected['focus_keyphrase'] );
			$saved['focus_keyphrase'] = true;
		}

		// Apply secondary keyphrases.
		if ( ! empty( $selected['secondary_keyphrases'] ) ) {
			$meta->save_secondary_keyphrases( $post_id, $selected['secondary_keyphrases'] );
			$saved['secondary_keyphrases'] = true;
		}

		// Apply OG title.
		if ( ! empty( $selected['og_title'] ) ) {
			$social->save_og_title( $post_id, $selected['og_title'] );
			$saved['og_title'] = true;
		}

		// Apply OG description.
		if ( ! empty( $selected['og_description'] ) ) {
			$social->save_og_description( $post_id, $selected['og_description'] );
			$saved['og_description'] = true;
		}

		// Apply Twitter title.
		if ( ! empty( $selected['twitter_title'] ) ) {
			$social->save_twitter_title( $post_id, $selected['twitter_title'] );
			$saved['twitter_title'] = true;
		}

		// Apply Twitter description.
		if ( ! empty( $selected['twitter_description'] ) ) {
			$social->save_twitter_description( $post_id, $selected['twitter_description'] );
			$saved['twitter_description'] = true;
		}

		// Apply alt tags (Pro feature).
		if ( ! empty( $selected['alt_tags'] )
			&& function_exists( 'topranker_is_pro' )
			&& topranker_is_pro()
			&& class_exists( 'TopRanker_Alt_Tags' )
		) {
			$alt_tags     = new TopRanker_Alt_Tags();
			$alt_result   = $alt_tags->apply_alt_tags( $selected['alt_tags'], $post_id );
			if ( ! empty( $alt_result['applied'] ) && $alt_result['applied'] > 0 ) {
				$saved['alt_tags'] = true;
			}
		}

		// Apply schema markup (Pro feature).
		if ( ! empty( $selected['schema'] )
			&& function_exists( 'topranker_is_pro' )
			&& topranker_is_pro()
			&& class_exists( 'TopRanker_Schema' )
		) {
			$schema = new TopRanker_Schema();
			$schema->save_schema( $post_id, $selected['schema'] );
			$saved['schema'] = true;
		}

		// Apply internal links (Pro feature).
		if ( ! empty( $selected['internal_links'] )
			&& function_exists( 'topranker_is_pro' )
			&& topranker_is_pro()
			&& class_exists( 'TopRanker_Internal_Links' )
		) {
			$internal_links   = new TopRanker_Internal_Links();
			$links_applied    = 0;

			foreach ( $selected['internal_links'] as $link ) {
				if ( ! empty( $link['anchor_text'] ) && ! empty( $link['target_url'] ) ) {
					$link_result = $internal_links->apply_link_to_content( $post_id, $link['anchor_text'], $link['target_url'] );
					if ( true === $link_result ) {
						++$links_applied;
					}
				}
			}

			if ( $links_applied > 0 ) {
				$saved['internal_links'] = true;
			}
		}

		// In sync mode, also write to the detected SEO plugin's meta fields.
		$synced      = array();
		$seo_compat  = new TopRanker_SEO_Compat();

		if ( $seo_compat->is_sync_mode() && $seo_compat->has_seo_plugin() ) {
			$synced = $this->sync_to_seo_plugin( $post_id, $selected, $seo_compat );
		}

		$result = array(
			'post_id' => $post_id,
			'saved'   => $saved,
			'message' => __( 'Selected optimizations have been applied.', 'topranker-ai' ),
		);

		if ( ! empty( $synced ) ) {
			$result['synced']       = $synced;
			$result['synced_to']    = $seo_compat->get_detected_plugin_name();
			$result['message']      = sprintf(
				/* translators: %s: SEO plugin name */
				__( 'Selected optimizations have been applied and synced to %s.', 'topranker-ai' ),
				$seo_compat->get_detected_plugin_name()
			);
		}

		// Recalculate SEO score after applying changes (Pro).
		if ( function_exists( 'topranker_is_pro' ) && topranker_is_pro() && class_exists( 'TopRanker_SEO_Audit' ) ) {
			$audit               = new TopRanker_SEO_Audit();
			$result['seo_score'] = $audit->update_score( $post_id );
		}

		return $result;
	}

	/**
	 * Sync selected values to the detected SEO plugin.
	 *
	 * @since 1.0.0
	 * @param int                  $post_id    Post ID.
	 * @param array                $selected   Array of selected values.
	 * @param TopRanker_SEO_Compat $seo_compat SEO compatibility instance.
	 * @return array Fields that were synced.
	 */
	private function sync_to_seo_plugin( $post_id, $selected, $seo_compat ) {
		$synced = array();

		// Map selected fields to SEO plugin fields.
		$field_map = array(
			'meta_title'          => 'meta_title',
			'meta_description'    => 'meta_description',
			'focus_keyphrase'     => 'focus_keyphrase',
			'og_title'            => 'og_title',
			'og_description'      => 'og_description',
			'twitter_title'       => 'twitter_title',
			'twitter_description' => 'twitter_description',
		);

		foreach ( $field_map as $selected_key => $field_type ) {
			if ( ! empty( $selected[ $selected_key ] ) ) {
				$result = $seo_compat->set_plugin_meta( $post_id, $field_type, $selected[ $selected_key ] );
				if ( $result ) {
					$synced[ $field_type ] = true;
				}
			}
		}

		return $synced;
	}

	/**
	 * Copy TopRanker meta to SEO plugin.
	 *
	 * Used in "suggest only" mode when user clicks "Copy to [SEO Plugin]" button.
	 *
	 * @since 1.0.0
	 * @param int   $post_id Post ID.
	 * @param array $fields  Optional. Specific fields to copy. If empty, copies all available fields.
	 * @return array|WP_Error Result array or WP_Error on failure.
	 */
	public function copy_to_seo_plugin( $post_id, $fields = array() ) {
		$post = get_post( $post_id );

		if ( ! $post ) {
			return new WP_Error(
				'invalid_post',
				__( 'Invalid post.', 'topranker-ai' )
			);
		}

		$seo_compat = new TopRanker_SEO_Compat();

		if ( ! $seo_compat->has_seo_plugin() ) {
			return new WP_Error(
				'no_seo_plugin',
				__( 'No SEO plugin detected.', 'topranker-ai' )
			);
		}

		// Get current TopRanker values.
		$existing = $this->get_existing_optimization( $post_id );

		// If no specific fields provided, sync all that have values.
		if ( empty( $fields ) ) {
			$fields = array(
				'meta_title',
				'meta_description',
				'focus_keyphrase',
				'og_title',
				'og_description',
				'twitter_title',
				'twitter_description',
			);
		}

		$synced = array();

		foreach ( $fields as $field ) {
			if ( ! empty( $existing[ $field ] ) ) {
				$result = $seo_compat->set_plugin_meta( $post_id, $field, $existing[ $field ] );
				if ( $result ) {
					$synced[ $field ] = true;
				}
			}
		}

		if ( empty( $synced ) ) {
			return new WP_Error(
				'nothing_to_copy',
				__( 'No TopRanker data to copy. Run optimization first.', 'topranker-ai' )
			);
		}

		return array(
			'post_id'   => $post_id,
			'synced'    => $synced,
			'synced_to' => $seo_compat->get_detected_plugin_name(),
			'message'   => sprintf(
				/* translators: %s: SEO plugin name */
				__( 'Data copied to %s successfully.', 'topranker-ai' ),
				$seo_compat->get_detected_plugin_name()
			),
		);
	}

	/**
	 * Cache suggestions in post meta.
	 *
	 * @since 1.0.0
	 * @param int   $post_id Post ID.
	 * @param array $results Optimization results to cache.
	 * @return bool True on success.
	 */
	private function cache_suggestions( $post_id, $results ) {
		$cache_data = array(
			'results'   => $results,
			'timestamp' => time(),
		);

		return (bool) update_post_meta( $post_id, '_topranker_cached_suggestions', $cache_data );
	}

	/**
	 * Get cached suggestions from post meta.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return array|null Cached results or null if not found.
	 */
	public function get_cached_suggestions( $post_id ) {
		$cached = get_post_meta( $post_id, '_topranker_cached_suggestions', true );

		if ( ! is_array( $cached ) || empty( $cached['results'] ) ) {
			return null;
		}

		return $cached['results'];
	}

	/**
	 * Get cached suggestions with metadata.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return array Cached data including timestamp, or empty array.
	 */
	public function get_cached_suggestions_with_meta( $post_id ) {
		$cached = get_post_meta( $post_id, '_topranker_cached_suggestions', true );

		if ( ! is_array( $cached ) ) {
			return array();
		}

		return $cached;
	}

	/**
	 * Clear cached suggestions for a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True on success.
	 */
	public function clear_cached_suggestions( $post_id ) {
		return delete_post_meta( $post_id, '_topranker_cached_suggestions' );
	}

	/**
	 * Get all suggestions for display in editor.
	 *
	 * Combines cached suggestions with currently saved meta values.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return array Combined suggestions and current values.
	 */
	public function get_suggestions_for_editor( $post_id ) {
		$cached   = $this->get_cached_suggestions_with_meta( $post_id );
		$existing = $this->get_existing_optimization( $post_id );
		$post     = get_post( $post_id );

		$response = array(
			'post_id'   => $post_id,
			'post_title' => $post ? $post->post_title : '',
			'permalink' => get_permalink( $post_id ),
			'cached'    => ! empty( $cached ),
		);

		// Add cached suggestions if available.
		if ( ! empty( $cached['results'] ) ) {
			$response['suggestions'] = $cached['results'];
			$response['cached_at']   = isset( $cached['timestamp'] ) ? $cached['timestamp'] : null;
		} else {
			$response['suggestions'] = null;
		}

		// Add currently saved values.
		$response['current'] = array(
			'meta_title'           => $existing['meta_title'],
			'meta_description'     => $existing['meta_description'],
			'focus_keyphrase'      => $existing['focus_keyphrase'],
			'secondary_keyphrases' => $existing['secondary_keyphrases'],
			'og_title'             => $existing['og_title'],
			'og_description'       => $existing['og_description'],
			'twitter_title'        => $existing['twitter_title'],
			'twitter_description'  => $existing['twitter_description'],
			'excerpt'              => $post ? $post->post_excerpt : '',
		);

		// Add WooCommerce product info if applicable.
		$is_product = $post && 'product' === $post->post_type && class_exists( 'WooCommerce' );
		$response['is_product'] = $is_product;

		if ( $is_product ) {
			$product = wc_get_product( $post_id );
			if ( $product ) {
				$response['current']['short_description'] = $product->get_short_description();
			}
		}

		// Add optimization metadata.
		$response['last_optimized'] = $existing['last_optimized'];
		$response['content_hash']   = $existing['content_hash'];
		$response['content_changed'] = $this->has_content_changed( $post_id );

		return $response;
	}

	/**
	 * Prepare post content for AI processing.
	 *
	 * Strips HTML, removes shortcodes, decodes entities, and truncates to limits.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post      Post ID or WP_Post object.
	 * @param int         $max_chars Optional. Maximum characters. Default is MAX_CHARS_STANDARD.
	 * @return string Prepared content.
	 */
	public function prepare_content( $post, $max_chars = null ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return '';
		}

		if ( null === $max_chars ) {
			$max_chars = self::MAX_CHARS_STANDARD;
		}

		$content = $post->post_content;

		// Append ACF field content if available.
		$acf_content = $this->get_acf_content( $post->ID );
		if ( ! empty( $acf_content ) ) {
			$content .= "\n\n" . $acf_content;
		}

		// Remove shortcodes.
		$content = $this->strip_shortcodes( $content );

		// Strip HTML tags.
		$content = wp_strip_all_tags( $content );

		// Decode HTML entities.
		$content = html_entity_decode( $content, ENT_QUOTES | ENT_HTML5, 'UTF-8' );

		// Collapse whitespace.
		$content = $this->collapse_whitespace( $content );

		// Trim.
		$content = trim( $content );

		// Handle long content.
		$content_length = mb_strlen( $content );

		if ( $content_length > $max_chars ) {
			$content = $this->truncate_smart( $content, $max_chars );
		}

		return $content;
	}

	/**
	 * Prepare content for long posts by combining first and last sections.
	 *
	 * For very long posts, this takes the first 12k chars and last 4k chars
	 * to capture both the intro and conclusion.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post Post ID or WP_Post object.
	 * @return string Prepared content.
	 */
	public function prepare_long_content( $post ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return '';
		}

		$content = $post->post_content;

		// Append ACF field content if available.
		$acf_content = $this->get_acf_content( $post->ID );
		if ( ! empty( $acf_content ) ) {
			$content .= "\n\n" . $acf_content;
		}

		// Remove shortcodes.
		$content = $this->strip_shortcodes( $content );

		// Strip HTML tags.
		$content = wp_strip_all_tags( $content );

		// Decode HTML entities.
		$content = html_entity_decode( $content, ENT_QUOTES | ENT_HTML5, 'UTF-8' );

		// Collapse whitespace.
		$content = $this->collapse_whitespace( $content );

		// Trim.
		$content = trim( $content );

		$content_length = mb_strlen( $content );
		$total_max      = self::MAX_CHARS_FIRST + self::MAX_CHARS_LAST;

		// If content is short enough, return as-is.
		if ( $content_length <= $total_max ) {
			return $content;
		}

		// Get first section.
		$first_section = mb_substr( $content, 0, self::MAX_CHARS_FIRST );

		// Get last section.
		$last_section = mb_substr( $content, -self::MAX_CHARS_LAST );

		// Combine with separator.
		return $first_section . "\n\n[...]\n\n" . $last_section;
	}

	/**
	 * Strip shortcodes from content.
	 *
	 * @since 1.0.0
	 * @param string $content Content to process.
	 * @return string Content with shortcodes removed.
	 */
	public function strip_shortcodes( $content ) {
		// Use WordPress strip_shortcodes function.
		$content = strip_shortcodes( $content );

		// Also remove any orphaned shortcode brackets.
		$content = preg_replace( '/\[[^\]]*\]/', '', $content );

		return $content;
	}

	/**
	 * Collapse whitespace in content.
	 *
	 * Replaces multiple spaces, tabs, and newlines with single spaces.
	 *
	 * @since 1.0.0
	 * @param string $content Content to process.
	 * @return string Content with collapsed whitespace.
	 */
	public function collapse_whitespace( $content ) {
		// Replace multiple newlines with single newline.
		$content = preg_replace( '/\n\s*\n/', "\n\n", $content );

		// Replace multiple spaces with single space.
		$content = preg_replace( '/[ \t]+/', ' ', $content );

		// Trim each line.
		$lines   = explode( "\n", $content );
		$lines   = array_map( 'trim', $lines );
		$content = implode( "\n", $lines );

		return $content;
	}

	/**
	 * Smart truncation that tries to break at sentence boundaries.
	 *
	 * @since 1.0.0
	 * @param string $content   Content to truncate.
	 * @param int    $max_chars Maximum characters.
	 * @return string Truncated content.
	 */
	public function truncate_smart( $content, $max_chars ) {
		if ( mb_strlen( $content ) <= $max_chars ) {
			return $content;
		}

		// Get the substring.
		$truncated = mb_substr( $content, 0, $max_chars );

		// Try to find the last sentence boundary.
		$last_period    = mb_strrpos( $truncated, '. ' );
		$last_question  = mb_strrpos( $truncated, '? ' );
		$last_exclaim   = mb_strrpos( $truncated, '! ' );
		$last_paragraph = mb_strrpos( $truncated, "\n\n" );

		// Find the best break point (must be in last 20% of content).
		$min_position = $max_chars * 0.8;
		$break_point  = false;

		$candidates = array_filter(
			array( $last_period, $last_question, $last_exclaim, $last_paragraph ),
			function( $pos ) use ( $min_position ) {
				return false !== $pos && $pos >= $min_position;
			}
		);

		if ( ! empty( $candidates ) ) {
			$break_point = max( $candidates );
		}

		if ( false !== $break_point ) {
			$truncated = mb_substr( $content, 0, $break_point + 1 );
		}

		return trim( $truncated );
	}

	/**
	 * Get text content from ACF fields for a post.
	 *
	 * Extracts content from text-based ACF fields to include in
	 * AI optimization prompts, giving the AI a fuller picture of the page.
	 *
	 * @since 1.1.0
	 * @param int $post_id Post ID.
	 * @return string Combined ACF field content, or empty string.
	 */
	private function get_acf_content( $post_id ) {
		// Check if ACF is active and integration is enabled.
		if ( ! function_exists( 'get_field_objects' ) ) {
			return '';
		}

		if ( ! get_option( 'topranker_acf_enabled', true ) ) {
			return '';
		}

		$field_objects = get_field_objects( $post_id );

		if ( empty( $field_objects ) || ! is_array( $field_objects ) ) {
			return '';
		}

		$allowed_types = get_option( 'topranker_acf_field_types', array( 'text', 'textarea', 'wysiwyg' ) );
		$parts         = array();

		foreach ( $field_objects as $field ) {
			if ( ! isset( $field['type'], $field['value'] ) ) {
				continue;
			}

			if ( ! in_array( $field['type'], $allowed_types, true ) ) {
				continue;
			}

			// Skip empty values.
			$value = $field['value'];
			if ( empty( $value ) || ! is_string( $value ) ) {
				continue;
			}

			// Skip private/internal fields.
			if ( isset( $field['name'] ) && 0 === strpos( $field['name'], '_' ) ) {
				continue;
			}

			// Strip HTML for WYSIWYG fields.
			$value = wp_strip_all_tags( $value );
			$value = trim( $value );

			if ( ! empty( $value ) ) {
				$parts[] = $value;
			}
		}

		return implode( "\n\n", $parts );
	}

	/**
	 * Generate a content hash for caching purposes.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post Post ID or WP_Post object.
	 * @return string MD5 hash of the prepared content.
	 */
	public function get_content_hash( $post ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return '';
		}

		// Include title, content, and ACF fields in the hash.
		$hash_content = $post->post_title . '::' . $post->post_content;

		$acf_content = $this->get_acf_content( $post->ID );
		if ( ! empty( $acf_content ) ) {
			$hash_content .= '::' . $acf_content;
		}

		return md5( $hash_content );
	}

	/**
	 * Check if content has changed since last optimization.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post Post ID or WP_Post object.
	 * @return bool True if content has changed.
	 */
	public function has_content_changed( $post ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return true;
		}

		$stored_hash  = get_post_meta( $post->ID, '_topranker_content_hash', true );
		$current_hash = $this->get_content_hash( $post );

		return $stored_hash !== $current_hash;
	}

	/**
	 * Store the content hash after optimization.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post Post ID or WP_Post object.
	 * @return bool True on success, false on failure.
	 */
	public function store_content_hash( $post ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return false;
		}

		$hash = $this->get_content_hash( $post );

		return (bool) update_post_meta( $post->ID, '_topranker_content_hash', $hash );
	}

	/**
	 * Build the context prefix for AI prompts.
	 *
	 * Includes site name, description, locale, and tone.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post Optional. Post for language detection.
	 * @return string Context prefix for prompts.
	 */
	public function build_context_prefix( $post = null ) {
		$locale = $this->detect_post_language( $post );
		$tone   = get_option( 'topranker_tone', 'professional' );

		$site_name        = get_bloginfo( 'name' );
		$site_description = get_bloginfo( 'description' );
		$language_name    = $this->get_language_name( $locale );

		$tone_labels = array(
			'professional' => __( 'professional', 'topranker-ai' ),
			'casual'       => __( 'casual', 'topranker-ai' ),
			'technical'    => __( 'technical', 'topranker-ai' ),
			'friendly'     => __( 'friendly', 'topranker-ai' ),
		);

		$tone_label = isset( $tone_labels[ $tone ] ) ? $tone_labels[ $tone ] : $tone_labels['professional'];

		$context = sprintf(
			/* translators: 1: Site name, 2: Site description, 3: Tone, 4: Language */
			__(
				'You are an SEO optimization assistant for the website "%1$s" (%2$s). ' .
				'Use a %3$s tone. Generate all output in %4$s.',
				'topranker-ai'
			),
			$site_name,
			$site_description,
			$tone_label,
			$language_name
		);

		return $context;
	}

	/**
	 * Detect the language of a post.
	 *
	 * Checks WPML, Polylang, then falls back to site locale.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post Post ID or WP_Post object.
	 * @return string Locale code.
	 */
	public function detect_post_language( $post = null ) {
		$post = get_post( $post );

		if ( $post ) {
			// Check WPML.
			if ( function_exists( 'wpml_get_language_information' ) ) {
				$lang_info = apply_filters( 'wpml_post_language_details', null, $post->ID );
				if ( is_array( $lang_info ) && ! empty( $lang_info['locale'] ) ) {
					return $lang_info['locale'];
				}
			}

			// Check Polylang.
			if ( function_exists( 'pll_get_post_language' ) ) {
				$lang = pll_get_post_language( $post->ID, 'locale' );
				if ( $lang ) {
					return $lang;
				}
			}
		}

		// Fall back to site locale.
		return get_locale();
	}

	/**
	 * Get the human-readable language name from a locale code.
	 *
	 * @since 1.0.0
	 * @param string $locale Locale code (e.g., 'en_US', 'sv_SE').
	 * @return string Language name.
	 */
	public function get_language_name( $locale ) {
		$languages = array(
			'en_US' => 'English',
			'en_GB' => 'English',
			'en_AU' => 'English',
			'sv_SE' => 'Swedish',
			'de_DE' => 'German',
			'de_AT' => 'German',
			'de_CH' => 'German',
			'fr_FR' => 'French',
			'fr_CA' => 'French',
			'es_ES' => 'Spanish',
			'es_MX' => 'Spanish',
			'it_IT' => 'Italian',
			'pt_BR' => 'Portuguese',
			'pt_PT' => 'Portuguese',
			'nl_NL' => 'Dutch',
			'nl_BE' => 'Dutch',
			'pl_PL' => 'Polish',
			'ru_RU' => 'Russian',
			'ja'    => 'Japanese',
			'ko_KR' => 'Korean',
			'zh_CN' => 'Chinese',
			'zh_TW' => 'Chinese',
			'ar'    => 'Arabic',
			'he_IL' => 'Hebrew',
			'fi'    => 'Finnish',
			'da_DK' => 'Danish',
			'nb_NO' => 'Norwegian',
			'nn_NO' => 'Norwegian',
		);

		// Check for exact match.
		if ( isset( $languages[ $locale ] ) ) {
			return $languages[ $locale ];
		}

		// Check for language code only (first 2 chars).
		$lang_code = substr( $locale, 0, 2 );
		foreach ( $languages as $loc => $name ) {
			if ( 0 === strpos( $loc, $lang_code ) ) {
				return $name;
			}
		}

		// Fallback to English.
		return 'English';
	}

	/**
	 * Get the word count of a post.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post Post ID or WP_Post object.
	 * @return int Word count.
	 */
	public function get_word_count( $post ) {
		$content = $this->prepare_content( $post );

		if ( empty( $content ) ) {
			return 0;
		}

		return str_word_count( $content );
	}

	/**
	 * Check if post has enough content for optimization.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post      Post ID or WP_Post object.
	 * @param int         $min_words Minimum words required. Default 50.
	 * @return bool True if post has enough content.
	 */
	public function has_enough_content( $post, $min_words = 50 ) {
		return $this->get_word_count( $post ) >= $min_words;
	}

	/**
	 * Get post data for AI context.
	 *
	 * Returns an array with title, prepared content, excerpt, and categories.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post Post ID or WP_Post object.
	 * @return array|false Post data array or false on failure.
	 */
	public function get_post_data( $post ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return false;
		}

		$categories = array();
		$terms      = get_the_terms( $post->ID, 'category' );
		if ( $terms && ! is_wp_error( $terms ) ) {
			foreach ( $terms as $term ) {
				$categories[] = $term->name;
			}
		}

		$tags      = array();
		$tag_terms = get_the_terms( $post->ID, 'post_tag' );
		if ( $tag_terms && ! is_wp_error( $tag_terms ) ) {
			foreach ( $tag_terms as $term ) {
				$tags[] = $term->name;
			}
		}

		return array(
			'id'           => $post->ID,
			'title'        => $post->post_title,
			'content'      => $this->prepare_content( $post ),
			'excerpt'      => $post->post_excerpt,
			'categories'   => $categories,
			'tags'         => $tags,
			'post_type'    => $post->post_type,
			'permalink'    => get_permalink( $post->ID ),
			'word_count'   => $this->get_word_count( $post ),
			'content_hash' => $this->get_content_hash( $post ),
		);
	}

	/**
	 * Get existing optimization data for a post.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post Post ID or WP_Post object.
	 * @return array Existing optimization data.
	 */
	public function get_existing_optimization( $post ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return array();
		}

		return array(
			'meta_title'            => get_post_meta( $post->ID, '_topranker_meta_title', true ),
			'meta_description'      => get_post_meta( $post->ID, '_topranker_meta_description', true ),
			'focus_keyphrase'       => get_post_meta( $post->ID, '_topranker_focus_keyphrase', true ),
			'secondary_keyphrases'  => get_post_meta( $post->ID, '_topranker_secondary_keyphrases', true ),
			'og_title'              => get_post_meta( $post->ID, '_topranker_og_title', true ),
			'og_description'        => get_post_meta( $post->ID, '_topranker_og_description', true ),
			'twitter_title'         => get_post_meta( $post->ID, '_topranker_twitter_title', true ),
			'twitter_description'   => get_post_meta( $post->ID, '_topranker_twitter_description', true ),
			'schema'                => get_post_meta( $post->ID, '_topranker_schema', true ),
			'seo_score'             => get_post_meta( $post->ID, '_topranker_seo_score', true ),
			'last_optimized'        => get_post_meta( $post->ID, '_topranker_last_optimized', true ),
			'content_hash'          => get_post_meta( $post->ID, '_topranker_content_hash', true ),
			'optimization_history'  => get_post_meta( $post->ID, '_topranker_optimization_history', true ),
		);
	}

	/**
	 * Store the timestamp of the last optimization.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post Post ID or WP_Post object.
	 * @return bool True on success.
	 */
	public function store_last_optimized( $post ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return false;
		}

		return (bool) update_post_meta( $post->ID, '_topranker_last_optimized', time() );
	}

	/**
	 * Extract images from post content.
	 *
	 * Returns an array of image data including IDs and URLs.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post Post ID or WP_Post object.
	 * @return array Array of image data.
	 */
	public function get_post_images( $post ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return array();
		}

		$images  = array();
		$content = $post->post_content;

		// Find all img tags.
		if ( preg_match_all( '/<img[^>]+>/i', $content, $matches ) ) {
			foreach ( $matches[0] as $img_tag ) {
				$image = array(
					'tag'     => $img_tag,
					'src'     => '',
					'alt'     => '',
					'id'      => 0,
					'caption' => '',
				);

				// Extract src.
				if ( preg_match( '/src=["\']([^"\']+)["\']/', $img_tag, $src_match ) ) {
					$image['src'] = $src_match[1];
				}

				// Extract alt.
				if ( preg_match( '/alt=["\']([^"\']*)["\']/', $img_tag, $alt_match ) ) {
					$image['alt'] = $alt_match[1];
				}

				// Extract class to find wp-image-{id}.
				if ( preg_match( '/class=["\'][^"\']*wp-image-(\d+)[^"\']*["\']/', $img_tag, $class_match ) ) {
					$image['id'] = (int) $class_match[1];
				}

				// Try to get attachment ID from URL if not found in class.
				if ( 0 === $image['id'] && ! empty( $image['src'] ) ) {
					$image['id'] = attachment_url_to_postid( $image['src'] );
				}

				// Get caption if available.
				if ( $image['id'] > 0 ) {
					$attachment   = get_post( $image['id'] );
					if ( $attachment ) {
						$image['caption'] = $attachment->post_excerpt;
					}
				}

				$images[] = $image;
			}
		}

		// Also include featured image.
		$featured_id = get_post_thumbnail_id( $post->ID );
		if ( $featured_id ) {
			$featured_url  = wp_get_attachment_url( $featured_id );
			$featured_alt  = get_post_meta( $featured_id, '_wp_attachment_image_alt', true );
			$featured_post = get_post( $featured_id );

			$images[] = array(
				'tag'        => '',
				'src'        => $featured_url,
				'alt'        => $featured_alt,
				'id'         => $featured_id,
				'caption'    => $featured_post ? $featured_post->post_excerpt : '',
				'is_featured' => true,
			);
		}

		return $images;
	}

	/**
	 * Get the first paragraph of content.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post Post ID or WP_Post object.
	 * @return string First paragraph.
	 */
	public function get_first_paragraph( $post ) {
		$content = $this->prepare_content( $post );

		if ( empty( $content ) ) {
			return '';
		}

		// Split by double newlines.
		$paragraphs = preg_split( '/\n\s*\n/', $content );

		if ( ! empty( $paragraphs ) ) {
			return trim( $paragraphs[0] );
		}

		return '';
	}

	/**
	 * Extract headings from post content.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post Post ID or WP_Post object.
	 * @return array Array of headings with level and text.
	 */
	public function get_headings( $post ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return array();
		}

		$headings = array();
		$content  = $post->post_content;

		// Find all heading tags.
		if ( preg_match_all( '/<h([1-6])[^>]*>(.*?)<\/h\1>/is', $content, $matches, PREG_SET_ORDER ) ) {
			foreach ( $matches as $match ) {
				$headings[] = array(
					'level' => (int) $match[1],
					'text'  => wp_strip_all_tags( $match[2] ),
				);
			}
		}

		return $headings;
	}
}
