<?php
/**
 * Post List modifications.
 *
 * Adds SEO status column and row actions to post list tables for enabled post types.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * TopRanker Post List class.
 *
 * @since 1.0.0
 */
class TopRanker_Post_List {

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		$this->init_hooks();
	}

	/**
	 * Initialize hooks.
	 *
	 * @since 1.0.0
	 */
	private function init_hooks() {
		// Add columns and row actions for each enabled post type.
		add_action( 'admin_init', array( $this, 'register_column_hooks' ) );

		// Make the column sortable.
		add_filter( 'request', array( $this, 'sort_by_seo_status' ) );

		// Enqueue admin styles for post list.
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_post_list_styles' ) );
	}

	/**
	 * Register column hooks for each enabled post type.
	 *
	 * @since 1.0.0
	 */
	public function register_column_hooks() {
		$enabled_post_types = get_option( 'topranker_post_types', array( 'post', 'page' ) );

		if ( empty( $enabled_post_types ) || ! is_array( $enabled_post_types ) ) {
			$enabled_post_types = array( 'post', 'page' );
		}

		foreach ( $enabled_post_types as $post_type ) {
			// Add custom column.
			if ( 'page' === $post_type ) {
				add_filter( 'manage_pages_columns', array( $this, 'add_seo_status_column' ) );
				add_action( 'manage_pages_custom_column', array( $this, 'render_seo_status_column' ), 10, 2 );
				add_filter( 'manage_edit-page_sortable_columns', array( $this, 'make_column_sortable' ) );
			} else {
				add_filter( "manage_{$post_type}_posts_columns", array( $this, 'add_seo_status_column' ) );
				add_action( "manage_{$post_type}_posts_custom_column", array( $this, 'render_seo_status_column' ), 10, 2 );
				add_filter( "manage_edit-{$post_type}_sortable_columns", array( $this, 'make_column_sortable' ) );
			}

			// Add row actions.
			if ( 'page' === $post_type ) {
				add_filter( 'page_row_actions', array( $this, 'add_optimize_row_action' ), 10, 2 );
			} else {
				add_filter( 'post_row_actions', array( $this, 'add_optimize_row_action' ), 10, 2 );
			}
		}
	}

	/**
	 * Add SEO Status column to the posts list.
	 *
	 * @since  1.0.0
	 * @param  array $columns Existing columns.
	 * @return array Modified columns.
	 */
	public function add_seo_status_column( $columns ) {
		$new_columns = array();

		foreach ( $columns as $key => $value ) {
			$new_columns[ $key ] = $value;

			// Insert after title column.
			if ( 'title' === $key ) {
				$new_columns['topranker_seo_status'] = __( 'SEO Status', 'topranker-ai' );

				// Add SEO Score column for Pro users.
				if ( topranker_is_pro() && class_exists( 'TopRanker_SEO_Audit' ) ) {
					$new_columns['topranker_seo_score'] = __( 'SEO Score', 'topranker-ai' );
				}
			}
		}

		return $new_columns;
	}

	/**
	 * Render the SEO Status column content.
	 *
	 * @since 1.0.0
	 * @param string $column_name The column name.
	 * @param int    $post_id     The post ID.
	 */
	public function render_seo_status_column( $column_name, $post_id ) {
		if ( 'topranker_seo_status' === $column_name ) {
			$status = $this->get_seo_status( $post_id );
			$this->output_status_indicator( $status );
			return;
		}

		// Handle SEO Score column for Pro users.
		if ( 'topranker_seo_score' === $column_name ) {
			if ( topranker_is_pro() && class_exists( 'TopRanker_SEO_Audit' ) ) {
				$audit = new TopRanker_SEO_Audit();
				$audit->render_score_column( $post_id );
			}
			return;
		}
	}

	/**
	 * Get the SEO status for a post.
	 *
	 * @since  1.0.0
	 * @param  int $post_id The post ID.
	 * @return array SEO status data.
	 */
	public function get_seo_status( $post_id ) {
		$meta_title       = get_post_meta( $post_id, '_topranker_meta_title', true );
		$meta_description = get_post_meta( $post_id, '_topranker_meta_description', true );
		$focus_keyphrase  = get_post_meta( $post_id, '_topranker_focus_keyphrase', true );
		$og_title         = get_post_meta( $post_id, '_topranker_og_title', true );
		$og_description   = get_post_meta( $post_id, '_topranker_og_description', true );

		// Count how many fields are filled.
		$filled = 0;
		$total  = 5; // meta_title, meta_description, focus_keyphrase, og_title, og_description.

		if ( ! empty( $meta_title ) ) {
			++$filled;
		}
		if ( ! empty( $meta_description ) ) {
			++$filled;
		}
		if ( ! empty( $focus_keyphrase ) ) {
			++$filled;
		}
		if ( ! empty( $og_title ) ) {
			++$filled;
		}
		if ( ! empty( $og_description ) ) {
			++$filled;
		}

		// Determine status level.
		if ( 0 === $filled ) {
			$level = 'missing';
			$label = __( 'Missing', 'topranker-ai' );
		} elseif ( $filled === $total ) {
			$level = 'complete';
			$label = __( 'Complete', 'topranker-ai' );
		} else {
			$level = 'partial';
			$label = sprintf(
				/* translators: 1: Number of filled fields, 2: Total fields */
				__( '%1$d/%2$d', 'topranker-ai' ),
				$filled,
				$total
			);
		}

		return array(
			'level'            => $level,
			'label'            => $label,
			'filled'           => $filled,
			'total'            => $total,
			'has_meta_title'   => ! empty( $meta_title ),
			'has_meta_desc'    => ! empty( $meta_description ),
			'has_keyphrase'    => ! empty( $focus_keyphrase ),
			'has_og_title'     => ! empty( $og_title ),
			'has_og_desc'      => ! empty( $og_description ),
		);
	}

	/**
	 * Output the status indicator HTML.
	 *
	 * @since 1.0.0
	 * @param array $status The SEO status data.
	 */
	private function output_status_indicator( $status ) {
		$class = 'topranker-seo-status is-' . esc_attr( $status['level'] );

		// Build tooltip/title text.
		$title_parts = array();

		if ( $status['has_meta_title'] ) {
			$title_parts[] = __( 'Meta title: Yes', 'topranker-ai' );
		} else {
			$title_parts[] = __( 'Meta title: No', 'topranker-ai' );
		}

		if ( $status['has_meta_desc'] ) {
			$title_parts[] = __( 'Meta description: Yes', 'topranker-ai' );
		} else {
			$title_parts[] = __( 'Meta description: No', 'topranker-ai' );
		}

		if ( $status['has_keyphrase'] ) {
			$title_parts[] = __( 'Focus keyphrase: Yes', 'topranker-ai' );
		} else {
			$title_parts[] = __( 'Focus keyphrase: No', 'topranker-ai' );
		}

		if ( $status['has_og_title'] ) {
			$title_parts[] = __( 'OG title: Yes', 'topranker-ai' );
		} else {
			$title_parts[] = __( 'OG title: No', 'topranker-ai' );
		}

		if ( $status['has_og_desc'] ) {
			$title_parts[] = __( 'OG description: Yes', 'topranker-ai' );
		} else {
			$title_parts[] = __( 'OG description: No', 'topranker-ai' );
		}

		$title = implode( "\n", $title_parts );

		printf(
			'<span class="%s" title="%s" role="img" aria-label="%s">%s</span>',
			esc_attr( $class ),
			esc_attr( $title ),
			esc_attr( $this->get_aria_label( $status ) ),
			esc_html( $status['label'] )
		);
	}

	/**
	 * Get aria-label for the status indicator.
	 *
	 * @since  1.0.0
	 * @param  array $status The SEO status data.
	 * @return string ARIA label.
	 */
	private function get_aria_label( $status ) {
		switch ( $status['level'] ) {
			case 'complete':
				return __( 'SEO status: Complete. All SEO fields are filled.', 'topranker-ai' );
			case 'partial':
				return sprintf(
					/* translators: 1: Number of filled fields, 2: Total fields */
					__( 'SEO status: Partial. %1$d of %2$d SEO fields are filled.', 'topranker-ai' ),
					$status['filled'],
					$status['total']
				);
			case 'missing':
			default:
				return __( 'SEO status: Missing. No SEO fields are filled.', 'topranker-ai' );
		}
	}

	/**
	 * Make the SEO Status column sortable.
	 *
	 * @since  1.0.0
	 * @param  array $columns Sortable columns.
	 * @return array Modified sortable columns.
	 */
	public function make_column_sortable( $columns ) {
		$columns['topranker_seo_status'] = 'topranker_seo_status';

		// Add SEO Score sortable for Pro users.
		if ( topranker_is_pro() && class_exists( 'TopRanker_SEO_Audit' ) ) {
			$columns['topranker_seo_score'] = 'topranker_seo_score';
		}

		return $columns;
	}

	/**
	 * Handle sorting by SEO status and SEO score.
	 *
	 * @since  1.0.0
	 * @param  array $vars Query vars.
	 * @return array Modified query vars.
	 */
	public function sort_by_seo_status( $vars ) {
		if ( ! is_admin() ) {
			return $vars;
		}

		if ( isset( $vars['orderby'] ) && 'topranker_seo_status' === $vars['orderby'] ) {
			// Sort by meta_title presence (posts with meta come first or last based on order).
			$vars = array_merge(
				$vars,
				array(
					'meta_key' => '_topranker_meta_title',
					'orderby'  => 'meta_value',
				)
			);

			// Use meta_query to include posts without the meta key.
			$vars['meta_query'] = array(
				'relation' => 'OR',
				array(
					'key'     => '_topranker_meta_title',
					'compare' => 'EXISTS',
				),
				array(
					'key'     => '_topranker_meta_title',
					'compare' => 'NOT EXISTS',
				),
			);
		}

		// Handle SEO Score sorting (Pro).
		if ( isset( $vars['orderby'] ) && 'topranker_seo_score' === $vars['orderby'] ) {
			$vars = array_merge(
				$vars,
				array(
					'meta_key' => '_topranker_seo_score',
					'orderby'  => 'meta_value_num',
				)
			);

			// Use meta_query to include posts without the meta key.
			$vars['meta_query'] = array(
				'relation' => 'OR',
				array(
					'key'     => '_topranker_seo_score',
					'compare' => 'EXISTS',
				),
				array(
					'key'     => '_topranker_seo_score',
					'compare' => 'NOT EXISTS',
				),
			);
		}

		return $vars;
	}

	/**
	 * Add "Optimize" link to row actions.
	 *
	 * @since  1.0.0
	 * @param  array   $actions Existing row actions.
	 * @param  WP_Post $post    The post object.
	 * @return array   Modified row actions.
	 */
	public function add_optimize_row_action( $actions, $post ) {
		// Check if user can edit this post.
		if ( ! current_user_can( 'edit_post', $post->ID ) ) {
			return $actions;
		}

		// Check if this post type is enabled.
		$enabled_post_types = get_option( 'topranker_post_types', array( 'post', 'page' ) );
		if ( ! in_array( $post->post_type, $enabled_post_types, true ) ) {
			return $actions;
		}

		// Build the edit URL with the optimize trigger parameter.
		$optimize_url = add_query_arg(
			array(
				'topranker_optimize' => '1',
			),
			get_edit_post_link( $post->ID, 'raw' )
		);

		// Add the Optimize link.
		$actions['topranker_optimize'] = sprintf(
			'<a href="%s" class="topranker-optimize-link" aria-label="%s">%s</a>',
			esc_url( $optimize_url ),
			/* translators: %s: Post title */
			esc_attr( sprintf( __( 'Optimize "%s" with TopRanker AI', 'topranker-ai' ), $post->post_title ) ),
			esc_html__( 'Optimize', 'topranker-ai' )
		);

		return $actions;
	}

	/**
	 * Enqueue admin styles for post list.
	 *
	 * @since 1.0.0
	 * @param string $hook_suffix The current admin page hook suffix.
	 */
	public function enqueue_post_list_styles( $hook_suffix ) {
		// Only on post list pages.
		if ( 'edit.php' !== $hook_suffix ) {
			return;
		}

		// Check if we're on an enabled post type.
		$screen = get_current_screen();
		if ( ! $screen || ! isset( $screen->post_type ) ) {
			return;
		}

		$enabled_post_types = get_option( 'topranker_post_types', array( 'post', 'page' ) );
		if ( ! in_array( $screen->post_type, $enabled_post_types, true ) ) {
			return;
		}

		// Enqueue admin CSS.
		wp_enqueue_style(
			'topranker-admin',
			TOPRANKER_URL . 'admin/css/topranker-admin.css',
			array(),
			TOPRANKER_VERSION
		);

		// Add inline CSS for column width.
		$custom_css = '
			.column-topranker_seo_status {
				width: 100px;
			}
			.column-topranker_seo_score {
				width: 90px;
			}
			@media screen and (max-width: 782px) {
				.column-topranker_seo_status,
				.column-topranker_seo_score {
					display: none !important;
				}
			}
		';
		wp_add_inline_style( 'topranker-admin', $custom_css );
	}
}
