<?php
/**
 * REST API endpoints.
 *
 * Registers and handles all REST API endpoints for the plugin.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * TopRanker REST API class.
 *
 * @since 1.0.0
 */
class TopRanker_REST_API {

	/**
	 * REST API namespace.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	const NAMESPACE = 'topranker/v1';

	/**
	 * OpenAI API endpoint.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	const OPENAI_API_URL = 'https://api.openai.com/v1/chat/completions';

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		$this->init_hooks();
	}

	/**
	 * Initialize hooks.
	 *
	 * @since 1.0.0
	 */
	private function init_hooks() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );
	}

	/**
	 * Register REST API routes.
	 *
	 * @since 1.0.0
	 */
	public function register_routes() {
		// Test API key endpoint.
		register_rest_route(
			self::NAMESPACE,
			'/test-api-key',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'test_api_key' ),
				'permission_callback' => array( $this, 'check_admin_permission' ),
				'args'                => array(
					'api_key' => array(
						'required'          => true,
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
						'validate_callback' => function( $param ) {
							return ! empty( $param );
						},
					),
				),
			)
		);

		// Optimize endpoint - trigger optimization for a post.
		register_rest_route(
			self::NAMESPACE,
			'/optimize',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'optimize_post' ),
				'permission_callback' => array( $this, 'check_edit_permission' ),
				'args'                => array(
					'post_id'       => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
						'validate_callback' => function( $param ) {
							return $param > 0;
						},
					),
					'force_refresh' => array(
						'required'          => false,
						'type'              => 'boolean',
						'default'           => false,
						'sanitize_callback' => 'rest_sanitize_boolean',
					),
				),
			)
		);

		// Suggestions endpoint - get cached suggestions for a post.
		register_rest_route(
			self::NAMESPACE,
			'/suggestions/(?P<post_id>\d+)',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_suggestions' ),
				'permission_callback' => array( $this, 'check_edit_permission' ),
				'args'                => array(
					'post_id' => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
						'validate_callback' => function( $param ) {
							return $param > 0;
						},
					),
				),
			)
		);

		// Apply endpoint - save selected suggestions to post meta.
		register_rest_route(
			self::NAMESPACE,
			'/apply',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'apply_suggestions' ),
				'permission_callback' => array( $this, 'check_edit_permission' ),
				'args'                => array(
					'post_id'              => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
						'validate_callback' => function( $param ) {
							return $param > 0;
						},
					),
					'meta_title'           => array(
						'required'          => false,
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
					),
					'meta_description'     => array(
						'required'          => false,
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_textarea_field',
					),
					'excerpt'              => array(
						'required'          => false,
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_textarea_field',
					),
					'focus_keyphrase'      => array(
						'required'          => false,
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
					),
					'secondary_keyphrases' => array(
						'required'          => false,
						'type'              => 'array',
						'sanitize_callback' => array( $this, 'sanitize_keyphrases_array' ),
					),
					'og_title'             => array(
						'required'          => false,
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
					),
					'og_description'       => array(
						'required'          => false,
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_textarea_field',
					),
					'twitter_title'        => array(
						'required'          => false,
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
					),
					'twitter_description'  => array(
						'required'          => false,
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_textarea_field',
					),
				),
			)
		);

		// Usage endpoint - get current usage stats.
		register_rest_route(
			self::NAMESPACE,
			'/usage',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_usage' ),
				'permission_callback' => array( $this, 'check_edit_permission' ),
			)
		);

		// Dashboard stats endpoint - get SEO coverage stats.
		register_rest_route(
			self::NAMESPACE,
			'/dashboard-stats',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_dashboard_stats' ),
				'permission_callback' => array( $this, 'check_edit_permission' ),
				'args'                => array(
					'refresh' => array(
						'required'          => false,
						'type'              => 'boolean',
						'default'           => false,
						'sanitize_callback' => 'rest_sanitize_boolean',
					),
				),
			)
		);

		// Copy to SEO plugin endpoint - copies TopRanker data to detected SEO plugin.
		register_rest_route(
			self::NAMESPACE,
			'/copy-to-seo-plugin',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'copy_to_seo_plugin' ),
				'permission_callback' => array( $this, 'check_edit_permission' ),
				'args'                => array(
					'post_id' => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
						'validate_callback' => function( $param ) {
							return $param > 0;
						},
					),
				),
			)
		);

		// SEO compatibility status endpoint.
		register_rest_route(
			self::NAMESPACE,
			'/seo-compat-status/(?P<post_id>\d+)',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_seo_compat_status' ),
				'permission_callback' => array( $this, 'check_edit_permission' ),
				'args'                => array(
					'post_id' => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
						'validate_callback' => function( $param ) {
							return $param > 0;
						},
					),
				),
			)
		);

		// Pro: Generate alt tags for images in a post.
		register_rest_route(
			self::NAMESPACE,
			'/generate-alt-tags',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'generate_alt_tags' ),
				'permission_callback' => array( $this, 'check_edit_permission' ),
				'args'                => array(
					'post_id' => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
						'validate_callback' => function( $param ) {
							return $param > 0;
						},
					),
				),
			)
		);

		// Pro: Apply generated alt tags.
		register_rest_route(
			self::NAMESPACE,
			'/apply-alt-tags',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'apply_alt_tags' ),
				'permission_callback' => array( $this, 'check_edit_permission' ),
				'args'                => array(
					'results' => array(
						'required'          => true,
						'type'              => 'array',
						'validate_callback' => function( $param ) {
							return is_array( $param ) && ! empty( $param );
						},
					),
				),
			)
		);

		// Pro: Generate alt tag for a single attachment (Media Library).
		register_rest_route(
			self::NAMESPACE,
			'/generate-attachment-alt',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'generate_attachment_alt' ),
				'permission_callback' => array( $this, 'check_edit_permission' ),
				'args'                => array(
					'attachment_id' => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
						'validate_callback' => function( $param ) {
							return $param > 0;
						},
					),
				),
			)
		);

		// Pro: Get SEO audit/score for a post.
		register_rest_route(
			self::NAMESPACE,
			'/seo-audit/(?P<post_id>\d+)',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_seo_audit' ),
				'permission_callback' => array( $this, 'check_edit_permission' ),
				'args'                => array(
					'post_id' => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
						'validate_callback' => function( $param ) {
							return $param > 0;
						},
					),
				),
			)
		);

		// Pro: Get optimization history for a post.
		register_rest_route(
			self::NAMESPACE,
			'/history/(?P<post_id>\d+)',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_history' ),
				'permission_callback' => array( $this, 'check_edit_permission' ),
				'args'                => array(
					'post_id' => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
						'validate_callback' => function( $param ) {
							return $param > 0;
						},
					),
				),
			)
		);

		// Pro: Get diff between current and history entry.
		register_rest_route(
			self::NAMESPACE,
			'/history/(?P<post_id>\d+)/diff/(?P<index>\d+)',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_history_diff' ),
				'permission_callback' => array( $this, 'check_edit_permission' ),
				'args'                => array(
					'post_id' => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
						'validate_callback' => function( $param ) {
							return $param > 0;
						},
					),
					'index'   => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
					),
				),
			)
		);

		// Pro: Revert to a history entry.
		register_rest_route(
			self::NAMESPACE,
			'/history/revert',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'revert_to_history' ),
				'permission_callback' => array( $this, 'check_edit_permission' ),
				'args'                => array(
					'post_id' => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
						'validate_callback' => function( $param ) {
							return $param > 0;
						},
					),
					'index'   => array(
						'required'          => true,
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
					),
				),
			)
		);
	}

	/**
	 * Check if user has admin permission.
	 *
	 * @since  1.0.0
	 * @return bool True if user can manage options.
	 */
	public function check_admin_permission() {
		return current_user_can( 'manage_options' );
	}

	/**
	 * Check if user has edit permission.
	 *
	 * @since  1.0.0
	 * @return bool True if user can edit posts.
	 */
	public function check_edit_permission() {
		return current_user_can( 'edit_posts' );
	}

	/**
	 * Check if user can edit a specific post.
	 *
	 * @since  1.0.0
	 * @param  int $post_id Post ID.
	 * @return bool True if user can edit the post.
	 */
	public function can_edit_post( $post_id ) {
		$post = get_post( $post_id );

		if ( ! $post ) {
			return false;
		}

		return current_user_can( 'edit_post', $post_id );
	}

	/**
	 * Sanitize an array of keyphrases.
	 *
	 * @since  1.0.0
	 * @param  array $keyphrases Array of keyphrase strings.
	 * @return array Sanitized array of keyphrases.
	 */
	public function sanitize_keyphrases_array( $keyphrases ) {
		if ( ! is_array( $keyphrases ) ) {
			return array();
		}

		return array_map( 'sanitize_text_field', $keyphrases );
	}

	/**
	 * Optimize a post.
	 *
	 * Triggers the optimization process for a single post.
	 *
	 * @since  1.0.0
	 * @param  WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response|WP_Error Response object or WP_Error.
	 */
	public function optimize_post( $request ) {
		$post_id       = $request->get_param( 'post_id' );
		$force_refresh = $request->get_param( 'force_refresh' );

		// Verify user can edit this specific post.
		if ( ! $this->can_edit_post( $post_id ) ) {
			return new WP_Error(
				'forbidden',
				__( 'You do not have permission to edit this post.', 'topranker-ai' ),
				array( 'status' => 403 )
			);
		}

		$post = get_post( $post_id );

		if ( ! $post ) {
			return new WP_Error(
				'not_found',
				__( 'Post not found.', 'topranker-ai' ),
				array( 'status' => 404 )
			);
		}

		// Check if post type is enabled.
		if ( ! $this->is_post_type_enabled( $post->post_type ) ) {
			return new WP_Error(
				'post_type_disabled',
				__( 'TopRanker AI is not enabled for this post type.', 'topranker-ai' ),
				array( 'status' => 400 )
			);
		}

		// Run the optimization.
		$optimizer = new TopRanker_Optimizer();
		$result    = $optimizer->optimize( $post, $force_refresh );

		if ( is_wp_error( $result ) ) {
			$status = 400;

			// Map specific errors to status codes.
			$error_code = $result->get_error_code();
			if ( 'usage_limit_reached' === $error_code ) {
				$status = 429;
			} elseif ( 'invalid_post' === $error_code ) {
				$status = 404;
			}

			return new WP_Error(
				$error_code,
				$result->get_error_message(),
				array( 'status' => $status )
			);
		}

		$this->log_debug( 'Optimization completed for post ' . $post_id );

		return rest_ensure_response( $result );
	}

	/**
	 * Get suggestions for a post.
	 *
	 * Returns cached suggestions and current saved values.
	 *
	 * @since  1.0.0
	 * @param  WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response|WP_Error Response object or WP_Error.
	 */
	public function get_suggestions( $request ) {
		$post_id = $request->get_param( 'post_id' );

		// Verify user can edit this specific post.
		if ( ! $this->can_edit_post( $post_id ) ) {
			return new WP_Error(
				'forbidden',
				__( 'You do not have permission to view this post.', 'topranker-ai' ),
				array( 'status' => 403 )
			);
		}

		$post = get_post( $post_id );

		if ( ! $post ) {
			return new WP_Error(
				'not_found',
				__( 'Post not found.', 'topranker-ai' ),
				array( 'status' => 404 )
			);
		}

		$optimizer = new TopRanker_Optimizer();
		$data      = $optimizer->get_suggestions_for_editor( $post_id );

		// Add usage stats.
		$usage         = new TopRanker_Usage();
		$data['usage'] = $usage->get_api_response();

		// Add auto-optimize status.
		if ( class_exists( 'TopRanker_Cron' ) ) {
			$cron                       = new TopRanker_Cron();
			$data['is_auto_optimizing'] = $cron->is_auto_optimizing( $post_id );
		} else {
			$data['is_auto_optimizing'] = false;
		}

		return rest_ensure_response( $data );
	}

	/**
	 * Apply selected suggestions to a post.
	 *
	 * Saves the selected optimization values to post meta.
	 *
	 * @since  1.0.0
	 * @param  WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response|WP_Error Response object or WP_Error.
	 */
	public function apply_suggestions( $request ) {
		$post_id = $request->get_param( 'post_id' );

		// Verify user can edit this specific post.
		if ( ! $this->can_edit_post( $post_id ) ) {
			return new WP_Error(
				'forbidden',
				__( 'You do not have permission to edit this post.', 'topranker-ai' ),
				array( 'status' => 403 )
			);
		}

		$post = get_post( $post_id );

		if ( ! $post ) {
			return new WP_Error(
				'not_found',
				__( 'Post not found.', 'topranker-ai' ),
				array( 'status' => 404 )
			);
		}

		// Collect selected values from request.
		$selected = array();

		$fields = array(
			'meta_title',
			'meta_description',
			'excerpt',
			'focus_keyphrase',
			'secondary_keyphrases',
			'og_title',
			'og_description',
			'twitter_title',
			'twitter_description',
		);

		foreach ( $fields as $field ) {
			$value = $request->get_param( $field );
			if ( null !== $value && '' !== $value ) {
				$selected[ $field ] = $value;
			}
		}

		if ( empty( $selected ) ) {
			return new WP_Error(
				'no_values',
				__( 'No values provided to apply.', 'topranker-ai' ),
				array( 'status' => 400 )
			);
		}

		// Apply the selected values.
		$optimizer = new TopRanker_Optimizer();
		$result    = $optimizer->apply( $post_id, $selected );

		if ( is_wp_error( $result ) ) {
			return new WP_Error(
				$result->get_error_code(),
				$result->get_error_message(),
				array( 'status' => 400 )
			);
		}

		$this->log_debug( 'Applied suggestions to post ' . $post_id );

		return rest_ensure_response( $result );
	}

	/**
	 * Get usage statistics.
	 *
	 * Returns the current usage count and limits.
	 *
	 * @since  1.0.0
	 * @param  WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response Response object.
	 */
	public function get_usage( $request ) {
		$usage = new TopRanker_Usage();

		return rest_ensure_response( $usage->get_api_response() );
	}

	/**
	 * Get dashboard statistics.
	 *
	 * Returns SEO coverage statistics for the dashboard.
	 *
	 * @since  1.0.0
	 * @param  WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response Response object.
	 */
	public function get_dashboard_stats( $request ) {
		$refresh = $request->get_param( 'refresh' );

		// Check for cached stats.
		$cache_key = 'topranker_dashboard_stats';
		$stats     = get_transient( $cache_key );

		if ( false === $stats || $refresh ) {
			$stats = $this->calculate_dashboard_stats();
			set_transient( $cache_key, $stats, HOUR_IN_SECONDS );
		}

		return rest_ensure_response( $stats );
	}

	/**
	 * Calculate dashboard statistics.
	 *
	 * Uses efficient COUNT queries to calculate SEO coverage.
	 *
	 * @since  1.0.0
	 * @return array Dashboard statistics.
	 */
	private function calculate_dashboard_stats() {
		global $wpdb;

		// Get enabled post types.
		$enabled_post_types = get_option( 'topranker_post_types', array( 'post', 'page' ) );

		if ( empty( $enabled_post_types ) ) {
			$enabled_post_types = array( 'post', 'page' );
		}

		// Prepare placeholders for post types.
		$placeholders = implode( ', ', array_fill( 0, count( $enabled_post_types ), '%s' ) );

		// Total published posts.
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $placeholders is safely generated.
		$total_posts = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_status = 'publish' AND post_type IN ({$placeholders})",
				...$enabled_post_types
			)
		);

		// Posts with meta titles.
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $placeholders is safely generated.
		$with_meta_title = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(DISTINCT p.ID) FROM {$wpdb->posts} p
				INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
				WHERE p.post_status = 'publish'
				AND p.post_type IN ({$placeholders})
				AND pm.meta_key = '_topranker_meta_title'
				AND pm.meta_value != ''",
				...$enabled_post_types
			)
		);

		// Posts with meta descriptions.
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $placeholders is safely generated.
		$with_meta_description = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(DISTINCT p.ID) FROM {$wpdb->posts} p
				INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
				WHERE p.post_status = 'publish'
				AND p.post_type IN ({$placeholders})
				AND pm.meta_key = '_topranker_meta_description'
				AND pm.meta_value != ''",
				...$enabled_post_types
			)
		);

		// Posts with focus keyphrase.
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- $placeholders is safely generated.
		$with_keyphrase = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(DISTINCT p.ID) FROM {$wpdb->posts} p
				INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
				WHERE p.post_status = 'publish'
				AND p.post_type IN ({$placeholders})
				AND pm.meta_key = '_topranker_focus_keyphrase'
				AND pm.meta_value != ''",
				...$enabled_post_types
			)
		);

		// Calculate percentages.
		$meta_title_percent       = $total_posts > 0 ? round( ( $with_meta_title / $total_posts ) * 100 ) : 0;
		$meta_description_percent = $total_posts > 0 ? round( ( $with_meta_description / $total_posts ) * 100 ) : 0;
		$keyphrase_percent        = $total_posts > 0 ? round( ( $with_keyphrase / $total_posts ) * 100 ) : 0;

		// Overall coverage (average of all metrics).
		$overall_coverage = ( $meta_title_percent + $meta_description_percent + $keyphrase_percent ) / 3;
		$overall_coverage = round( $overall_coverage );

		// Calculate missing counts.
		$missing_meta_title       = $total_posts - $with_meta_title;
		$missing_meta_description = $total_posts - $with_meta_description;
		$missing_keyphrase        = $total_posts - $with_keyphrase;

		return array(
			'total_posts'              => $total_posts,
			'meta_title'               => array(
				'count'   => $with_meta_title,
				'missing' => $missing_meta_title,
				'percent' => $meta_title_percent,
			),
			'meta_description'         => array(
				'count'   => $with_meta_description,
				'missing' => $missing_meta_description,
				'percent' => $meta_description_percent,
			),
			'keyphrase'                => array(
				'count'   => $with_keyphrase,
				'missing' => $missing_keyphrase,
				'percent' => $keyphrase_percent,
			),
			'overall_coverage'         => $overall_coverage,
			'post_types'               => $enabled_post_types,
			'last_updated'             => time(),
			'is_pro'                   => topranker_is_pro(),
		);
	}

	/**
	 * Check if a post type is enabled for TopRanker.
	 *
	 * @since  1.0.0
	 * @param  string $post_type Post type slug.
	 * @return bool True if post type is enabled.
	 */
	private function is_post_type_enabled( $post_type ) {
		$enabled_post_types = get_option( 'topranker_post_types', array( 'post', 'page' ) );

		if ( empty( $enabled_post_types ) ) {
			$enabled_post_types = array( 'post', 'page' );
		}

		return in_array( $post_type, $enabled_post_types, true );
	}

	/**
	 * Test OpenAI API key.
	 *
	 * Makes a minimal request to OpenAI to verify the API key is valid.
	 *
	 * @since  1.0.0
	 * @param  WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response|WP_Error Response object on success, or WP_Error on failure.
	 */
	public function test_api_key( $request ) {
		$api_key = $request->get_param( 'api_key' );

		// Validate API key format (should start with sk-).
		if ( strpos( $api_key, 'sk-' ) !== 0 ) {
			return new WP_Error(
				'invalid_api_key',
				__( 'Invalid API key format. OpenAI API keys start with "sk-".', 'topranker-ai' ),
				array( 'status' => 400 )
			);
		}

		// Make a minimal request to OpenAI.
		$response = $this->make_openai_test_request( $api_key );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		// Log success in debug mode.
		$this->log_debug( 'API key test successful.' );

		// Set transient for success notice on settings page.
		set_transient( 'topranker_api_test_success', true, 60 );

		// Clear the welcome notice since API key is now configured.
		delete_option( 'topranker_show_welcome' );

		return rest_ensure_response(
			array(
				'success' => true,
				'message' => __( 'Connection successful! Your API key is valid.', 'topranker-ai' ),
			)
		);
	}

	/**
	 * Make a minimal test request to OpenAI.
	 *
	 * @since  1.0.0
	 * @param  string $api_key The API key to test.
	 * @return true|WP_Error True on success, WP_Error on failure.
	 */
	private function make_openai_test_request( $api_key ) {
		$model = get_option( 'topranker_model', 'gpt-5.2' );

		$body = array(
			'model'           => $model,
			'messages'        => array(
				array(
					'role'    => 'user',
					'content' => 'Respond with JSON: {"status": "ok"}',
				),
			),
			'response_format' => array( 'type' => 'json_object' ),
		);

		// GPT-5.x models need reasoning_effort + max_completion_tokens.
		if ( 0 === strpos( $model, 'gpt-5' ) ) {
			$body['max_completion_tokens'] = 500;
			$body['reasoning_effort']      = 'none';
		} else {
			$body['max_tokens'] = 500;
		}

		$this->log_debug( 'Testing API key with model: ' . $model );

		$response = wp_remote_post(
			self::OPENAI_API_URL,
			array(
				'timeout' => 30,
				'headers' => array(
					'Authorization' => 'Bearer ' . $api_key,
					'Content-Type'  => 'application/json',
				),
				'body'    => wp_json_encode( $body ),
			)
		);

		// Check for network error.
		if ( is_wp_error( $response ) ) {
			$this->log_debug( 'Network error: ' . $response->get_error_message() );
			return new WP_Error(
				'network_error',
				__( 'Could not connect to OpenAI. Check your server\'s internet connection.', 'topranker-ai' ),
				array( 'status' => 503 )
			);
		}

		$response_code = wp_remote_retrieve_response_code( $response );
		$response_body = wp_remote_retrieve_body( $response );
		$body_data     = json_decode( $response_body, true );

		$this->log_debug( 'OpenAI response code: ' . $response_code );

		// Handle different response codes.
		switch ( $response_code ) {
			case 200:
				// Success.
				return true;

			case 401:
				return new WP_Error(
					'invalid_api_key',
					__( 'Your API key is invalid. Please check it in Settings.', 'topranker-ai' ),
					array( 'status' => 401 )
				);

			case 429:
				// Rate limit or quota exceeded.
				$error_message = isset( $body_data['error']['message'] ) ? $body_data['error']['message'] : '';
				if ( strpos( strtolower( $error_message ), 'quota' ) !== false ) {
					return new WP_Error(
						'no_credits',
						__( 'Your OpenAI account may have no credits. Check your OpenAI billing.', 'topranker-ai' ),
						array( 'status' => 429 )
					);
				}
				return new WP_Error(
					'rate_limit',
					__( 'OpenAI rate limit reached. Please wait a minute and try again.', 'topranker-ai' ),
					array( 'status' => 429 )
				);

			case 500:
			case 502:
			case 503:
				return new WP_Error(
					'server_error',
					__( 'OpenAI service is temporarily unavailable. Please try again later.', 'topranker-ai' ),
					array( 'status' => $response_code )
				);

			default:
				// Try to get error message from response.
				$error_message = __( 'Unexpected response from OpenAI. Please try again.', 'topranker-ai' );
				if ( isset( $body_data['error']['message'] ) ) {
					$error_message = sanitize_text_field( $body_data['error']['message'] );
				}
				$this->log_debug( 'Unexpected response: ' . $response_body );
				return new WP_Error(
					'api_error',
					$error_message,
					array( 'status' => $response_code )
				);
		}
	}

	/**
	 * Copy TopRanker data to detected SEO plugin.
	 *
	 * Used in "suggest only" mode when user clicks "Copy to [SEO Plugin]" button.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response|WP_Error Response object or WP_Error.
	 */
	public function copy_to_seo_plugin( $request ) {
		$post_id = $request->get_param( 'post_id' );

		// Verify user can edit this specific post.
		if ( ! $this->can_edit_post( $post_id ) ) {
			return new WP_Error(
				'forbidden',
				__( 'You do not have permission to edit this post.', 'topranker-ai' ),
				array( 'status' => 403 )
			);
		}

		$post = get_post( $post_id );

		if ( ! $post ) {
			return new WP_Error(
				'not_found',
				__( 'Post not found.', 'topranker-ai' ),
				array( 'status' => 404 )
			);
		}

		// Copy to SEO plugin.
		$optimizer = new TopRanker_Optimizer();
		$result    = $optimizer->copy_to_seo_plugin( $post_id );

		if ( is_wp_error( $result ) ) {
			return new WP_Error(
				$result->get_error_code(),
				$result->get_error_message(),
				array( 'status' => 400 )
			);
		}

		$this->log_debug( 'Copied TopRanker data to SEO plugin for post ' . $post_id );

		return rest_ensure_response( $result );
	}

	/**
	 * Get SEO compatibility status for a post.
	 *
	 * Returns information about detected SEO plugin, current mode,
	 * and which fields have data in both TopRanker and the SEO plugin.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response|WP_Error Response object or WP_Error.
	 */
	public function get_seo_compat_status( $request ) {
		$post_id = $request->get_param( 'post_id' );

		// Verify user can edit this specific post.
		if ( ! $this->can_edit_post( $post_id ) ) {
			return new WP_Error(
				'forbidden',
				__( 'You do not have permission to view this post.', 'topranker-ai' ),
				array( 'status' => 403 )
			);
		}

		$post = get_post( $post_id );

		if ( ! $post ) {
			return new WP_Error(
				'not_found',
				__( 'Post not found.', 'topranker-ai' ),
				array( 'status' => 404 )
			);
		}

		$seo_compat = new TopRanker_SEO_Compat();
		$status     = $seo_compat->get_compatibility_status( $post_id );

		return rest_ensure_response( $status );
	}

	/**
	 * Log debug message if debug mode is enabled.
	 *
	 * @since 1.0.0
	 * @param string $message Debug message.
	 */
	private function log_debug( $message ) {
		if ( defined( 'TOPRANKER_DEBUG' ) && TOPRANKER_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging.
			error_log( '[TopRanker AI] ' . $message );
		}
	}

	/**
	 * Generate alt tags for images in a post.
	 *
	 * Pro feature: Uses OpenAI Vision to generate SEO-optimized alt text.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response|WP_Error Response object or WP_Error.
	 */
	public function generate_alt_tags( $request ) {
		// Check if Pro is active.
		if ( ! topranker_is_pro() ) {
			return new WP_Error(
				'pro_required',
				__( 'Alt tag generation is a Pro feature. Please upgrade to unlock.', 'topranker-ai' ),
				array( 'status' => 403 )
			);
		}

		// Check if Pro class exists.
		if ( ! class_exists( 'TopRanker_Alt_Tags' ) ) {
			return new WP_Error(
				'feature_unavailable',
				__( 'Alt tag generation is not available.', 'topranker-ai' ),
				array( 'status' => 500 )
			);
		}

		$post_id = $request->get_param( 'post_id' );

		// Verify user can edit this specific post.
		if ( ! $this->can_edit_post( $post_id ) ) {
			return new WP_Error(
				'forbidden',
				__( 'You do not have permission to edit this post.', 'topranker-ai' ),
				array( 'status' => 403 )
			);
		}

		$post = get_post( $post_id );

		if ( ! $post ) {
			return new WP_Error(
				'not_found',
				__( 'Post not found.', 'topranker-ai' ),
				array( 'status' => 404 )
			);
		}

		// Get focus keyphrase if available.
		$focus_keyphrase = get_post_meta( $post_id, '_topranker_focus_keyphrase', true );

		// Generate alt tags.
		$alt_tags = new TopRanker_Alt_Tags();
		$result   = $alt_tags->generate_alt_tags( $post, $focus_keyphrase );

		if ( is_wp_error( $result ) ) {
			return new WP_Error(
				$result->get_error_code(),
				$result->get_error_message(),
				array( 'status' => 400 )
			);
		}

		$this->log_debug( 'Generated alt tags for post ' . $post_id . ': ' . $result['success'] . ' success, ' . $result['failed'] . ' failed' );

		return rest_ensure_response( $result );
	}

	/**
	 * Apply generated alt tags to attachments.
	 *
	 * Pro feature: Saves the generated alt text to attachment meta.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response|WP_Error Response object or WP_Error.
	 */
	public function apply_alt_tags( $request ) {
		// Check if Pro is active.
		if ( ! topranker_is_pro() ) {
			return new WP_Error(
				'pro_required',
				__( 'Alt tag generation is a Pro feature. Please upgrade to unlock.', 'topranker-ai' ),
				array( 'status' => 403 )
			);
		}

		// Check if Pro class exists.
		if ( ! class_exists( 'TopRanker_Alt_Tags' ) ) {
			return new WP_Error(
				'feature_unavailable',
				__( 'Alt tag generation is not available.', 'topranker-ai' ),
				array( 'status' => 500 )
			);
		}

		$results = $request->get_param( 'results' );

		if ( ! is_array( $results ) || empty( $results ) ) {
			return new WP_Error(
				'invalid_results',
				__( 'No valid alt tag results provided.', 'topranker-ai' ),
				array( 'status' => 400 )
			);
		}

		// Verify user can edit the attachments.
		foreach ( $results as $result ) {
			$image_id = isset( $result['image_id'] ) ? (int) $result['image_id'] : 0;
			if ( $image_id > 0 && ! current_user_can( 'edit_post', $image_id ) ) {
				return new WP_Error(
					'forbidden',
					__( 'You do not have permission to edit one or more of these images.', 'topranker-ai' ),
					array( 'status' => 403 )
				);
			}
		}

		// Get post ID if provided (to update alt in post content HTML).
		$post_id = $request->get_param( 'post_id' );
		$post_id = $post_id ? absint( $post_id ) : 0;

		// Apply alt tags.
		$alt_tags    = new TopRanker_Alt_Tags();
		$apply_result = $alt_tags->apply_alt_tags( $results, $post_id );

		$this->log_debug( 'Applied alt tags: ' . $apply_result['applied'] . ' applied, ' . $apply_result['failed'] . ' failed' );

		return rest_ensure_response( $apply_result );
	}

	/**
	 * Generate alt tag for a single attachment.
	 *
	 * Pro feature: Generates alt text for an image in the Media Library.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response|WP_Error Response object or WP_Error.
	 */
	public function generate_attachment_alt( $request ) {
		// Check if Pro is active.
		if ( ! topranker_is_pro() ) {
			return new WP_Error(
				'pro_required',
				__( 'Alt tag generation is a Pro feature. Please upgrade to unlock.', 'topranker-ai' ),
				array( 'status' => 403 )
			);
		}

		// Check if Pro class exists.
		if ( ! class_exists( 'TopRanker_Alt_Tags' ) ) {
			return new WP_Error(
				'feature_unavailable',
				__( 'Alt tag generation is not available.', 'topranker-ai' ),
				array( 'status' => 500 )
			);
		}

		$attachment_id = $request->get_param( 'attachment_id' );

		// Verify user can edit this attachment.
		if ( ! current_user_can( 'edit_post', $attachment_id ) ) {
			return new WP_Error(
				'forbidden',
				__( 'You do not have permission to edit this image.', 'topranker-ai' ),
				array( 'status' => 403 )
			);
		}

		// Generate alt tag.
		$alt_tags = new TopRanker_Alt_Tags();
		$result   = $alt_tags->generate_for_attachment( $attachment_id );

		if ( is_wp_error( $result ) ) {
			return new WP_Error(
				$result->get_error_code(),
				$result->get_error_message(),
				array( 'status' => 400 )
			);
		}

		$this->log_debug( 'Generated alt tag for attachment ' . $attachment_id );

		return rest_ensure_response( $result );
	}

	/**
	 * Get SEO audit/score data for a post.
	 *
	 * Pro feature: Returns detailed SEO analysis with score and improvement suggestions.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response|WP_Error Response object or WP_Error.
	 */
	public function get_seo_audit( $request ) {
		// Check if Pro is active.
		if ( ! topranker_is_pro() ) {
			return new WP_Error(
				'pro_required',
				__( 'SEO audit is a Pro feature. Please upgrade to unlock.', 'topranker-ai' ),
				array( 'status' => 403 )
			);
		}

		// Check if Pro class exists.
		if ( ! class_exists( 'TopRanker_SEO_Audit' ) ) {
			return new WP_Error(
				'feature_unavailable',
				__( 'SEO audit is not available.', 'topranker-ai' ),
				array( 'status' => 500 )
			);
		}

		$post_id = $request->get_param( 'post_id' );

		// Verify user can edit this specific post.
		if ( ! $this->can_edit_post( $post_id ) ) {
			return new WP_Error(
				'forbidden',
				__( 'You do not have permission to view this post.', 'topranker-ai' ),
				array( 'status' => 403 )
			);
		}

		$post = get_post( $post_id );

		if ( ! $post ) {
			return new WP_Error(
				'not_found',
				__( 'Post not found.', 'topranker-ai' ),
				array( 'status' => 404 )
			);
		}

		// Get SEO audit data.
		$audit = new TopRanker_SEO_Audit();
		$data  = $audit->get_api_data( $post );

		$data['post_id'] = $post_id;

		$this->log_debug( 'Got SEO audit for post ' . $post_id . ': score ' . $data['score'] );

		return rest_ensure_response( $data );
	}

	/**
	 * Get optimization history for a post.
	 *
	 * Pro feature: Returns the last 10 optimization versions for a post.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response|WP_Error Response object or WP_Error.
	 */
	public function get_history( $request ) {
		// Check if Pro is active.
		if ( ! topranker_is_pro() ) {
			return new WP_Error(
				'pro_required',
				__( 'Optimization history is a Pro feature. Please upgrade to unlock.', 'topranker-ai' ),
				array( 'status' => 403 )
			);
		}

		// Check if Pro class exists.
		if ( ! class_exists( 'TopRanker_History' ) ) {
			return new WP_Error(
				'feature_unavailable',
				__( 'Optimization history is not available.', 'topranker-ai' ),
				array( 'status' => 500 )
			);
		}

		$post_id = $request->get_param( 'post_id' );

		// Verify user can edit this specific post.
		if ( ! $this->can_edit_post( $post_id ) ) {
			return new WP_Error(
				'forbidden',
				__( 'You do not have permission to view this post.', 'topranker-ai' ),
				array( 'status' => 403 )
			);
		}

		$post = get_post( $post_id );

		if ( ! $post ) {
			return new WP_Error(
				'not_found',
				__( 'Post not found.', 'topranker-ai' ),
				array( 'status' => 404 )
			);
		}

		// Get history data.
		$history = new TopRanker_History();
		$data    = $history->get_api_response( $post_id );

		$this->log_debug( 'Got history for post ' . $post_id . ': ' . $data['count'] . ' entries' );

		return rest_ensure_response( $data );
	}

	/**
	 * Get diff between current values and a history entry.
	 *
	 * Pro feature: Shows what changed between current state and a historical version.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response|WP_Error Response object or WP_Error.
	 */
	public function get_history_diff( $request ) {
		// Check if Pro is active.
		if ( ! topranker_is_pro() ) {
			return new WP_Error(
				'pro_required',
				__( 'Optimization history is a Pro feature. Please upgrade to unlock.', 'topranker-ai' ),
				array( 'status' => 403 )
			);
		}

		// Check if Pro class exists.
		if ( ! class_exists( 'TopRanker_History' ) ) {
			return new WP_Error(
				'feature_unavailable',
				__( 'Optimization history is not available.', 'topranker-ai' ),
				array( 'status' => 500 )
			);
		}

		$post_id = $request->get_param( 'post_id' );
		$index   = $request->get_param( 'index' );

		// Verify user can edit this specific post.
		if ( ! $this->can_edit_post( $post_id ) ) {
			return new WP_Error(
				'forbidden',
				__( 'You do not have permission to view this post.', 'topranker-ai' ),
				array( 'status' => 403 )
			);
		}

		$post = get_post( $post_id );

		if ( ! $post ) {
			return new WP_Error(
				'not_found',
				__( 'Post not found.', 'topranker-ai' ),
				array( 'status' => 404 )
			);
		}

		// Get diff data.
		$history = new TopRanker_History();
		$diff    = $history->get_diff( $post_id, $index );

		if ( is_wp_error( $diff ) ) {
			return new WP_Error(
				$diff->get_error_code(),
				$diff->get_error_message(),
				array( 'status' => 400 )
			);
		}

		$this->log_debug( 'Got history diff for post ' . $post_id . ' entry ' . $index );

		return rest_ensure_response( $diff );
	}

	/**
	 * Revert to a history entry.
	 *
	 * Pro feature: Restores a historical version of optimization values.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request Full details about the request.
	 * @return WP_REST_Response|WP_Error Response object or WP_Error.
	 */
	public function revert_to_history( $request ) {
		// Check if Pro is active.
		if ( ! topranker_is_pro() ) {
			return new WP_Error(
				'pro_required',
				__( 'Optimization history is a Pro feature. Please upgrade to unlock.', 'topranker-ai' ),
				array( 'status' => 403 )
			);
		}

		// Check if Pro class exists.
		if ( ! class_exists( 'TopRanker_History' ) ) {
			return new WP_Error(
				'feature_unavailable',
				__( 'Optimization history is not available.', 'topranker-ai' ),
				array( 'status' => 500 )
			);
		}

		$post_id = $request->get_param( 'post_id' );
		$index   = $request->get_param( 'index' );

		// Verify user can edit this specific post.
		if ( ! $this->can_edit_post( $post_id ) ) {
			return new WP_Error(
				'forbidden',
				__( 'You do not have permission to edit this post.', 'topranker-ai' ),
				array( 'status' => 403 )
			);
		}

		$post = get_post( $post_id );

		if ( ! $post ) {
			return new WP_Error(
				'not_found',
				__( 'Post not found.', 'topranker-ai' ),
				array( 'status' => 404 )
			);
		}

		// Perform revert.
		$history = new TopRanker_History();
		$result  = $history->revert_to_entry( $post_id, $index );

		if ( is_wp_error( $result ) ) {
			return new WP_Error(
				$result->get_error_code(),
				$result->get_error_message(),
				array( 'status' => 400 )
			);
		}

		$this->log_debug( 'Reverted post ' . $post_id . ' to history entry ' . $index );

		return rest_ensure_response( $result );
	}
}
