<?php
/**
 * SEO plugin compatibility.
 *
 * Detects installed SEO plugins and provides compatibility features
 * including meta field mapping and sync capabilities.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * TopRanker SEO Compat class.
 *
 * @since 1.0.0
 */
class TopRanker_SEO_Compat {

	/**
	 * Yoast SEO plugin identifier.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	const PLUGIN_YOAST = 'yoast';

	/**
	 * Rank Math plugin identifier.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	const PLUGIN_RANKMATH = 'rankmath';

	/**
	 * SEOPress plugin identifier.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	const PLUGIN_SEOPRESS = 'seopress';

	/**
	 * All in One SEO Pack plugin identifier.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	const PLUGIN_AIOSEO = 'aioseo';

	/**
	 * The SEO Framework plugin identifier.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	const PLUGIN_SEO_FRAMEWORK = 'seo_framework';

	/**
	 * Detected SEO plugin slug.
	 *
	 * @since 1.0.0
	 * @var   string|false
	 */
	private $detected_plugin = null;

	/**
	 * Plugin display names.
	 *
	 * @since 1.0.0
	 * @var   array
	 */
	private $plugin_names = array();

	/**
	 * Meta field mappings for each plugin.
	 *
	 * @since 1.0.0
	 * @var   array
	 */
	private $meta_field_map = array();

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		$this->init_plugin_names();
		$this->init_meta_field_map();
		$this->init_hooks();
	}

	/**
	 * Initialize plugin display names.
	 *
	 * @since 1.0.0
	 */
	private function init_plugin_names() {
		$this->plugin_names = array(
			self::PLUGIN_YOAST         => 'Yoast SEO',
			self::PLUGIN_RANKMATH      => 'Rank Math',
			self::PLUGIN_SEOPRESS      => 'SEOPress',
			self::PLUGIN_AIOSEO        => 'All in One SEO',
			self::PLUGIN_SEO_FRAMEWORK => 'The SEO Framework',
		);
	}

	/**
	 * Initialize meta field mappings for each SEO plugin.
	 *
	 * Maps TopRanker meta keys to the corresponding keys used by each SEO plugin.
	 *
	 * @since 1.0.0
	 */
	private function init_meta_field_map() {
		$this->meta_field_map = array(
			self::PLUGIN_YOAST         => array(
				'meta_title'          => '_yoast_wpseo_title',
				'meta_description'    => '_yoast_wpseo_metadesc',
				'focus_keyphrase'     => '_yoast_wpseo_focuskw',
				'og_title'            => '_yoast_wpseo_opengraph-title',
				'og_description'      => '_yoast_wpseo_opengraph-description',
				'twitter_title'       => '_yoast_wpseo_twitter-title',
				'twitter_description' => '_yoast_wpseo_twitter-description',
			),
			self::PLUGIN_RANKMATH      => array(
				'meta_title'          => 'rank_math_title',
				'meta_description'    => 'rank_math_description',
				'focus_keyphrase'     => 'rank_math_focus_keyword',
				'og_title'            => 'rank_math_facebook_title',
				'og_description'      => 'rank_math_facebook_description',
				'twitter_title'       => 'rank_math_twitter_title',
				'twitter_description' => 'rank_math_twitter_description',
			),
			self::PLUGIN_SEOPRESS      => array(
				'meta_title'          => '_seopress_titles_title',
				'meta_description'    => '_seopress_titles_desc',
				'focus_keyphrase'     => '_seopress_analysis_target_kw',
				'og_title'            => '_seopress_social_fb_title',
				'og_description'      => '_seopress_social_fb_desc',
				'twitter_title'       => '_seopress_social_twitter_title',
				'twitter_description' => '_seopress_social_twitter_desc',
			),
			self::PLUGIN_AIOSEO        => array(
				'meta_title'          => '_aioseo_title',
				'meta_description'    => '_aioseo_description',
				'focus_keyphrase'     => '_aioseo_keywords',
				'og_title'            => '_aioseo_og_title',
				'og_description'      => '_aioseo_og_description',
				'twitter_title'       => '_aioseo_twitter_title',
				'twitter_description' => '_aioseo_twitter_description',
			),
			self::PLUGIN_SEO_FRAMEWORK => array(
				'meta_title'          => '_genesis_title',
				'meta_description'    => '_genesis_description',
				'focus_keyphrase'     => '', // SEO Framework doesn't have focus keyword.
				'og_title'            => '_open_graph_title',
				'og_description'      => '_open_graph_description',
				'twitter_title'       => '_twitter_title',
				'twitter_description' => '_twitter_description',
			),
		);
	}

	/**
	 * Initialize hooks.
	 *
	 * @since 1.0.0
	 */
	private function init_hooks() {
		// Run detection on admin_init.
		add_action( 'admin_init', array( $this, 'detect_seo_plugin' ), 5 );

		// Set smart default SEO mode on first run.
		add_action( 'admin_init', array( $this, 'set_smart_default_mode' ), 10 );
	}

	/**
	 * Detect which SEO plugin is active.
	 *
	 * Checks for popular SEO plugins in order of market share.
	 * Stores the result for later use.
	 *
	 * @since 1.0.0
	 * @return string|false Plugin slug or false if none detected.
	 */
	public function detect_seo_plugin() {
		if ( null !== $this->detected_plugin ) {
			return $this->detected_plugin;
		}

		// Check Yoast SEO (most popular).
		if ( defined( 'WPSEO_VERSION' ) ) {
			$this->detected_plugin = self::PLUGIN_YOAST;
			return $this->detected_plugin;
		}

		// Check Rank Math (second most popular).
		if ( class_exists( 'RankMath' ) ) {
			$this->detected_plugin = self::PLUGIN_RANKMATH;
			return $this->detected_plugin;
		}

		// Check SEOPress.
		if ( function_exists( 'seopress_init' ) ) {
			$this->detected_plugin = self::PLUGIN_SEOPRESS;
			return $this->detected_plugin;
		}

		// Check All in One SEO Pack (v4+ or legacy).
		if ( class_exists( 'AIOSEO\Plugin\AIOSEO' ) || defined( 'AIOSEOP_VERSION' ) ) {
			$this->detected_plugin = self::PLUGIN_AIOSEO;
			return $this->detected_plugin;
		}

		// Check The SEO Framework.
		if ( function_exists( 'the_seo_framework' ) ) {
			$this->detected_plugin = self::PLUGIN_SEO_FRAMEWORK;
			return $this->detected_plugin;
		}

		$this->detected_plugin = false;
		return $this->detected_plugin;
	}

	/**
	 * Set smart default SEO mode based on detected plugins.
	 *
	 * On first run, if an SEO plugin is detected, default to 'suggest' mode.
	 * Otherwise, default to 'standalone' mode.
	 *
	 * @since 1.0.0
	 */
	public function set_smart_default_mode() {
		// Only run once (check for first activation flag).
		$first_run_handled = get_option( 'topranker_seo_mode_set', false );

		if ( $first_run_handled ) {
			return;
		}

		// Detect SEO plugin.
		$detected = $this->detect_seo_plugin();

		// If a plugin is detected and current mode is default, set to 'suggest'.
		if ( $detected ) {
			$current_mode = get_option( 'topranker_seo_mode', 'standalone' );

			// Only change if still on default 'standalone'.
			if ( 'standalone' === $current_mode ) {
				update_option( 'topranker_seo_mode', 'suggest' );
			}
		}

		// Mark as handled.
		update_option( 'topranker_seo_mode_set', true );
	}

	/**
	 * Check if any SEO plugin is detected.
	 *
	 * @since  1.0.0
	 * @return bool True if an SEO plugin is detected.
	 */
	public function has_seo_plugin() {
		return false !== $this->detect_seo_plugin();
	}

	/**
	 * Get the detected SEO plugin slug.
	 *
	 * @since  1.0.0
	 * @return string|false Plugin slug or false.
	 */
	public function get_detected_plugin() {
		return $this->detect_seo_plugin();
	}

	/**
	 * Get the detected SEO plugin display name.
	 *
	 * @since  1.0.0
	 * @return string|false Plugin name or false if none detected.
	 */
	public function get_detected_plugin_name() {
		$plugin = $this->detect_seo_plugin();

		if ( ! $plugin ) {
			return false;
		}

		return isset( $this->plugin_names[ $plugin ] ) ? $this->plugin_names[ $plugin ] : $plugin;
	}

	/**
	 * Get all available plugin display names.
	 *
	 * @since  1.0.0
	 * @return array Plugin slug => name pairs.
	 */
	public function get_plugin_names() {
		return $this->plugin_names;
	}

	/**
	 * Get the meta field key for a specific plugin and field type.
	 *
	 * @since  1.0.0
	 * @param  string      $field_type TopRanker field type (meta_title, meta_description, etc.).
	 * @param  string|null $plugin     Optional. Plugin slug. Uses detected plugin if not provided.
	 * @return string|false Meta field key or false if not found.
	 */
	public function get_meta_field_key( $field_type, $plugin = null ) {
		if ( null === $plugin ) {
			$plugin = $this->detect_seo_plugin();
		}

		if ( ! $plugin ) {
			return false;
		}

		if ( ! isset( $this->meta_field_map[ $plugin ] ) ) {
			return false;
		}

		if ( ! isset( $this->meta_field_map[ $plugin ][ $field_type ] ) ) {
			return false;
		}

		return $this->meta_field_map[ $plugin ][ $field_type ];
	}

	/**
	 * Get all meta field mappings for a specific plugin.
	 *
	 * @since  1.0.0
	 * @param  string|null $plugin Optional. Plugin slug. Uses detected plugin if not provided.
	 * @return array|false Field mappings or false if plugin not found.
	 */
	public function get_meta_field_map( $plugin = null ) {
		if ( null === $plugin ) {
			$plugin = $this->detect_seo_plugin();
		}

		if ( ! $plugin ) {
			return false;
		}

		return isset( $this->meta_field_map[ $plugin ] ) ? $this->meta_field_map[ $plugin ] : false;
	}

	/**
	 * Get meta value from detected SEO plugin.
	 *
	 * Retrieves the meta value from the detected SEO plugin's field.
	 *
	 * @since  1.0.0
	 * @param  int    $post_id    Post ID.
	 * @param  string $field_type TopRanker field type (meta_title, meta_description, etc.).
	 * @return string|false Meta value or false if not found.
	 */
	public function get_plugin_meta( $post_id, $field_type ) {
		$meta_key = $this->get_meta_field_key( $field_type );

		if ( ! $meta_key ) {
			return false;
		}

		$value = get_post_meta( $post_id, $meta_key, true );

		return is_string( $value ) ? $value : false;
	}

	/**
	 * Set meta value in detected SEO plugin.
	 *
	 * Writes the meta value to the detected SEO plugin's field.
	 *
	 * @since  1.0.0
	 * @param  int    $post_id    Post ID.
	 * @param  string $field_type TopRanker field type (meta_title, meta_description, etc.).
	 * @param  string $value      Value to set.
	 * @return bool True on success, false on failure.
	 */
	public function set_plugin_meta( $post_id, $field_type, $value ) {
		$meta_key = $this->get_meta_field_key( $field_type );

		if ( ! $meta_key ) {
			return false;
		}

		// Sanitize based on field type.
		if ( in_array( $field_type, array( 'meta_description', 'og_description', 'twitter_description' ), true ) ) {
			$value = sanitize_textarea_field( $value );
		} else {
			$value = sanitize_text_field( $value );
		}

		return (bool) update_post_meta( $post_id, $meta_key, $value );
	}

	/**
	 * Sync all TopRanker meta to detected SEO plugin.
	 *
	 * Writes all TopRanker meta values to the corresponding SEO plugin fields.
	 *
	 * @since  1.0.0
	 * @param  int $post_id Post ID.
	 * @return array Results array with field_type => success pairs.
	 */
	public function sync_all_meta_to_plugin( $post_id ) {
		$results = array();

		if ( ! $this->has_seo_plugin() ) {
			return $results;
		}

		// Map TopRanker meta keys to field types.
		$topranker_fields = array(
			'_topranker_meta_title'          => 'meta_title',
			'_topranker_meta_description'    => 'meta_description',
			'_topranker_focus_keyphrase'     => 'focus_keyphrase',
			'_topranker_og_title'            => 'og_title',
			'_topranker_og_description'      => 'og_description',
			'_topranker_twitter_title'       => 'twitter_title',
			'_topranker_twitter_description' => 'twitter_description',
		);

		foreach ( $topranker_fields as $topranker_key => $field_type ) {
			$value = get_post_meta( $post_id, $topranker_key, true );

			if ( ! empty( $value ) ) {
				$results[ $field_type ] = $this->set_plugin_meta( $post_id, $field_type, $value );
			}
		}

		return $results;
	}

	/**
	 * Check if SEO plugin handles meta tag output.
	 *
	 * Determines if the detected SEO plugin will output meta tags,
	 * which means TopRanker should not output duplicates.
	 *
	 * @since  1.0.0
	 * @return bool True if SEO plugin handles meta output.
	 */
	public function plugin_handles_meta_output() {
		$plugin = $this->detect_seo_plugin();

		if ( ! $plugin ) {
			return false;
		}

		// All detected plugins handle meta output by default.
		// Specific plugin checks can be added here if needed.
		switch ( $plugin ) {
			case self::PLUGIN_YOAST:
				// Yoast always handles meta unless disabled.
				return true;

			case self::PLUGIN_RANKMATH:
				// RankMath always handles meta.
				return true;

			case self::PLUGIN_SEOPRESS:
				// Check if SEOPress titles module is active.
				$toggle = get_option( 'seopress_toggle', array() );
				return isset( $toggle['toggle-titles'] ) && '1' === $toggle['toggle-titles'];

			case self::PLUGIN_AIOSEO:
				return true;

			case self::PLUGIN_SEO_FRAMEWORK:
				return true;
		}

		return false;
	}

	/**
	 * Check if SEO plugin handles Open Graph output.
	 *
	 * @since  1.0.0
	 * @return bool True if SEO plugin handles OG output.
	 */
	public function plugin_handles_og_output() {
		$plugin = $this->detect_seo_plugin();

		if ( ! $plugin ) {
			return false;
		}

		switch ( $plugin ) {
			case self::PLUGIN_YOAST:
				// Check Yoast social settings.
				$social = get_option( 'wpseo_social', array() );
				return ! empty( $social['opengraph'] );

			case self::PLUGIN_RANKMATH:
				// RankMath handles OG by default.
				return true;

			case self::PLUGIN_SEOPRESS:
				// Check if SEOPress social module is active.
				$toggle = get_option( 'seopress_toggle', array() );
				return isset( $toggle['toggle-social'] ) && '1' === $toggle['toggle-social'];

			case self::PLUGIN_AIOSEO:
				return true;

			case self::PLUGIN_SEO_FRAMEWORK:
				return true;
		}

		return false;
	}

	/**
	 * Check if SEO plugin handles Twitter Card output.
	 *
	 * @since  1.0.0
	 * @return bool True if SEO plugin handles Twitter output.
	 */
	public function plugin_handles_twitter_output() {
		$plugin = $this->detect_seo_plugin();

		if ( ! $plugin ) {
			return false;
		}

		switch ( $plugin ) {
			case self::PLUGIN_YOAST:
				// Check Yoast social settings.
				$social = get_option( 'wpseo_social', array() );
				return ! empty( $social['twitter'] );

			case self::PLUGIN_RANKMATH:
				// RankMath handles Twitter by default.
				return true;

			case self::PLUGIN_SEOPRESS:
				// Check if SEOPress social module is active.
				$toggle = get_option( 'seopress_toggle', array() );
				return isset( $toggle['toggle-social'] ) && '1' === $toggle['toggle-social'];

			case self::PLUGIN_AIOSEO:
				return true;

			case self::PLUGIN_SEO_FRAMEWORK:
				return true;
		}

		return false;
	}

	/**
	 * Get the current SEO mode.
	 *
	 * @since  1.0.0
	 * @return string 'standalone', 'suggest', or 'sync'.
	 */
	public function get_seo_mode() {
		return get_option( 'topranker_seo_mode', 'standalone' );
	}

	/**
	 * Check if mode is standalone (output meta tags).
	 *
	 * @since  1.0.0
	 * @return bool True if standalone mode.
	 */
	public function is_standalone_mode() {
		return 'standalone' === $this->get_seo_mode();
	}

	/**
	 * Check if mode is suggest only (no meta output).
	 *
	 * @since  1.0.0
	 * @return bool True if suggest only mode.
	 */
	public function is_suggest_mode() {
		return 'suggest' === $this->get_seo_mode();
	}

	/**
	 * Check if mode is sync (write to SEO plugin fields).
	 *
	 * @since  1.0.0
	 * @return bool True if sync mode.
	 */
	public function is_sync_mode() {
		return 'sync' === $this->get_seo_mode();
	}

	/**
	 * Should TopRanker output meta description tag?
	 *
	 * @since  1.0.0
	 * @return bool True if TopRanker should output meta description.
	 */
	public function should_output_meta_description() {
		// Only output in standalone mode.
		if ( ! $this->is_standalone_mode() ) {
			return false;
		}

		// Don't output if another plugin handles it.
		if ( $this->plugin_handles_meta_output() ) {
			return false;
		}

		return true;
	}

	/**
	 * Should TopRanker output Open Graph tags?
	 *
	 * @since  1.0.0
	 * @return bool True if TopRanker should output OG tags.
	 */
	public function should_output_og_tags() {
		// Only output in standalone mode.
		if ( ! $this->is_standalone_mode() ) {
			return false;
		}

		// Don't output if another plugin handles it.
		if ( $this->plugin_handles_og_output() ) {
			return false;
		}

		return true;
	}

	/**
	 * Should TopRanker output Twitter Card tags?
	 *
	 * @since  1.0.0
	 * @return bool True if TopRanker should output Twitter tags.
	 */
	public function should_output_twitter_tags() {
		// Only output in standalone mode.
		if ( ! $this->is_standalone_mode() ) {
			return false;
		}

		// Don't output if another plugin handles it.
		if ( $this->plugin_handles_twitter_output() ) {
			return false;
		}

		return true;
	}

	/**
	 * Get import data from detected SEO plugin.
	 *
	 * Retrieves existing meta values from the detected SEO plugin
	 * that can be used as starting point for TopRanker.
	 *
	 * @since  1.0.0
	 * @param  int $post_id Post ID.
	 * @return array Array of imported meta values.
	 */
	public function get_import_data( $post_id ) {
		$data = array();

		if ( ! $this->has_seo_plugin() ) {
			return $data;
		}

		$field_types = array(
			'meta_title',
			'meta_description',
			'focus_keyphrase',
			'og_title',
			'og_description',
			'twitter_title',
			'twitter_description',
		);

		foreach ( $field_types as $field_type ) {
			$value = $this->get_plugin_meta( $post_id, $field_type );

			if ( false !== $value && '' !== $value ) {
				$data[ $field_type ] = $value;
			}
		}

		return $data;
	}

	/**
	 * Get compatibility status for a post.
	 *
	 * Returns information about which SEO plugin fields have values
	 * and which TopRanker fields have values.
	 *
	 * @since  1.0.0
	 * @param  int $post_id Post ID.
	 * @return array Status array with plugin and topranker data.
	 */
	public function get_compatibility_status( $post_id ) {
		$status = array(
			'detected_plugin'      => $this->get_detected_plugin(),
			'detected_plugin_name' => $this->get_detected_plugin_name(),
			'seo_mode'             => $this->get_seo_mode(),
			'plugin_has_data'      => false,
			'topranker_has_data'   => false,
			'plugin_fields'        => array(),
			'topranker_fields'     => array(),
		);

		// Check TopRanker fields.
		$topranker_keys = array(
			'meta_title'          => '_topranker_meta_title',
			'meta_description'    => '_topranker_meta_description',
			'focus_keyphrase'     => '_topranker_focus_keyphrase',
			'og_title'            => '_topranker_og_title',
			'og_description'      => '_topranker_og_description',
			'twitter_title'       => '_topranker_twitter_title',
			'twitter_description' => '_topranker_twitter_description',
		);

		foreach ( $topranker_keys as $field_type => $meta_key ) {
			$value = get_post_meta( $post_id, $meta_key, true );
			if ( ! empty( $value ) ) {
				$status['topranker_fields'][ $field_type ] = true;
				$status['topranker_has_data']              = true;
			}
		}

		// Check plugin fields.
		if ( $this->has_seo_plugin() ) {
			$plugin_data = $this->get_import_data( $post_id );

			if ( ! empty( $plugin_data ) ) {
				$status['plugin_has_data'] = true;
				$status['plugin_fields']   = array_keys( $plugin_data );
			}
		}

		return $status;
	}
}
