<?php
/**
 * Social meta (Open Graph and Twitter Card) generation.
 *
 * Handles generation of Open Graph and Twitter Card meta tags using OpenAI.
 * Includes prompt engineering, response validation, and meta storage.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * TopRanker Social class.
 *
 * @since 1.0.0
 */
class TopRanker_Social {

	/**
	 * Maximum OG title length.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const OG_TITLE_MAX_LENGTH = 60;

	/**
	 * Maximum OG description length.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const OG_DESCRIPTION_MAX_LENGTH = 200;

	/**
	 * Maximum Twitter title length.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const TWITTER_TITLE_MAX_LENGTH = 70;

	/**
	 * Maximum Twitter description length.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const TWITTER_DESCRIPTION_MAX_LENGTH = 200;

	/**
	 * API instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_API|null
	 */
	private $api = null;

	/**
	 * Optimizer instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_Optimizer|null
	 */
	private $optimizer = null;

	/**
	 * SEO Compat instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_SEO_Compat|null
	 */
	private $seo_compat = null;

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		// Classes will be initialized lazily when needed.
		$this->init_hooks();
	}

	/**
	 * Initialize hooks for social meta tag output.
	 *
	 * @since 1.0.0
	 */
	private function init_hooks() {
		// Hook into TopRanker's wp_head action for meta tag output.
		add_action( 'topranker_wp_head', array( $this, 'output_social_meta_tags' ) );
	}

	/**
	 * Get the API instance.
	 *
	 * @since  1.0.0
	 * @return TopRanker_API
	 */
	private function get_api() {
		if ( null === $this->api ) {
			$this->api = new TopRanker_API();
		}
		return $this->api;
	}

	/**
	 * Get the optimizer instance.
	 *
	 * @since  1.0.0
	 * @return TopRanker_Optimizer
	 */
	private function get_optimizer() {
		if ( null === $this->optimizer ) {
			$this->optimizer = new TopRanker_Optimizer();
		}
		return $this->optimizer;
	}

	/**
	 * Get the SEO Compat instance.
	 *
	 * @since  1.0.0
	 * @return TopRanker_SEO_Compat
	 */
	private function get_seo_compat() {
		if ( null === $this->seo_compat ) {
			$this->seo_compat = new TopRanker_SEO_Compat();
		}
		return $this->seo_compat;
	}

	/**
	 * Generate social meta suggestions for a post.
	 *
	 * Generates Open Graph and Twitter Card meta data.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post           Post ID or WP_Post object.
	 * @param string      $focus_keyphrase Optional. Focus keyphrase to include.
	 * @return array|WP_Error Array with social meta suggestions or WP_Error on failure.
	 */
	public function generate_social_meta( $post, $focus_keyphrase = '' ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return new WP_Error(
				'invalid_post',
				__( 'Invalid post.', 'topranker-ai' )
			);
		}

		$optimizer  = $this->get_optimizer();
		$api        = $this->get_api();
		$post_title = $post->post_title;
		$content    = $optimizer->prepare_content( $post, 8000 );
		$context    = $optimizer->build_context_prefix( $post );

		// Build the prompt.
		$prompt = $this->build_social_meta_prompt(
			$post_title,
			$content,
			$focus_keyphrase,
			$context
		);

		// Call the API.
		$messages = array(
			array(
				'role'    => 'system',
				'content' => $context,
			),
			array(
				'role'    => 'user',
				'content' => $prompt,
			),
		);

		$response = $api->chat_completion( $messages );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		// Parse the JSON response.
		$parsed = $api->parse_json_response( $response['content'] );

		if ( is_wp_error( $parsed ) ) {
			// Retry once.
			$response = $api->chat_completion( $messages );

			if ( is_wp_error( $response ) ) {
				return $response;
			}

			$parsed = $api->parse_json_response( $response['content'] );

			if ( is_wp_error( $parsed ) ) {
				return $parsed;
			}
		}

		// Validate and format social meta.
		$social_meta = $this->validate_social_meta_response( $parsed );

		if ( is_wp_error( $social_meta ) ) {
			return $social_meta;
		}

		return array(
			'og_title'            => $social_meta['og_title'],
			'og_description'      => $social_meta['og_description'],
			'twitter_title'       => $social_meta['twitter_title'],
			'twitter_description' => $social_meta['twitter_description'],
			'post_id'             => $post->ID,
		);
	}

	/**
	 * Build the social meta prompt.
	 *
	 * @since 1.0.0
	 * @param string $post_title      Post title.
	 * @param string $content         Prepared post content.
	 * @param string $focus_keyphrase Focus keyphrase if provided.
	 * @param string $context         Context prefix.
	 * @return string The formatted prompt.
	 */
	private function build_social_meta_prompt( $post_title, $content, $focus_keyphrase, $context ) {
		$keyphrase_instruction = '';
		if ( ! empty( $focus_keyphrase ) ) {
			$keyphrase_instruction = sprintf(
				/* translators: %s: focus keyphrase */
				__( 'Incorporate the focus keyphrase "%s" naturally where appropriate.', 'topranker-ai' ),
				$focus_keyphrase
			);
		}

		$prompt = sprintf(
			/* translators: 1: Post title, 2: Content excerpt, 3: OG title max, 4: OG desc max, 5: Twitter title max, 6: Twitter desc max, 7: Keyphrase instruction */
			__(
				'Generate social media meta tags for the following content. These are for Open Graph (Facebook, LinkedIn) and Twitter Cards.

Post Title: %1$s

Content:
%2$s

Requirements:
- og_title: Maximum %3$d characters. Optimized for social clicks, can differ from SEO title. Be engaging and shareable.
- og_description: Maximum %4$d characters. More engaging and shareable than SEO meta description. Include a hook to encourage clicks.
- twitter_title: Maximum %5$d characters. Optimized for Twitter engagement. Can be more casual/punchy than og_title.
- twitter_description: Maximum %6$d characters. Compelling summary for Twitter.
%7$s

Respond with valid JSON only, in this exact format:
{
  "og_title": "The Open Graph title",
  "og_description": "The Open Graph description",
  "twitter_title": "The Twitter Card title",
  "twitter_description": "The Twitter Card description"
}',
				'topranker-ai'
			),
			$post_title,
			mb_substr( $content, 0, 2000 ),
			self::OG_TITLE_MAX_LENGTH,
			self::OG_DESCRIPTION_MAX_LENGTH,
			self::TWITTER_TITLE_MAX_LENGTH,
			self::TWITTER_DESCRIPTION_MAX_LENGTH,
			$keyphrase_instruction
		);

		return $prompt;
	}

	/**
	 * Validate and format social meta response.
	 *
	 * @since 1.0.0
	 * @param array $parsed Parsed JSON response.
	 * @return array|WP_Error Formatted social meta or error.
	 */
	private function validate_social_meta_response( $parsed ) {
		$required_fields = array( 'og_title', 'og_description', 'twitter_title', 'twitter_description' );

		foreach ( $required_fields as $field ) {
			if ( ! isset( $parsed[ $field ] ) || ! is_string( $parsed[ $field ] ) ) {
				return new WP_Error(
					'invalid_response',
					/* translators: %s: field name */
					sprintf( __( 'Invalid response format from AI. Missing field: %s', 'topranker-ai' ), $field )
				);
			}
		}

		// Sanitize and truncate each field.
		$og_title = sanitize_text_field( $parsed['og_title'] );
		if ( mb_strlen( $og_title ) > self::OG_TITLE_MAX_LENGTH ) {
			$og_title = $this->smart_truncate( $og_title, self::OG_TITLE_MAX_LENGTH );
		}

		$og_description = sanitize_textarea_field( $parsed['og_description'] );
		if ( mb_strlen( $og_description ) > self::OG_DESCRIPTION_MAX_LENGTH ) {
			$og_description = $this->smart_truncate( $og_description, self::OG_DESCRIPTION_MAX_LENGTH );
		}

		$twitter_title = sanitize_text_field( $parsed['twitter_title'] );
		if ( mb_strlen( $twitter_title ) > self::TWITTER_TITLE_MAX_LENGTH ) {
			$twitter_title = $this->smart_truncate( $twitter_title, self::TWITTER_TITLE_MAX_LENGTH );
		}

		$twitter_description = sanitize_textarea_field( $parsed['twitter_description'] );
		if ( mb_strlen( $twitter_description ) > self::TWITTER_DESCRIPTION_MAX_LENGTH ) {
			$twitter_description = $this->smart_truncate( $twitter_description, self::TWITTER_DESCRIPTION_MAX_LENGTH );
		}

		// Check for empty results.
		if ( empty( $og_title ) || empty( $og_description ) ) {
			return new WP_Error(
				'empty_social_meta',
				__( 'Generated social meta was empty.', 'topranker-ai' )
			);
		}

		return array(
			'og_title'            => $og_title,
			'og_description'      => $og_description,
			'twitter_title'       => $twitter_title,
			'twitter_description' => $twitter_description,
		);
	}

	/**
	 * Smart truncate text at word boundary.
	 *
	 * @since 1.0.0
	 * @param string $text   Text to truncate.
	 * @param int    $length Maximum length.
	 * @return string Truncated text.
	 */
	private function smart_truncate( $text, $length ) {
		if ( mb_strlen( $text ) <= $length ) {
			return $text;
		}

		// Truncate to length.
		$truncated = mb_substr( $text, 0, $length );

		// Try to break at word boundary.
		$last_space = mb_strrpos( $truncated, ' ' );

		if ( false !== $last_space && $last_space > $length * 0.8 ) {
			$truncated = mb_substr( $truncated, 0, $last_space );
		}

		return trim( $truncated );
	}

	/**
	 * Save OG title for a post.
	 *
	 * @since 1.0.0
	 * @param int    $post_id  Post ID.
	 * @param string $og_title OG title to save.
	 * @return bool True on success.
	 */
	public function save_og_title( $post_id, $og_title ) {
		$og_title = sanitize_text_field( $og_title );

		if ( mb_strlen( $og_title ) > self::OG_TITLE_MAX_LENGTH ) {
			$og_title = $this->smart_truncate( $og_title, self::OG_TITLE_MAX_LENGTH );
		}

		return (bool) update_post_meta( $post_id, '_topranker_og_title', $og_title );
	}

	/**
	 * Save OG description for a post.
	 *
	 * @since 1.0.0
	 * @param int    $post_id        Post ID.
	 * @param string $og_description OG description to save.
	 * @return bool True on success.
	 */
	public function save_og_description( $post_id, $og_description ) {
		$og_description = sanitize_textarea_field( $og_description );

		if ( mb_strlen( $og_description ) > self::OG_DESCRIPTION_MAX_LENGTH ) {
			$og_description = $this->smart_truncate( $og_description, self::OG_DESCRIPTION_MAX_LENGTH );
		}

		return (bool) update_post_meta( $post_id, '_topranker_og_description', $og_description );
	}

	/**
	 * Save Twitter title for a post.
	 *
	 * @since 1.0.0
	 * @param int    $post_id       Post ID.
	 * @param string $twitter_title Twitter title to save.
	 * @return bool True on success.
	 */
	public function save_twitter_title( $post_id, $twitter_title ) {
		$twitter_title = sanitize_text_field( $twitter_title );

		if ( mb_strlen( $twitter_title ) > self::TWITTER_TITLE_MAX_LENGTH ) {
			$twitter_title = $this->smart_truncate( $twitter_title, self::TWITTER_TITLE_MAX_LENGTH );
		}

		return (bool) update_post_meta( $post_id, '_topranker_twitter_title', $twitter_title );
	}

	/**
	 * Save Twitter description for a post.
	 *
	 * @since 1.0.0
	 * @param int    $post_id             Post ID.
	 * @param string $twitter_description Twitter description to save.
	 * @return bool True on success.
	 */
	public function save_twitter_description( $post_id, $twitter_description ) {
		$twitter_description = sanitize_textarea_field( $twitter_description );

		if ( mb_strlen( $twitter_description ) > self::TWITTER_DESCRIPTION_MAX_LENGTH ) {
			$twitter_description = $this->smart_truncate( $twitter_description, self::TWITTER_DESCRIPTION_MAX_LENGTH );
		}

		return (bool) update_post_meta( $post_id, '_topranker_twitter_description', $twitter_description );
	}

	/**
	 * Save all social meta for a post.
	 *
	 * @since 1.0.0
	 * @param int   $post_id     Post ID.
	 * @param array $social_meta Array with og_title, og_description, twitter_title, twitter_description.
	 * @return bool True if all saved successfully.
	 */
	public function save_social_meta( $post_id, $social_meta ) {
		$success = true;

		if ( isset( $social_meta['og_title'] ) ) {
			$success = $this->save_og_title( $post_id, $social_meta['og_title'] ) && $success;
		}

		if ( isset( $social_meta['og_description'] ) ) {
			$success = $this->save_og_description( $post_id, $social_meta['og_description'] ) && $success;
		}

		if ( isset( $social_meta['twitter_title'] ) ) {
			$success = $this->save_twitter_title( $post_id, $social_meta['twitter_title'] ) && $success;
		}

		if ( isset( $social_meta['twitter_description'] ) ) {
			$success = $this->save_twitter_description( $post_id, $social_meta['twitter_description'] ) && $success;
		}

		return $success;
	}

	/**
	 * Get saved OG title for a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return string OG title or empty string.
	 */
	public function get_og_title( $post_id ) {
		return get_post_meta( $post_id, '_topranker_og_title', true );
	}

	/**
	 * Get saved OG description for a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return string OG description or empty string.
	 */
	public function get_og_description( $post_id ) {
		return get_post_meta( $post_id, '_topranker_og_description', true );
	}

	/**
	 * Get saved Twitter title for a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return string Twitter title or empty string.
	 */
	public function get_twitter_title( $post_id ) {
		return get_post_meta( $post_id, '_topranker_twitter_title', true );
	}

	/**
	 * Get saved Twitter description for a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return string Twitter description or empty string.
	 */
	public function get_twitter_description( $post_id ) {
		return get_post_meta( $post_id, '_topranker_twitter_description', true );
	}

	/**
	 * Get all saved social meta for a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return array Array with all social meta values.
	 */
	public function get_social_meta( $post_id ) {
		return array(
			'og_title'            => $this->get_og_title( $post_id ),
			'og_description'      => $this->get_og_description( $post_id ),
			'twitter_title'       => $this->get_twitter_title( $post_id ),
			'twitter_description' => $this->get_twitter_description( $post_id ),
		);
	}

	/**
	 * Check if post has OG meta.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True if OG title and description exist.
	 */
	public function has_og_meta( $post_id ) {
		return ! empty( $this->get_og_title( $post_id ) ) && ! empty( $this->get_og_description( $post_id ) );
	}

	/**
	 * Check if post has Twitter meta.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True if Twitter title and description exist.
	 */
	public function has_twitter_meta( $post_id ) {
		return ! empty( $this->get_twitter_title( $post_id ) ) && ! empty( $this->get_twitter_description( $post_id ) );
	}

	/**
	 * Check if post has any social meta.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True if any social meta exists.
	 */
	public function has_social_meta( $post_id ) {
		return $this->has_og_meta( $post_id ) || $this->has_twitter_meta( $post_id );
	}

	/**
	 * Output social meta tags in wp_head.
	 *
	 * Outputs Open Graph and Twitter Card meta tags when:
	 * 1. We're on a singular post/page
	 * 2. The post has TopRanker social meta values
	 * 3. SEO mode is 'standalone' (not 'suggest')
	 * 4. No duplicate tags from another SEO plugin
	 *
	 * @since 1.0.0
	 */
	public function output_social_meta_tags() {
		// Only output on singular posts/pages.
		if ( ! is_singular() ) {
			return;
		}

		$seo_compat = $this->get_seo_compat();

		// Check SEO mode - only output in 'standalone' mode.
		if ( ! $seo_compat->is_standalone_mode() ) {
			return;
		}

		$post_id = get_queried_object_id();

		if ( ! $post_id ) {
			return;
		}

		// Check if post type is enabled.
		$enabled_post_types = get_option( 'topranker_post_types', array( 'post', 'page' ) );
		if ( ! is_array( $enabled_post_types ) ) {
			$enabled_post_types = array( 'post', 'page' );
		}

		$post_type = get_post_type( $post_id );
		if ( ! in_array( $post_type, $enabled_post_types, true ) ) {
			return;
		}

		// Output Open Graph tags (if no conflict using central SEO compat check + hook detection).
		if ( $seo_compat->should_output_og_tags() && ! $this->is_og_already_output() ) {
			$this->output_open_graph_tags( $post_id );
		}

		// Output Twitter Card tags (if no conflict using central SEO compat check + hook detection).
		if ( $seo_compat->should_output_twitter_tags() && ! $this->is_twitter_already_output() ) {
			$this->output_twitter_card_tags( $post_id );
		}
	}

	/**
	 * Output Open Graph meta tags.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 */
	private function output_open_graph_tags( $post_id ) {
		$post      = get_post( $post_id );
		$og_title  = $this->get_og_title( $post_id );
		$og_desc   = $this->get_og_description( $post_id );
		$og_image  = $this->get_og_image( $post_id );
		$permalink = get_permalink( $post_id );

		// Use post title as fallback for OG title.
		if ( empty( $og_title ) && $post ) {
			$og_title = $post->post_title;
		}

		// Need at least a title to output OG tags.
		if ( empty( $og_title ) ) {
			return;
		}

		// Basic required OG tags.
		echo "\n<!-- TopRanker AI Open Graph Tags -->\n";

		printf(
			'<meta property="og:title" content="%s" />' . "\n",
			esc_attr( $og_title )
		);

		printf(
			'<meta property="og:type" content="%s" />' . "\n",
			'article'
		);

		printf(
			'<meta property="og:url" content="%s" />' . "\n",
			esc_url( $permalink )
		);

		if ( ! empty( $og_desc ) ) {
			printf(
				'<meta property="og:description" content="%s" />' . "\n",
				esc_attr( $og_desc )
			);
		}

		if ( ! empty( $og_image ) ) {
			printf(
				'<meta property="og:image" content="%s" />' . "\n",
				esc_url( $og_image )
			);

			// Add image dimensions if available.
			$dimensions = $this->get_og_image_dimensions( $post_id );
			if ( ! empty( $dimensions['width'] ) ) {
				printf(
					'<meta property="og:image:width" content="%d" />' . "\n",
					absint( $dimensions['width'] )
				);
			}
			if ( ! empty( $dimensions['height'] ) ) {
				printf(
					'<meta property="og:image:height" content="%d" />' . "\n",
					absint( $dimensions['height'] )
				);
			}
		}

		// Site name.
		$site_name = get_bloginfo( 'name' );
		if ( ! empty( $site_name ) ) {
			printf(
				'<meta property="og:site_name" content="%s" />' . "\n",
				esc_attr( $site_name )
			);
		}

		// Locale.
		printf(
			'<meta property="og:locale" content="%s" />' . "\n",
			esc_attr( get_locale() )
		);
	}

	/**
	 * Output Twitter Card meta tags.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 */
	private function output_twitter_card_tags( $post_id ) {
		$post          = get_post( $post_id );
		$twitter_title = $this->get_twitter_title( $post_id );
		$twitter_desc  = $this->get_twitter_description( $post_id );
		$og_image      = $this->get_og_image( $post_id );

		// Use OG title as fallback.
		if ( empty( $twitter_title ) ) {
			$twitter_title = $this->get_og_title( $post_id );
		}

		// Use post title as fallback.
		if ( empty( $twitter_title ) && $post ) {
			$twitter_title = $post->post_title;
		}

		// Need at least a title to output Twitter tags.
		if ( empty( $twitter_title ) ) {
			return;
		}

		echo "\n<!-- TopRanker AI Twitter Card Tags -->\n";

		// Card type: summary_large_image if we have an image, otherwise summary.
		$card_type = ! empty( $og_image ) ? 'summary_large_image' : 'summary';

		printf(
			'<meta name="twitter:card" content="%s" />' . "\n",
			esc_attr( $card_type )
		);

		printf(
			'<meta name="twitter:title" content="%s" />' . "\n",
			esc_attr( $twitter_title )
		);

		if ( ! empty( $twitter_desc ) ) {
			printf(
				'<meta name="twitter:description" content="%s" />' . "\n",
				esc_attr( $twitter_desc )
			);
		}

		if ( ! empty( $og_image ) ) {
			printf(
				'<meta name="twitter:image" content="%s" />' . "\n",
				esc_url( $og_image )
			);
		}
	}

	/**
	 * Check if Open Graph tags are already being output by another plugin.
	 *
	 * Uses hook-based detection to check if OG tags are already handled.
	 *
	 * @since  1.0.0
	 * @return bool True if another plugin outputs OG tags.
	 */
	private function is_og_already_output() {
		// Check Yoast SEO specific hooks for OG.
		if ( defined( 'WPSEO_VERSION' ) ) {
			$yoast_social = get_option( 'wpseo_social', array() );
			// Yoast outputs OG if opengraph option is enabled or not explicitly disabled.
			if ( ! isset( $yoast_social['opengraph'] ) || ! empty( $yoast_social['opengraph'] ) ) {
				// Check if Yoast's OG hook is registered.
				if ( has_action( 'wpseo_head' ) ) {
					return true;
				}
			}
		}

		// Check RankMath specific hooks for OG.
		if ( class_exists( 'RankMath' ) ) {
			// RankMath uses rank_math/opengraph/facebook hook.
			if ( has_action( 'rank_math/opengraph/facebook' ) || has_action( 'rank_math/head' ) ) {
				return true;
			}
		}

		// Check SEOPress.
		if ( function_exists( 'seopress_init' ) ) {
			$seopress_toggle = get_option( 'seopress_toggle', array() );
			if ( isset( $seopress_toggle['toggle-social'] ) && '1' === $seopress_toggle['toggle-social'] ) {
				// Check if SEOPress social hooks are registered.
				if ( has_action( 'wp_head', 'seopress_social_fb_og_title' ) ) {
					return true;
				}
				return true;
			}
		}

		// Check All in One SEO Pack.
		if ( class_exists( 'AIOSEO\Plugin\AIOSEO' ) ) {
			return true;
		}
		if ( defined( 'AIOSEOP_VERSION' ) ) {
			// Legacy AIOSEO.
			if ( has_action( 'wp_head', 'aiosp_opengraph' ) ) {
				return true;
			}
			return true;
		}

		// Check The SEO Framework.
		if ( function_exists( 'the_seo_framework' ) ) {
			return true;
		}

		// Check Jetpack Open Graph module.
		if ( class_exists( 'Jetpack' ) && method_exists( 'Jetpack', 'is_module_active' ) ) {
			if ( Jetpack::is_module_active( 'publicize' ) ) {
				return true;
			}
			if ( Jetpack::is_module_active( 'sharedaddy' ) ) {
				return true;
			}
		}

		// Check Open Graph plugin.
		if ( class_exists( 'Open_Graph' ) ) {
			return true;
		}

		// Check WP Open Graph plugin.
		if ( function_exists( 'wpog_add_og_tags' ) ) {
			return true;
		}

		// Check Facebook for WordPress plugin.
		if ( defined( 'STARTER_TEMPLATE_VERSION' ) || class_exists( 'Facebook\PixelCaffeine' ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Check if Twitter Card tags are already being output by another plugin.
	 *
	 * Uses hook-based detection to check if Twitter tags are already handled.
	 *
	 * @since  1.0.0
	 * @return bool True if another plugin outputs Twitter tags.
	 */
	private function is_twitter_already_output() {
		// Check Yoast SEO specific hooks for Twitter.
		if ( defined( 'WPSEO_VERSION' ) ) {
			$yoast_social = get_option( 'wpseo_social', array() );
			// Yoast outputs Twitter cards if twitter option is enabled or not explicitly disabled.
			if ( ! isset( $yoast_social['twitter'] ) || ! empty( $yoast_social['twitter'] ) ) {
				if ( has_action( 'wpseo_head' ) ) {
					return true;
				}
			}
		}

		// Check RankMath specific hooks for Twitter.
		if ( class_exists( 'RankMath' ) ) {
			// RankMath uses rank_math/opengraph/twitter hook.
			if ( has_action( 'rank_math/opengraph/twitter' ) || has_action( 'rank_math/head' ) ) {
				return true;
			}
		}

		// Check SEOPress.
		if ( function_exists( 'seopress_init' ) ) {
			$seopress_toggle = get_option( 'seopress_toggle', array() );
			if ( isset( $seopress_toggle['toggle-social'] ) && '1' === $seopress_toggle['toggle-social'] ) {
				return true;
			}
		}

		// Check All in One SEO Pack.
		if ( class_exists( 'AIOSEO\Plugin\AIOSEO' ) || defined( 'AIOSEOP_VERSION' ) ) {
			return true;
		}

		// Check The SEO Framework.
		if ( function_exists( 'the_seo_framework' ) ) {
			return true;
		}

		// Check JM Twitter Cards plugin.
		if ( function_exists( 'jm_twitter_cards' ) ) {
			return true;
		}

		// Check Twitter Cards Meta plugin.
		if ( class_exists( 'TwitterCardsMeta' ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Remove conflicting Open Graph hooks from other plugins.
	 *
	 * This method can be used to remove other plugin's OG output
	 * if TopRanker is set to handle OG exclusively.
	 *
	 * Note: This is an aggressive approach and should only be used
	 * when explicitly enabled by the user.
	 *
	 * @since 1.0.0
	 */
	public function remove_conflicting_og_hooks() {
		// Remove Jetpack OG output.
		if ( class_exists( 'Jetpack' ) ) {
			remove_action( 'wp_head', 'jetpack_og_tags' );
		}

		// Remove Open Graph plugin.
		if ( function_exists( 'opengraph_action_head' ) ) {
			remove_action( 'wp_head', 'opengraph_action_head' );
		}

		// Remove WP Open Graph.
		if ( function_exists( 'wpog_add_og_tags' ) ) {
			remove_action( 'wp_head', 'wpog_add_og_tags' );
		}
	}

	/**
	 * Remove conflicting Twitter Card hooks from other plugins.
	 *
	 * This method can be used to remove other plugin's Twitter output
	 * if TopRanker is set to handle Twitter cards exclusively.
	 *
	 * Note: This is an aggressive approach and should only be used
	 * when explicitly enabled by the user.
	 *
	 * @since 1.0.0
	 */
	public function remove_conflicting_twitter_hooks() {
		// Remove JM Twitter Cards.
		if ( function_exists( 'jm_twitter_cards' ) ) {
			remove_action( 'wp_head', 'jm_twitter_cards' );
		}

		// Remove Twitter Cards Meta.
		if ( class_exists( 'TwitterCardsMeta' ) ) {
			remove_filter( 'wp_head', array( 'TwitterCardsMeta', 'output_twitter_card_meta' ) );
		}
	}

	/**
	 * Get the featured image URL for OG image.
	 *
	 * @since 1.0.0
	 * @param int    $post_id Post ID.
	 * @param string $size    Image size. Default 'large'.
	 * @return string Image URL or empty string.
	 */
	public function get_og_image( $post_id, $size = 'large' ) {
		$thumbnail_id = get_post_thumbnail_id( $post_id );

		if ( ! $thumbnail_id ) {
			return '';
		}

		$image = wp_get_attachment_image_src( $thumbnail_id, $size );

		if ( $image && isset( $image[0] ) ) {
			return $image[0];
		}

		return '';
	}

	/**
	 * Get the featured image dimensions.
	 *
	 * @since 1.0.0
	 * @param int    $post_id Post ID.
	 * @param string $size    Image size. Default 'large'.
	 * @return array Array with 'width' and 'height' or empty array.
	 */
	public function get_og_image_dimensions( $post_id, $size = 'large' ) {
		$thumbnail_id = get_post_thumbnail_id( $post_id );

		if ( ! $thumbnail_id ) {
			return array();
		}

		$image = wp_get_attachment_image_src( $thumbnail_id, $size );

		if ( $image && isset( $image[1] ) && isset( $image[2] ) ) {
			return array(
				'width'  => $image[1],
				'height' => $image[2],
			);
		}

		return array();
	}
}
