<?php
/**
 * Usage tracking and limits.
 *
 * Handles tracking of optimization usage for free tier limits.
 * Free users get 25 optimizations per calendar month.
 * Pro users have unlimited optimizations.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Usage tracking class.
 *
 * @since 1.0.0
 */
class TopRanker_Usage {

	/**
	 * Option name for usage data.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	const OPTION_NAME = 'topranker_usage';

	/**
	 * Free tier monthly limit.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const FREE_LIMIT = 25;

	/**
	 * Warning threshold (show warning when this many used).
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const WARNING_THRESHOLD = 20;

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		// Check and reset usage on every page load if needed.
		$this->maybe_reset_usage();
	}

	/**
	 * Check if the user can perform an optimization.
	 *
	 * Pro users always return true. Free users are limited to FREE_LIMIT per month.
	 *
	 * @since  1.0.0
	 * @return bool True if the user can optimize, false if limit reached.
	 */
	public function can_optimize() {
		// Pro users have unlimited optimizations.
		if ( topranker_is_pro() ) {
			return true;
		}

		$usage = $this->get_usage();
		return $usage['count'] < self::FREE_LIMIT;
	}

	/**
	 * Increment the usage counter.
	 *
	 * Should be called after each successful optimization.
	 * Does nothing for Pro users (they don't need tracking).
	 *
	 * @since  1.0.0
	 * @return bool True if incremented, false if at limit or Pro user.
	 */
	public function increment() {
		// Pro users don't need tracking.
		if ( topranker_is_pro() ) {
			return true;
		}

		// Use a transient lock to prevent race conditions from concurrent requests.
		$lock_key = 'topranker_usage_lock';
		if ( get_transient( $lock_key ) ) {
			// Another request is incrementing; re-read usage to get latest state.
			usleep( 50000 ); // 50ms.
			$usage = $this->get_usage();
			return $usage['count'] < self::FREE_LIMIT;
		}
		set_transient( $lock_key, 1, 5 );

		$usage = $this->get_usage();

		// Check if already at limit.
		if ( $usage['count'] >= self::FREE_LIMIT ) {
			delete_transient( $lock_key );
			return false;
		}

		// Increment count.
		$usage['count'] = $usage['count'] + 1;

		$result = update_option( self::OPTION_NAME, $usage );
		delete_transient( $lock_key );

		return $result;
	}

	/**
	 * Get current usage data.
	 *
	 * @since  1.0.0
	 * @return array {
	 *     Usage data array.
	 *
	 *     @type int    $count      Number of optimizations used this month.
	 *     @type string $month      Current month in Y-m format.
	 *     @type string $reset_date Date when counter resets (Y-m-d format).
	 * }
	 */
	public function get_usage() {
		$default = $this->get_default_usage();
		$usage   = get_option( self::OPTION_NAME, $default );

		// Ensure usage is an array with all required keys.
		if ( ! is_array( $usage ) ) {
			$usage = $default;
		}

		$usage = wp_parse_args( $usage, $default );

		return $usage;
	}

	/**
	 * Get remaining optimizations for the current month.
	 *
	 * @since  1.0.0
	 * @return int|string Number remaining, or 'unlimited' for Pro users.
	 */
	public function get_remaining() {
		if ( topranker_is_pro() ) {
			return 'unlimited';
		}

		$usage     = $this->get_usage();
		$remaining = self::FREE_LIMIT - $usage['count'];

		return max( 0, $remaining );
	}

	/**
	 * Get the monthly limit.
	 *
	 * @since  1.0.0
	 * @return int|string The limit, or 'unlimited' for Pro users.
	 */
	public function get_limit() {
		if ( topranker_is_pro() ) {
			return 'unlimited';
		}

		return self::FREE_LIMIT;
	}

	/**
	 * Check if the user is approaching the limit (warning threshold).
	 *
	 * @since  1.0.0
	 * @return bool True if at or above warning threshold.
	 */
	public function is_approaching_limit() {
		if ( topranker_is_pro() ) {
			return false;
		}

		$usage = $this->get_usage();
		return $usage['count'] >= self::WARNING_THRESHOLD;
	}

	/**
	 * Check if the limit has been reached.
	 *
	 * @since  1.0.0
	 * @return bool True if limit reached.
	 */
	public function is_limit_reached() {
		if ( topranker_is_pro() ) {
			return false;
		}

		$usage = $this->get_usage();
		return $usage['count'] >= self::FREE_LIMIT;
	}

	/**
	 * Get formatted usage stats for display.
	 *
	 * @since  1.0.0
	 * @return array {
	 *     Formatted usage statistics.
	 *
	 *     @type int          $used       Number of optimizations used.
	 *     @type int|string   $limit      Monthly limit or 'unlimited'.
	 *     @type int|string   $remaining  Remaining optimizations or 'unlimited'.
	 *     @type string       $reset_date Human-readable reset date.
	 *     @type bool         $is_pro     Whether user is Pro.
	 *     @type bool         $warning    Whether to show warning.
	 *     @type bool         $at_limit   Whether limit is reached.
	 * }
	 */
	public function get_stats() {
		$usage    = $this->get_usage();
		$is_pro   = topranker_is_pro();
		$at_limit = $this->is_limit_reached();

		// Format reset date for display.
		$reset_timestamp = strtotime( $usage['reset_date'] );
		$reset_formatted = wp_date( 'F j', $reset_timestamp );

		return array(
			'used'       => $is_pro ? 0 : $usage['count'],
			'limit'      => $this->get_limit(),
			'remaining'  => $this->get_remaining(),
			'reset_date' => $reset_formatted,
			'is_pro'     => $is_pro,
			'warning'    => $this->is_approaching_limit(),
			'at_limit'   => $at_limit,
		);
	}

	/**
	 * Get the limit reached message.
	 *
	 * @since  1.0.0
	 * @return string The message to display when limit is reached.
	 */
	public function get_limit_message() {
		$stats = $this->get_stats();

		return sprintf(
			/* translators: 1: number used, 2: limit, 3: reset date */
			__( "You've used %1\$d/%2\$d free optimizations this month. Resets on %3\$s. Upgrade to Pro for unlimited.", 'topranker-ai' ),
			$stats['used'],
			$stats['limit'],
			$stats['reset_date']
		);
	}

	/**
	 * Get the warning message.
	 *
	 * @since  1.0.0
	 * @return string The warning message when approaching limit.
	 */
	public function get_warning_message() {
		$remaining = $this->get_remaining();

		return sprintf(
			/* translators: %d: number of optimizations remaining */
			_n(
				'%d optimization remaining this month.',
				'%d optimizations remaining this month.',
				$remaining,
				'topranker-ai'
			),
			$remaining
		);
	}

	/**
	 * Check if usage should be reset and reset if needed.
	 *
	 * Compares current month with stored month. If different, resets counter.
	 * This is comparison-based, not cron-based, for reliability.
	 *
	 * @since  1.0.0
	 * @return bool True if reset was performed, false otherwise.
	 */
	private function maybe_reset_usage() {
		$usage         = $this->get_usage();
		$current_month = gmdate( 'Y-m' );

		// If the stored month is different from current, reset.
		if ( $usage['month'] !== $current_month ) {
			$new_usage = $this->get_default_usage();
			update_option( self::OPTION_NAME, $new_usage );
			return true;
		}

		return false;
	}

	/**
	 * Get default usage data for a new month.
	 *
	 * @since  1.0.0
	 * @return array Default usage data.
	 */
	private function get_default_usage() {
		$current_month = gmdate( 'Y-m' );
		$next_month    = gmdate( 'Y-m-01', strtotime( 'first day of next month' ) );

		return array(
			'count'      => 0,
			'month'      => $current_month,
			'reset_date' => $next_month,
		);
	}

	/**
	 * Manually reset usage (for testing or admin override).
	 *
	 * @since  1.0.0
	 * @return bool True on success.
	 */
	public function reset() {
		return update_option( self::OPTION_NAME, $this->get_default_usage() );
	}

	/**
	 * Get usage data for REST API response.
	 *
	 * @since  1.0.0
	 * @return array API-formatted usage data.
	 */
	public function get_api_response() {
		$stats = $this->get_stats();

		return array(
			'used'        => $stats['used'],
			'limit'       => $stats['limit'],
			'remaining'   => $stats['remaining'],
			'resets_on'   => $stats['reset_date'],
			'is_pro'      => $stats['is_pro'],
			'can_optimize' => $this->can_optimize(),
		);
	}
}
