<?php
/**
 * Image alt tag generation using OpenAI Vision.
 *
 * Handles generation of SEO-optimized alt text for images in posts using
 * OpenAI's vision capabilities. Pro feature only.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * TopRanker Alt Tags class.
 *
 * @since 1.0.0
 */
class TopRanker_Alt_Tags {

	/**
	 * Maximum alt text length.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const ALT_TEXT_MAX_LENGTH = 125;

	/**
	 * API instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_API|null
	 */
	private $api = null;

	/**
	 * Optimizer instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_Optimizer|null
	 */
	private $optimizer = null;

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		// Classes will be initialized lazily when needed.
	}

	/**
	 * Get the API instance.
	 *
	 * @since  1.0.0
	 * @return TopRanker_API
	 */
	private function get_api() {
		if ( null === $this->api ) {
			$this->api = new TopRanker_API();
		}
		return $this->api;
	}

	/**
	 * Get the optimizer instance.
	 *
	 * @since  1.0.0
	 * @return TopRanker_Optimizer
	 */
	private function get_optimizer() {
		if ( null === $this->optimizer ) {
			$this->optimizer = new TopRanker_Optimizer();
		}
		return $this->optimizer;
	}

	/**
	 * Generate alt tags for all images in a post.
	 *
	 * Processes images sequentially to avoid timeouts.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post           Post ID or WP_Post object.
	 * @param string      $focus_keyphrase Optional. Focus keyphrase to incorporate.
	 * @return array|WP_Error Array of generated alt tags or WP_Error on failure.
	 */
	public function generate_alt_tags( $post, $focus_keyphrase = '' ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return new WP_Error(
				'invalid_post',
				__( 'Invalid post.', 'topranker-ai' )
			);
		}

		$optimizer = $this->get_optimizer();
		$images    = $optimizer->get_post_images( $post );

		if ( empty( $images ) ) {
			return new WP_Error(
				'no_images',
				__( 'No images found in the post.', 'topranker-ai' )
			);
		}

		// Get content context for alt text generation.
		$content_context = $optimizer->prepare_content( $post, 2000 );
		$context_prefix  = $optimizer->build_context_prefix( $post );

		$results = array();
		$errors  = array();

		// Process images sequentially to avoid timeouts.
		foreach ( $images as $image ) {
			$image_result = $this->generate_single_alt_tag(
				$image,
				$content_context,
				$focus_keyphrase,
				$context_prefix
			);

			if ( is_wp_error( $image_result ) ) {
				$errors[] = array(
					'image_id' => isset( $image['id'] ) ? $image['id'] : 0,
					'src'      => isset( $image['src'] ) ? $image['src'] : '',
					'error'    => $image_result->get_error_message(),
				);
			} else {
				$results[] = $image_result;
			}
		}

		if ( empty( $results ) && ! empty( $errors ) ) {
			return new WP_Error(
				'all_failed',
				__( 'Failed to generate alt text for all images.', 'topranker-ai' ),
				array( 'errors' => $errors )
			);
		}

		return array(
			'post_id' => $post->ID,
			'results' => $results,
			'errors'  => $errors,
			'total'   => count( $images ),
			'success' => count( $results ),
			'failed'  => count( $errors ),
		);
	}

	/**
	 * Generate alt tag for a single image.
	 *
	 * @since 1.0.0
	 * @param array  $image           Image data array with src, alt, id, caption.
	 * @param string $content_context Post content context.
	 * @param string $focus_keyphrase Focus keyphrase if provided.
	 * @param string $context_prefix  Context prefix for prompts.
	 * @return array|WP_Error Generated alt tag data or WP_Error.
	 */
	public function generate_single_alt_tag( $image, $content_context = '', $focus_keyphrase = '', $context_prefix = '' ) {
		$api       = $this->get_api();
		$image_url = isset( $image['src'] ) ? $image['src'] : '';
		$image_id  = isset( $image['id'] ) ? (int) $image['id'] : 0;
		$caption   = isset( $image['caption'] ) ? $image['caption'] : '';
		$old_alt   = isset( $image['alt'] ) ? $image['alt'] : '';

		if ( empty( $image_url ) ) {
			return new WP_Error(
				'no_image_url',
				__( 'Image URL is missing.', 'topranker-ai' )
			);
		}

		// Check if image URL is publicly accessible.
		$is_public_url = $this->is_public_url( $image_url );

		if ( $is_public_url ) {
			// Use vision API with the actual image.
			$alt_text = $this->generate_alt_with_vision( $image_url, $content_context, $focus_keyphrase, $caption, $context_prefix );
		} else {
			// Fall back to context-only generation.
			$alt_text = $this->generate_alt_from_context( $image_url, $content_context, $focus_keyphrase, $caption, $context_prefix );
		}

		if ( is_wp_error( $alt_text ) ) {
			return $alt_text;
		}

		return array(
			'image_id'       => $image_id,
			'src'            => $image_url,
			'old_alt'        => $old_alt,
			'new_alt'        => $alt_text,
			'caption'        => $caption,
			'used_vision'    => $is_public_url,
			'is_featured'    => isset( $image['is_featured'] ) ? (bool) $image['is_featured'] : false,
		);
	}

	/**
	 * Generate alt text using OpenAI Vision API.
	 *
	 * @since 1.0.0
	 * @param string $image_url       URL of the image.
	 * @param string $content_context Post content context.
	 * @param string $focus_keyphrase Focus keyphrase if provided.
	 * @param string $caption         Image caption if available.
	 * @param string $context_prefix  Context prefix for prompts.
	 * @return string|WP_Error Generated alt text or WP_Error.
	 */
	private function generate_alt_with_vision( $image_url, $content_context, $focus_keyphrase, $caption, $context_prefix ) {
		$api    = $this->get_api();
		$prompt = $this->build_vision_alt_prompt( $content_context, $focus_keyphrase, $caption, $context_prefix );

		$response = $api->chat_completion_with_vision( $prompt, $image_url );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		// Parse the response.
		$alt_text = $this->parse_alt_response( $response['content'] );

		if ( is_wp_error( $alt_text ) ) {
			// Retry once.
			$response = $api->chat_completion_with_vision( $prompt, $image_url );

			if ( is_wp_error( $response ) ) {
				return $response;
			}

			$alt_text = $this->parse_alt_response( $response['content'] );

			if ( is_wp_error( $alt_text ) ) {
				return $alt_text;
			}
		}

		return $alt_text;
	}

	/**
	 * Generate alt text from context only (fallback when image URL is not public).
	 *
	 * @since 1.0.0
	 * @param string $image_url       URL of the image.
	 * @param string $content_context Post content context.
	 * @param string $focus_keyphrase Focus keyphrase if provided.
	 * @param string $caption         Image caption if available.
	 * @param string $context_prefix  Context prefix for prompts.
	 * @return string|WP_Error Generated alt text or WP_Error.
	 */
	private function generate_alt_from_context( $image_url, $content_context, $focus_keyphrase, $caption, $context_prefix ) {
		$api    = $this->get_api();
		$prompt = $this->build_context_only_alt_prompt( $image_url, $content_context, $focus_keyphrase, $caption, $context_prefix );

		$messages = array(
			array(
				'role'    => 'system',
				'content' => $context_prefix,
			),
			array(
				'role'    => 'user',
				'content' => $prompt,
			),
		);

		$response = $api->chat_completion( $messages );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		// Parse the response.
		$alt_text = $this->parse_alt_response( $response['content'] );

		if ( is_wp_error( $alt_text ) ) {
			// Retry once.
			$response = $api->chat_completion( $messages );

			if ( is_wp_error( $response ) ) {
				return $response;
			}

			$alt_text = $this->parse_alt_response( $response['content'] );

			if ( is_wp_error( $alt_text ) ) {
				return $alt_text;
			}
		}

		return $alt_text;
	}

	/**
	 * Build the vision alt text prompt.
	 *
	 * @since 1.0.0
	 * @param string $content_context Post content context.
	 * @param string $focus_keyphrase Focus keyphrase if provided.
	 * @param string $caption         Image caption if available.
	 * @param string $context_prefix  Context prefix.
	 * @return string The formatted prompt.
	 */
	private function build_vision_alt_prompt( $content_context, $focus_keyphrase, $caption, $context_prefix ) {
		$keyphrase_instruction = '';
		if ( ! empty( $focus_keyphrase ) ) {
			$keyphrase_instruction = sprintf(
				/* translators: %s: focus keyphrase */
				__( 'If relevant and natural, include the focus keyphrase "%s" in the alt text.', 'topranker-ai' ),
				$focus_keyphrase
			);
		}

		$caption_instruction = '';
		if ( ! empty( $caption ) ) {
			$caption_instruction = sprintf(
				/* translators: %s: image caption */
				__( 'The image has this caption: "%s"', 'topranker-ai' ),
				$caption
			);
		}

		$content_instruction = '';
		if ( ! empty( $content_context ) ) {
			$content_instruction = sprintf(
				/* translators: %s: content context */
				__( 'The image appears in an article with this context: %s', 'topranker-ai' ),
				mb_substr( $content_context, 0, 500 )
			);
		}

		$prompt = sprintf(
			/* translators: 1: Max length, 2: Keyphrase instruction, 3: Caption instruction, 4: Content instruction */
			__(
				'Generate SEO-optimized alt text for this image.

Requirements:
- Maximum %1$d characters (strict limit)
- Describe what is ACTUALLY visible in the image
- Be specific and descriptive (not "a person" but "woman reviewing financial charts at a desk")
- Do NOT start with "image of", "photo of", "picture of", or similar phrases
- Use natural language that reads well
%2$s
%3$s
%4$s

Respond with valid JSON only, in this exact format:
{
  "alt": "The generated alt text here"
}',
				'topranker-ai'
			),
			self::ALT_TEXT_MAX_LENGTH,
			$keyphrase_instruction,
			$caption_instruction,
			$content_instruction
		);

		return $prompt;
	}

	/**
	 * Build the context-only alt text prompt (fallback).
	 *
	 * @since 1.0.0
	 * @param string $image_url       URL of the image.
	 * @param string $content_context Post content context.
	 * @param string $focus_keyphrase Focus keyphrase if provided.
	 * @param string $caption         Image caption if available.
	 * @param string $context_prefix  Context prefix.
	 * @return string The formatted prompt.
	 */
	private function build_context_only_alt_prompt( $image_url, $content_context, $focus_keyphrase, $caption, $context_prefix ) {
		$keyphrase_instruction = '';
		if ( ! empty( $focus_keyphrase ) ) {
			$keyphrase_instruction = sprintf(
				/* translators: %s: focus keyphrase */
				__( 'If relevant and natural, include the focus keyphrase "%s" in the alt text.', 'topranker-ai' ),
				$focus_keyphrase
			);
		}

		$caption_instruction = '';
		if ( ! empty( $caption ) ) {
			$caption_instruction = sprintf(
				/* translators: %s: image caption */
				__( 'The image has this caption: "%s"', 'topranker-ai' ),
				$caption
			);
		}

		// Try to extract filename for context.
		$filename = basename( wp_parse_url( $image_url, PHP_URL_PATH ) );
		$filename = preg_replace( '/[-_]/', ' ', pathinfo( $filename, PATHINFO_FILENAME ) );
		$filename = preg_replace( '/\d+x\d+$/', '', $filename ); // Remove size suffix.
		$filename = trim( $filename );

		$prompt = sprintf(
			/* translators: 1: Max length, 2: Content context, 3: Filename, 4: Keyphrase instruction, 5: Caption instruction */
			__(
				'Generate SEO-optimized alt text for an image. Note: The image cannot be directly analyzed, so base the alt text on the available context.

Image filename: %3$s

Article context:
%2$s

Requirements:
- Maximum %1$d characters (strict limit)
- Create descriptive alt text based on the context and filename
- Be specific and relevant to the article topic
- Do NOT start with "image of", "photo of", "picture of", or similar phrases
- Use natural language that reads well
%4$s
%5$s

Note: Since the image cannot be viewed directly, generate contextually appropriate alt text that would make sense for an image in this article.

Respond with valid JSON only, in this exact format:
{
  "alt": "The generated alt text here"
}',
				'topranker-ai'
			),
			self::ALT_TEXT_MAX_LENGTH,
			mb_substr( $content_context, 0, 500 ),
			$filename,
			$keyphrase_instruction,
			$caption_instruction
		);

		return $prompt;
	}

	/**
	 * Parse the alt text response from AI.
	 *
	 * @since 1.0.0
	 * @param string $content Raw response content.
	 * @return string|WP_Error Parsed alt text or WP_Error.
	 */
	private function parse_alt_response( $content ) {
		$api    = $this->get_api();
		$parsed = $api->parse_json_response( $content );

		if ( is_wp_error( $parsed ) ) {
			// Try to extract alt text directly if JSON parsing fails.
			$content = trim( $content );
			$content = preg_replace( '/^["\']|["\']$/', '', $content );

			if ( ! empty( $content ) && mb_strlen( $content ) <= self::ALT_TEXT_MAX_LENGTH * 1.5 ) {
				return $this->sanitize_alt_text( $content );
			}

			return $parsed;
		}

		if ( ! isset( $parsed['alt'] ) || ! is_string( $parsed['alt'] ) ) {
			return new WP_Error(
				'invalid_response',
				__( 'Invalid response format from AI.', 'topranker-ai' )
			);
		}

		return $this->sanitize_alt_text( $parsed['alt'] );
	}

	/**
	 * Sanitize and validate alt text.
	 *
	 * @since 1.0.0
	 * @param string $alt_text Raw alt text.
	 * @return string Sanitized alt text.
	 */
	private function sanitize_alt_text( $alt_text ) {
		$alt_text = sanitize_text_field( $alt_text );

		// Remove common prefix phrases.
		$prefixes = array(
			'image of ',
			'photo of ',
			'picture of ',
			'photograph of ',
			'an image of ',
			'a photo of ',
			'a picture of ',
			'a photograph of ',
		);

		foreach ( $prefixes as $prefix ) {
			if ( 0 === stripos( $alt_text, $prefix ) ) {
				$alt_text = mb_substr( $alt_text, mb_strlen( $prefix ) );
				$alt_text = ucfirst( $alt_text );
				break;
			}
		}

		// Trim if too long.
		if ( mb_strlen( $alt_text ) > self::ALT_TEXT_MAX_LENGTH ) {
			$alt_text = $this->smart_truncate( $alt_text, self::ALT_TEXT_MAX_LENGTH );
		}

		return $alt_text;
	}

	/**
	 * Smart truncate text at word boundary.
	 *
	 * @since 1.0.0
	 * @param string $text   Text to truncate.
	 * @param int    $length Maximum length.
	 * @return string Truncated text.
	 */
	private function smart_truncate( $text, $length ) {
		if ( mb_strlen( $text ) <= $length ) {
			return $text;
		}

		// Truncate to length.
		$truncated = mb_substr( $text, 0, $length );

		// Try to break at word boundary.
		$last_space = mb_strrpos( $truncated, ' ' );

		if ( false !== $last_space && $last_space > $length * 0.8 ) {
			$truncated = mb_substr( $truncated, 0, $last_space );
		}

		return trim( $truncated );
	}

	/**
	 * Check if a URL is publicly accessible.
	 *
	 * @since 1.0.0
	 * @param string $url URL to check.
	 * @return bool True if URL is public.
	 */
	private function is_public_url( $url ) {
		// Check for localhost or local development URLs.
		$local_patterns = array(
			'localhost',
			'127.0.0.1',
			'.local',
			'.test',
			'.dev',
			'.localhost',
			'192.168.',
			'10.0.',
			'172.16.',
			'172.17.',
			'172.18.',
			'172.19.',
			'172.20.',
			'172.21.',
			'172.22.',
			'172.23.',
			'172.24.',
			'172.25.',
			'172.26.',
			'172.27.',
			'172.28.',
			'172.29.',
			'172.30.',
			'172.31.',
		);

		$parsed = wp_parse_url( $url );
		$host   = isset( $parsed['host'] ) ? $parsed['host'] : '';

		foreach ( $local_patterns as $pattern ) {
			if ( false !== strpos( $host, $pattern ) ) {
				return false;
			}
		}

		// Check if URL uses HTTPS or HTTP (required for OpenAI).
		$scheme = isset( $parsed['scheme'] ) ? $parsed['scheme'] : '';
		if ( ! in_array( $scheme, array( 'http', 'https' ), true ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Apply generated alt tags to attachments and post content.
	 *
	 * Updates both the attachment meta (_wp_attachment_image_alt) and the
	 * alt attributes in the post content HTML so SEO audits and search
	 * engines see the updated alt text.
	 *
	 * @since 1.0.0
	 * @param array $results Array of generated alt tag results.
	 * @param int   $post_id Optional. Post ID to update content alt attributes.
	 * @return array Results with applied status.
	 */
	public function apply_alt_tags( $results, $post_id = 0 ) {
		if ( ! is_array( $results ) || empty( $results ) ) {
			return array(
				'applied' => 0,
				'failed'  => 0,
				'results' => array(),
			);
		}

		$applied = 0;
		$failed  = 0;
		$applied_results = array();

		foreach ( $results as $result ) {
			$image_id = isset( $result['image_id'] ) ? (int) $result['image_id'] : 0;
			$new_alt  = isset( $result['new_alt'] ) ? $result['new_alt'] : '';

			if ( empty( $new_alt ) ) {
				++$failed;
				$result['applied'] = false;
				$applied_results[] = $result;
				continue;
			}

			// Update attachment meta when we have a WordPress attachment ID.
			if ( $image_id > 0 ) {
				update_post_meta( $image_id, '_wp_attachment_image_alt', $new_alt );
			}

			// Mark as applied — post_content HTML will be updated below.
			++$applied;
			$result['applied'] = true;
			$applied_results[] = $result;
		}

		// Update alt attributes in post content HTML.
		if ( $post_id > 0 && $applied > 0 ) {
			$this->update_post_content_alts( $post_id, $applied_results );
		}

		return array(
			'applied' => $applied,
			'failed'  => $failed,
			'results' => $applied_results,
		);
	}

	/**
	 * Update alt attributes in post content HTML.
	 *
	 * Replaces the alt attribute value on img tags in the saved post content
	 * so that SEO audits and search engine crawlers see the updated alt text.
	 * Uses wp-image-{id} class matching first, falls back to src URL matching.
	 *
	 * @since 1.1.0
	 * @param int   $post_id Post ID.
	 * @param array $results Applied alt tag results.
	 */
	private function update_post_content_alts( $post_id, $results ) {
		$post = get_post( $post_id );
		if ( ! $post || empty( $post->post_content ) ) {
			return;
		}

		$content = $post->post_content;
		$updated = false;

		foreach ( $results as $result ) {
			if ( empty( $result['applied'] ) || empty( $result['new_alt'] ) ) {
				continue;
			}

			// Skip featured images — they are not in post_content.
			if ( ! empty( $result['is_featured'] ) ) {
				continue;
			}

			$image_id = isset( $result['image_id'] ) ? (int) $result['image_id'] : 0;
			$src      = isset( $result['src'] ) ? $result['src'] : '';
			$new_alt  = esc_attr( $result['new_alt'] );
			$matched  = false;

			// Match by wp-image-{id} class. Use a two-step approach:
			// 1. Match the full <img> tag containing wp-image-{id} (any attribute order).
			// 2. Replace the alt attribute within the matched tag.
			if ( $image_id > 0 ) {
				$content = preg_replace_callback(
					'/<img\b[^>]*\bclass=["\'][^"\']*\bwp-image-' . $image_id . '\b[^"\']*["\'][^>]*\/?>/i',
					function ( $matches ) use ( $new_alt ) {
						$tag = $matches[0];
						// Replace existing alt attribute.
						$replaced = preg_replace( '/\balt=["\'][^"\']*["\']/', 'alt="' . $new_alt . '"', $tag, 1, $alt_count );
						if ( $alt_count > 0 ) {
							return $replaced;
						}
						// No alt attribute found — insert one before the closing.
						return preg_replace( '/\s*\/?>$/', ' alt="' . $new_alt . '" />', $tag );
					},
					$content,
					1,
					$count
				);

				if ( $count > 0 ) {
					$updated = true;
					$matched = true;
				}
			}

			// Fall back to matching by src URL (same two-step approach).
			if ( ! $matched && ! empty( $src ) ) {
				$escaped_src = preg_quote( $src, '/' );
				$content = preg_replace_callback(
					'/<img\b[^>]*\bsrc=["\']' . $escaped_src . '["\'][^>]*\/?>/i',
					function ( $matches ) use ( $new_alt ) {
						$tag = $matches[0];
						$replaced = preg_replace( '/\balt=["\'][^"\']*["\']/', 'alt="' . $new_alt . '"', $tag, 1, $alt_count );
						if ( $alt_count > 0 ) {
							return $replaced;
						}
						return preg_replace( '/\s*\/?>$/', ' alt="' . $new_alt . '" />', $tag );
					},
					$content,
					1,
					$count
				);

				if ( $count > 0 ) {
					$updated = true;
				}
			}
		}

		if ( $updated ) {
			// Use wpdb directly to avoid triggering save_post hooks and revisions.
			global $wpdb;
			$wpdb->update(
				$wpdb->posts,
				array( 'post_content' => $content ),
				array( 'ID' => $post_id ),
				array( '%s' ),
				array( '%d' )
			);
			clean_post_cache( $post_id );
		}
	}

	/**
	 * Generate alt tag for a single attachment (Media Library).
	 *
	 * @since 1.0.0
	 * @param int    $attachment_id   Attachment ID.
	 * @param string $focus_keyphrase Optional. Focus keyphrase.
	 * @return array|WP_Error Generated alt tag data or WP_Error.
	 */
	public function generate_for_attachment( $attachment_id, $focus_keyphrase = '' ) {
		$attachment = get_post( $attachment_id );

		if ( ! $attachment || 'attachment' !== $attachment->post_type ) {
			return new WP_Error(
				'invalid_attachment',
				__( 'Invalid attachment.', 'topranker-ai' )
			);
		}

		// Check if it's an image.
		if ( ! wp_attachment_is_image( $attachment_id ) ) {
			return new WP_Error(
				'not_image',
				__( 'Attachment is not an image.', 'topranker-ai' )
			);
		}

		$image_url = wp_get_attachment_url( $attachment_id );
		$caption   = $attachment->post_excerpt;
		$old_alt   = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );

		// Get parent post context if available.
		$content_context = '';
		$context_prefix  = '';
		$parent_id       = $attachment->post_parent;

		if ( $parent_id > 0 ) {
			$parent = get_post( $parent_id );
			if ( $parent ) {
				$optimizer       = $this->get_optimizer();
				$content_context = $optimizer->prepare_content( $parent, 2000 );
				$context_prefix  = $optimizer->build_context_prefix( $parent );

				// Get focus keyphrase from parent if not provided.
				if ( empty( $focus_keyphrase ) ) {
					$focus_keyphrase = get_post_meta( $parent_id, '_topranker_focus_keyphrase', true );
				}
			}
		}

		if ( empty( $context_prefix ) ) {
			$optimizer      = $this->get_optimizer();
			$context_prefix = $optimizer->build_context_prefix();
		}

		$image = array(
			'id'      => $attachment_id,
			'src'     => $image_url,
			'alt'     => $old_alt,
			'caption' => $caption,
		);

		return $this->generate_single_alt_tag( $image, $content_context, $focus_keyphrase, $context_prefix );
	}

	/**
	 * Bulk generate alt tags for multiple attachments (Media Library).
	 *
	 * @since 1.0.0
	 * @param array $attachment_ids Array of attachment IDs.
	 * @return array Results of bulk generation.
	 */
	public function bulk_generate_for_attachments( $attachment_ids ) {
		if ( ! is_array( $attachment_ids ) || empty( $attachment_ids ) ) {
			return array(
				'total'   => 0,
				'success' => 0,
				'failed'  => 0,
				'results' => array(),
				'errors'  => array(),
			);
		}

		$results = array();
		$errors  = array();

		foreach ( $attachment_ids as $attachment_id ) {
			$result = $this->generate_for_attachment( (int) $attachment_id );

			if ( is_wp_error( $result ) ) {
				$errors[] = array(
					'attachment_id' => $attachment_id,
					'error'         => $result->get_error_message(),
				);
			} else {
				$results[] = $result;
			}
		}

		return array(
			'total'   => count( $attachment_ids ),
			'success' => count( $results ),
			'failed'  => count( $errors ),
			'results' => $results,
			'errors'  => $errors,
		);
	}

	/**
	 * Get images in post that are missing alt tags.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post Post ID or WP_Post object.
	 * @return array Array of images without alt tags.
	 */
	public function get_images_missing_alt( $post ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return array();
		}

		$optimizer = $this->get_optimizer();
		$images    = $optimizer->get_post_images( $post );

		$missing = array();

		foreach ( $images as $image ) {
			$alt = isset( $image['alt'] ) ? trim( $image['alt'] ) : '';

			// Check attachment meta if we have an ID.
			if ( empty( $alt ) && ! empty( $image['id'] ) ) {
				$alt = get_post_meta( $image['id'], '_wp_attachment_image_alt', true );
			}

			if ( empty( $alt ) ) {
				$missing[] = $image;
			}
		}

		return $missing;
	}

	/**
	 * Count images missing alt tags in a post.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post Post ID or WP_Post object.
	 * @return int Count of images missing alt tags.
	 */
	public function count_images_missing_alt( $post ) {
		return count( $this->get_images_missing_alt( $post ) );
	}

	/**
	 * Check if a post has all images with alt tags.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post Post ID or WP_Post object.
	 * @return bool True if all images have alt tags.
	 */
	public function has_all_alt_tags( $post ) {
		return 0 === $this->count_images_missing_alt( $post );
	}
}
