<?php
/**
 * Bulk Optimization (Pro).
 *
 * Handles bulk optimization operations for multiple posts at once.
 * Provides post selection, progress tracking, and batch processing.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * TopRanker Bulk class.
 *
 * @since 1.0.0
 */
class TopRanker_Bulk {

	/**
	 * Maximum posts per batch.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const MAX_BATCH_SIZE = 100;

	/**
	 * Posts per page for pagination.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const POSTS_PER_PAGE = 50;

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		// Class is loaded but hooks are initialized in init method.
	}

	/**
	 * Get posts for bulk optimization table.
	 *
	 * @since  1.0.0
	 * @param  array $args Query arguments.
	 * @return array Posts data with pagination info.
	 */
	public function get_posts( $args = array() ) {
		$defaults = array(
			'post_type'      => 'post',
			'posts_per_page' => self::POSTS_PER_PAGE,
			'paged'          => 1,
			'orderby'        => 'date',
			'order'          => 'DESC',
			'filter'         => 'all', // 'all', 'missing_meta', 'missing_alt'.
			's'              => '',
		);

		$args = wp_parse_args( $args, $defaults );

		// Validate post type is enabled.
		$enabled_types = $this->get_enabled_post_types();
		if ( ! in_array( $args['post_type'], $enabled_types, true ) ) {
			$args['post_type'] = ! empty( $enabled_types ) ? $enabled_types[0] : 'post';
		}

		// Build query args.
		$query_args = array(
			'post_type'      => $args['post_type'],
			'post_status'    => 'publish',
			'posts_per_page' => min( $args['posts_per_page'], self::POSTS_PER_PAGE ),
			'paged'          => max( 1, intval( $args['paged'] ) ),
			'orderby'        => $args['orderby'],
			'order'          => $args['order'],
		);

		// Search.
		if ( ! empty( $args['s'] ) ) {
			$query_args['s'] = sanitize_text_field( $args['s'] );
		}

		// Filter by meta.
		if ( 'missing_meta' === $args['filter'] ) {
			$query_args['meta_query'] = array(
				'relation' => 'OR',
				array(
					'key'     => '_topranker_meta_title',
					'compare' => 'NOT EXISTS',
				),
				array(
					'key'     => '_topranker_meta_title',
					'value'   => '',
					'compare' => '=',
				),
			);
		} elseif ( 'missing_alt' === $args['filter'] ) {
			// This is handled differently - need posts with images missing alt.
			// For simplicity, just return all posts and check alt tags in template.
		}

		$query = new WP_Query( $query_args );

		$posts = array();
		if ( $query->have_posts() ) {
			while ( $query->have_posts() ) {
				$query->the_post();
				$post = get_post();
				$posts[] = $this->prepare_post_data( $post );
			}
			wp_reset_postdata();
		}

		return array(
			'posts'         => $posts,
			'total_posts'   => $query->found_posts,
			'total_pages'   => $query->max_num_pages,
			'current_page'  => $args['paged'],
			'posts_per_page' => $args['posts_per_page'],
			'post_type'     => $args['post_type'],
			'filter'        => $args['filter'],
		);
	}

	/**
	 * Prepare post data for display.
	 *
	 * @since  1.0.0
	 * @param  WP_Post $post Post object.
	 * @return array Post data array.
	 */
	private function prepare_post_data( $post ) {
		$meta_title       = get_post_meta( $post->ID, '_topranker_meta_title', true );
		$meta_description = get_post_meta( $post->ID, '_topranker_meta_description', true );
		$focus_keyphrase  = get_post_meta( $post->ID, '_topranker_focus_keyphrase', true );
		$seo_score        = get_post_meta( $post->ID, '_topranker_seo_score', true );
		$last_optimized   = get_post_meta( $post->ID, '_topranker_last_optimized', true );

		// Check for alt tags.
		$images_missing_alt = $this->count_images_missing_alt( $post );

		// Determine SEO status.
		$has_meta_title = ! empty( $meta_title );
		$has_meta_desc  = ! empty( $meta_description );
		$has_keyphrase  = ! empty( $focus_keyphrase );

		if ( $has_meta_title && $has_meta_desc && $has_keyphrase ) {
			$seo_status = 'complete';
		} elseif ( $has_meta_title || $has_meta_desc || $has_keyphrase ) {
			$seo_status = 'partial';
		} else {
			$seo_status = 'missing';
		}

		return array(
			'ID'                 => $post->ID,
			'title'              => get_the_title( $post ),
			'post_type'          => $post->post_type,
			'edit_link'          => get_edit_post_link( $post->ID, 'raw' ),
			'has_meta_title'     => $has_meta_title,
			'has_meta_desc'      => $has_meta_desc,
			'has_keyphrase'      => $has_keyphrase,
			'seo_status'         => $seo_status,
			'seo_score'          => $seo_score ? intval( $seo_score ) : null,
			'images_missing_alt' => $images_missing_alt,
			'last_optimized'     => $last_optimized ? $this->format_time_ago( $last_optimized ) : null,
		);
	}

	/**
	 * Count images missing alt text in a post.
	 *
	 * @since  1.0.0
	 * @param  WP_Post $post Post object.
	 * @return int Number of images missing alt.
	 */
	private function count_images_missing_alt( $post ) {
		$content = $post->post_content;

		if ( empty( $content ) ) {
			return 0;
		}

		// Use lightweight regex instead of get_post_images() which runs
		// expensive attachment_url_to_postid() queries per image.
		if ( ! preg_match_all( '/<img[^>]+>/i', $content, $matches ) ) {
			return 0;
		}

		$missing = 0;
		foreach ( $matches[0] as $img_tag ) {
			// Count images with no alt attribute or empty alt="".
			if ( ! preg_match( '/\balt\s*=\s*["\'][^"\']+["\']/', $img_tag ) ) {
				$missing++;
			}
		}

		return $missing;
	}

	/**
	 * Format a timestamp as "X time ago".
	 *
	 * @since  1.0.0
	 * @param  int $timestamp Unix timestamp.
	 * @return string Formatted time.
	 */
	private function format_time_ago( $timestamp ) {
		return sprintf(
			/* translators: %s: Human-readable time difference */
			__( '%s ago', 'topranker-ai' ),
			human_time_diff( $timestamp, time() )
		);
	}

	/**
	 * Get enabled post types.
	 *
	 * @since  1.0.0
	 * @return array Array of enabled post type slugs.
	 */
	public function get_enabled_post_types() {
		$post_types = get_option( 'topranker_post_types', array( 'post', 'page' ) );
		return is_array( $post_types ) ? $post_types : array( 'post', 'page' );
	}

	/**
	 * Get counts for quick stats.
	 *
	 * @since  1.0.0
	 * @param  string $post_type Post type to count.
	 * @return array Counts array.
	 */
	public function get_counts( $post_type = 'post' ) {
		global $wpdb;

		// Total published posts.
		$total = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$wpdb->posts}
				WHERE post_status = 'publish'
				AND post_type = %s",
				$post_type
			)
		);

		// Posts with meta title.
		$with_meta = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(DISTINCT p.ID) FROM {$wpdb->posts} p
				INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
				WHERE p.post_status = 'publish'
				AND p.post_type = %s
				AND pm.meta_key = '_topranker_meta_title'
				AND pm.meta_value != ''",
				$post_type
			)
		);

		// Posts with SEO score.
		$with_score = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(DISTINCT p.ID) FROM {$wpdb->posts} p
				INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
				WHERE p.post_status = 'publish'
				AND p.post_type = %s
				AND pm.meta_key = '_topranker_seo_score'
				AND pm.meta_value != ''",
				$post_type
			)
		);

		return array(
			'total'        => $total,
			'with_meta'    => $with_meta,
			'missing_meta' => $total - $with_meta,
			'with_score'   => $with_score,
			'percent'      => $total > 0 ? round( ( $with_meta / $total ) * 100 ) : 0,
		);
	}

	/**
	 * Estimate API calls for a batch.
	 *
	 * Each optimization uses one API call for the main optimization.
	 * Additional calls may be made for alt tags (per image).
	 *
	 * @since  1.0.0
	 * @param  int  $post_count     Number of posts to optimize.
	 * @param  bool $include_alt_tags Whether alt tag generation is included.
	 * @param  int  $avg_images     Average images per post (default 3).
	 * @return array Estimate data.
	 */
	public function estimate_api_calls( $post_count, $include_alt_tags = false, $avg_images = 3 ) {
		// Base optimization is 1 call per post.
		$base_calls = $post_count;

		// Alt tags add calls per image.
		$alt_tag_calls = $include_alt_tags ? ( $post_count * $avg_images ) : 0;

		$total_calls = $base_calls + $alt_tag_calls;

		// Estimate cost (very rough - based on gpt-5.2 pricing).
		// About $0.01-0.03 per text optimization.
		$estimated_cost_min = $total_calls * 0.01;
		$estimated_cost_max = $total_calls * 0.03;

		return array(
			'post_count'         => $post_count,
			'base_calls'         => $base_calls,
			'alt_tag_calls'      => $alt_tag_calls,
			'total_calls'        => $total_calls,
			'estimated_cost_min' => $estimated_cost_min,
			'estimated_cost_max' => $estimated_cost_max,
			'cost_range'         => sprintf(
				'$%.2f - $%.2f',
				$estimated_cost_min,
				$estimated_cost_max
			),
		);
	}

	/**
	 * Validate a batch of post IDs.
	 *
	 * @since  1.0.0
	 * @param  array $post_ids Array of post IDs.
	 * @return array|WP_Error Validated post IDs or error.
	 */
	public function validate_batch( $post_ids ) {
		if ( ! is_array( $post_ids ) || empty( $post_ids ) ) {
			return new WP_Error(
				'invalid_batch',
				__( 'No posts selected for optimization.', 'topranker-ai' )
			);
		}

		// Enforce maximum batch size.
		if ( count( $post_ids ) > self::MAX_BATCH_SIZE ) {
			return new WP_Error(
				'batch_too_large',
				sprintf(
					/* translators: %d: Maximum batch size */
					__( 'Maximum batch size is %d posts.', 'topranker-ai' ),
					self::MAX_BATCH_SIZE
				)
			);
		}

		$enabled_types = $this->get_enabled_post_types();
		$valid_ids     = array();

		foreach ( $post_ids as $post_id ) {
			$post_id = absint( $post_id );
			if ( $post_id <= 0 ) {
				continue;
			}

			$post = get_post( $post_id );
			if ( ! $post ) {
				continue;
			}

			// Check post type is enabled.
			if ( ! in_array( $post->post_type, $enabled_types, true ) ) {
				continue;
			}

			// Check user can edit this post.
			if ( ! current_user_can( 'edit_post', $post_id ) ) {
				continue;
			}

			$valid_ids[] = $post_id;
		}

		if ( empty( $valid_ids ) ) {
			return new WP_Error(
				'no_valid_posts',
				__( 'No valid posts found in selection.', 'topranker-ai' )
			);
		}

		return $valid_ids;
	}

	/**
	 * Get view data for the bulk page.
	 *
	 * @since  1.0.0
	 * @param  array $args Query arguments.
	 * @return array View data.
	 */
	public function get_view_data( $args = array() ) {
		$enabled_types = $this->get_enabled_post_types();
		$current_type  = isset( $args['post_type'] ) ? $args['post_type'] : ( ! empty( $enabled_types ) ? $enabled_types[0] : 'post' );

		// Get post type objects for labels.
		$post_type_options = array();
		foreach ( $enabled_types as $type ) {
			$obj = get_post_type_object( $type );
			if ( $obj ) {
				$post_type_options[ $type ] = $obj->labels->name;
			}
		}

		// Get posts.
		$posts_data = $this->get_posts( array_merge( $args, array( 'post_type' => $current_type ) ) );

		// Get counts.
		$counts = $this->get_counts( $current_type );

		return array(
			'posts_data'        => $posts_data,
			'counts'            => $counts,
			'post_type_options' => $post_type_options,
			'current_type'      => $current_type,
			'enabled_types'     => $enabled_types,
			'max_batch_size'    => self::MAX_BATCH_SIZE,
			'is_pro'            => topranker_is_pro(),
			'nonce'             => wp_create_nonce( 'topranker_ajax_nonce' ),
		);
	}
}
