<?php
/**
 * Optimization history management.
 *
 * Stores and manages the last 10 optimization results per post,
 * allowing users to view history and revert to earlier versions.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * TopRanker History class.
 *
 * @since 1.0.0
 */
class TopRanker_History {

	/**
	 * Maximum number of history entries to store per post.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const MAX_ENTRIES = 10;

	/**
	 * Post meta key for optimization history.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	const META_KEY = '_topranker_optimization_history';

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		// History is managed programmatically, no hooks needed.
	}

	/**
	 * Add a new history entry.
	 *
	 * Stores the current optimization values before they are overwritten.
	 * Automatically prunes oldest entries when exceeding MAX_ENTRIES.
	 *
	 * @since 1.0.0
	 * @param int   $post_id Post ID.
	 * @param array $values  Array of optimization values being applied.
	 * @return bool True on success, false on failure.
	 */
	public function add_entry( $post_id, $values ) {
		$post = get_post( $post_id );

		if ( ! $post ) {
			return false;
		}

		// Only add entry if there's actually data to save.
		if ( empty( $values ) ) {
			return false;
		}

		// Get existing history.
		$history = $this->get_history( $post_id );

		// Create new entry.
		$entry = array(
			'timestamp'   => time(),
			'date'        => gmdate( 'Y-m-d H:i:s' ),
			'values'      => $values,
			'user_id'     => get_current_user_id(),
			'user_name'   => $this->get_user_display_name( get_current_user_id() ),
		);

		// Add new entry at the beginning.
		array_unshift( $history, $entry );

		// Prune oldest entries if exceeding max.
		if ( count( $history ) > self::MAX_ENTRIES ) {
			$history = array_slice( $history, 0, self::MAX_ENTRIES );
		}

		// Save updated history.
		return (bool) update_post_meta( $post_id, self::META_KEY, $history );
	}

	/**
	 * Get optimization history for a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return array Array of history entries, newest first.
	 */
	public function get_history( $post_id ) {
		$history = get_post_meta( $post_id, self::META_KEY, true );

		if ( ! is_array( $history ) ) {
			return array();
		}

		return $history;
	}

	/**
	 * Get a specific history entry by index.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @param int $index   Entry index (0 = most recent).
	 * @return array|null History entry or null if not found.
	 */
	public function get_entry( $post_id, $index ) {
		$history = $this->get_history( $post_id );

		if ( isset( $history[ $index ] ) ) {
			return $history[ $index ];
		}

		return null;
	}

	/**
	 * Get the history entry count for a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return int Number of history entries.
	 */
	public function get_count( $post_id ) {
		$history = $this->get_history( $post_id );
		return count( $history );
	}

	/**
	 * Revert to a specific history entry.
	 *
	 * Restores the values from a history entry to the post meta.
	 * Also creates a new history entry for the current state before reverting.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @param int $index   History entry index to revert to.
	 * @return array|WP_Error Result array or WP_Error on failure.
	 */
	public function revert_to_entry( $post_id, $index ) {
		$post = get_post( $post_id );

		if ( ! $post ) {
			return new WP_Error(
				'invalid_post',
				__( 'Invalid post.', 'topranker-ai' )
			);
		}

		$entry = $this->get_entry( $post_id, $index );

		if ( null === $entry ) {
			return new WP_Error(
				'invalid_entry',
				__( 'History entry not found.', 'topranker-ai' )
			);
		}

		// Get current values to save before reverting.
		$current_values = $this->get_current_values( $post_id );

		// Save current state to history before reverting.
		if ( ! empty( $current_values ) ) {
			$this->add_entry( $post_id, $current_values );
		}

		// Apply the historical values.
		$values  = isset( $entry['values'] ) ? $entry['values'] : array();
		$applied = array();

		// Apply each value.
		if ( ! empty( $values['meta_title'] ) ) {
			update_post_meta( $post_id, '_topranker_meta_title', sanitize_text_field( $values['meta_title'] ) );
			$applied['meta_title'] = true;
		}

		if ( ! empty( $values['meta_description'] ) ) {
			update_post_meta( $post_id, '_topranker_meta_description', sanitize_textarea_field( $values['meta_description'] ) );
			$applied['meta_description'] = true;
		}

		if ( ! empty( $values['focus_keyphrase'] ) ) {
			update_post_meta( $post_id, '_topranker_focus_keyphrase', sanitize_text_field( $values['focus_keyphrase'] ) );
			$applied['focus_keyphrase'] = true;
		}

		if ( ! empty( $values['secondary_keyphrases'] ) && is_array( $values['secondary_keyphrases'] ) ) {
			update_post_meta( $post_id, '_topranker_secondary_keyphrases', array_map( 'sanitize_text_field', $values['secondary_keyphrases'] ) );
			$applied['secondary_keyphrases'] = true;
		}

		if ( ! empty( $values['og_title'] ) ) {
			update_post_meta( $post_id, '_topranker_og_title', sanitize_text_field( $values['og_title'] ) );
			$applied['og_title'] = true;
		}

		if ( ! empty( $values['og_description'] ) ) {
			update_post_meta( $post_id, '_topranker_og_description', sanitize_textarea_field( $values['og_description'] ) );
			$applied['og_description'] = true;
		}

		if ( ! empty( $values['twitter_title'] ) ) {
			update_post_meta( $post_id, '_topranker_twitter_title', sanitize_text_field( $values['twitter_title'] ) );
			$applied['twitter_title'] = true;
		}

		if ( ! empty( $values['twitter_description'] ) ) {
			update_post_meta( $post_id, '_topranker_twitter_description', sanitize_textarea_field( $values['twitter_description'] ) );
			$applied['twitter_description'] = true;
		}

		if ( ! empty( $values['excerpt'] ) ) {
			wp_update_post(
				array(
					'ID'           => $post_id,
					'post_excerpt' => sanitize_textarea_field( $values['excerpt'] ),
				)
			);
			$applied['excerpt'] = true;
		}

		// Recalculate SEO score if available.
		$seo_score = null;
		if ( class_exists( 'TopRanker_SEO_Audit' ) ) {
			$audit     = new TopRanker_SEO_Audit();
			$seo_score = $audit->update_score( $post_id );
		}

		$result = array(
			'post_id'            => $post_id,
			'reverted_to'        => $index,
			'reverted_timestamp' => $entry['timestamp'],
			'reverted_date'      => $entry['date'],
			'applied'            => $applied,
			'reverted'           => $values,
			'message'            => sprintf(
				/* translators: %s: Date/time of the restored version */
				__( 'Successfully reverted to version from %s.', 'topranker-ai' ),
				$this->format_timestamp( $entry['timestamp'] )
			),
		);

		if ( null !== $seo_score ) {
			$result['seo_score'] = $seo_score;
		}

		return $result;
	}

	/**
	 * Get the current optimization values for a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return array Current values.
	 */
	public function get_current_values( $post_id ) {
		$post = get_post( $post_id );

		if ( ! $post ) {
			return array();
		}

		$values = array();

		$meta_title = get_post_meta( $post_id, '_topranker_meta_title', true );
		if ( ! empty( $meta_title ) ) {
			$values['meta_title'] = $meta_title;
		}

		$meta_description = get_post_meta( $post_id, '_topranker_meta_description', true );
		if ( ! empty( $meta_description ) ) {
			$values['meta_description'] = $meta_description;
		}

		$focus_keyphrase = get_post_meta( $post_id, '_topranker_focus_keyphrase', true );
		if ( ! empty( $focus_keyphrase ) ) {
			$values['focus_keyphrase'] = $focus_keyphrase;
		}

		$secondary_keyphrases = get_post_meta( $post_id, '_topranker_secondary_keyphrases', true );
		if ( ! empty( $secondary_keyphrases ) && is_array( $secondary_keyphrases ) ) {
			$values['secondary_keyphrases'] = $secondary_keyphrases;
		}

		$og_title = get_post_meta( $post_id, '_topranker_og_title', true );
		if ( ! empty( $og_title ) ) {
			$values['og_title'] = $og_title;
		}

		$og_description = get_post_meta( $post_id, '_topranker_og_description', true );
		if ( ! empty( $og_description ) ) {
			$values['og_description'] = $og_description;
		}

		$twitter_title = get_post_meta( $post_id, '_topranker_twitter_title', true );
		if ( ! empty( $twitter_title ) ) {
			$values['twitter_title'] = $twitter_title;
		}

		$twitter_description = get_post_meta( $post_id, '_topranker_twitter_description', true );
		if ( ! empty( $twitter_description ) ) {
			$values['twitter_description'] = $twitter_description;
		}

		if ( ! empty( $post->post_excerpt ) ) {
			$values['excerpt'] = $post->post_excerpt;
		}

		return $values;
	}

	/**
	 * Get history formatted for API response.
	 *
	 * Adds additional display information for the frontend.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return array Formatted history for API response.
	 */
	public function get_api_response( $post_id ) {
		$raw_history = $this->get_history( $post_id );

		$formatted = array();

		foreach ( $raw_history as $index => $entry ) {
			$formatted[] = array(
				'index'          => $index,
				'timestamp'      => $entry['timestamp'],
				'date'           => $entry['date'],
				'date_formatted' => $this->format_timestamp( $entry['timestamp'] ),
				'time_ago'       => $this->get_time_ago( $entry['timestamp'] ),
				'user_name'      => isset( $entry['user_name'] ) ? $entry['user_name'] : __( 'Unknown', 'topranker-ai' ),
				'fields_count'   => $this->count_fields( isset( $entry['values'] ) ? $entry['values'] : array() ),
				'fields_summary' => $this->get_fields_summary( isset( $entry['values'] ) ? $entry['values'] : array() ),
				'data'           => isset( $entry['values'] ) ? $entry['values'] : array(),
			);
		}

		return array(
			'post_id' => $post_id,
			'count'   => count( $formatted ),
			'history' => $formatted,
		);
	}

	/**
	 * Get a comparison between current values and a history entry.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @param int $index   History entry index to compare.
	 * @return array|WP_Error Comparison data or WP_Error on failure.
	 */
	public function get_diff( $post_id, $index ) {
		$entry = $this->get_entry( $post_id, $index );

		if ( null === $entry ) {
			return new WP_Error(
				'invalid_entry',
				__( 'History entry not found.', 'topranker-ai' )
			);
		}

		$current   = $this->get_current_values( $post_id );
		$historical = isset( $entry['values'] ) ? $entry['values'] : array();

		$diff = array();
		$all_fields = array(
			'meta_title',
			'meta_description',
			'focus_keyphrase',
			'og_title',
			'og_description',
			'twitter_title',
			'twitter_description',
			'excerpt',
		);

		foreach ( $all_fields as $field ) {
			$current_value    = isset( $current[ $field ] ) ? $current[ $field ] : '';
			$historical_value = isset( $historical[ $field ] ) ? $historical[ $field ] : '';

			// Skip arrays for simple diff.
			if ( is_array( $current_value ) || is_array( $historical_value ) ) {
				continue;
			}

			$diff[ $field ] = array(
				'current'    => $current_value,
				'historical' => $historical_value,
				'changed'    => $current_value !== $historical_value,
			);
		}

		return array(
			'post_id'            => $post_id,
			'entry_index'        => $index,
			'entry_timestamp'    => $entry['timestamp'],
			'entry_date'         => $this->format_timestamp( $entry['timestamp'] ),
			'diff'               => $diff,
			'has_changes'        => $this->has_differences( $diff ),
		);
	}

	/**
	 * Clear all history for a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True on success.
	 */
	public function clear_history( $post_id ) {
		return delete_post_meta( $post_id, self::META_KEY );
	}

	/**
	 * Format a timestamp for display.
	 *
	 * @since 1.0.0
	 * @param int $timestamp Unix timestamp.
	 * @return string Formatted date string.
	 */
	private function format_timestamp( $timestamp ) {
		return wp_date( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $timestamp );
	}

	/**
	 * Get time ago string for a timestamp.
	 *
	 * @since 1.0.0
	 * @param int $timestamp Unix timestamp.
	 * @return string Human-readable time difference.
	 */
	private function get_time_ago( $timestamp ) {
		return sprintf(
			/* translators: %s: Human-readable time difference */
			__( '%s ago', 'topranker-ai' ),
			human_time_diff( $timestamp, time() )
		);
	}

	/**
	 * Get user display name.
	 *
	 * @since 1.0.0
	 * @param int $user_id User ID.
	 * @return string User display name.
	 */
	private function get_user_display_name( $user_id ) {
		$user = get_userdata( $user_id );

		if ( ! $user ) {
			return __( 'Unknown', 'topranker-ai' );
		}

		return $user->display_name;
	}

	/**
	 * Count non-empty fields in values array.
	 *
	 * @since 1.0.0
	 * @param array $values Values array.
	 * @return int Number of non-empty fields.
	 */
	private function count_fields( $values ) {
		if ( ! is_array( $values ) ) {
			return 0;
		}

		$count = 0;

		foreach ( $values as $value ) {
			if ( ! empty( $value ) ) {
				$count++;
			}
		}

		return $count;
	}

	/**
	 * Get a summary of fields in a values array.
	 *
	 * @since 1.0.0
	 * @param array $values Values array.
	 * @return string Summary string.
	 */
	private function get_fields_summary( $values ) {
		if ( ! is_array( $values ) || empty( $values ) ) {
			return __( 'No fields', 'topranker-ai' );
		}

		$labels = array(
			'meta_title'           => __( 'Title', 'topranker-ai' ),
			'meta_description'     => __( 'Description', 'topranker-ai' ),
			'focus_keyphrase'      => __( 'Keyphrase', 'topranker-ai' ),
			'secondary_keyphrases' => __( 'Secondary', 'topranker-ai' ),
			'og_title'             => __( 'OG Title', 'topranker-ai' ),
			'og_description'       => __( 'OG Desc', 'topranker-ai' ),
			'twitter_title'        => __( 'Twitter Title', 'topranker-ai' ),
			'twitter_description'  => __( 'Twitter Desc', 'topranker-ai' ),
			'excerpt'              => __( 'Excerpt', 'topranker-ai' ),
		);

		$field_names = array();

		foreach ( $values as $key => $value ) {
			if ( ! empty( $value ) && isset( $labels[ $key ] ) ) {
				$field_names[] = $labels[ $key ];
			}
		}

		if ( empty( $field_names ) ) {
			return __( 'No fields', 'topranker-ai' );
		}

		return implode( ', ', $field_names );
	}

	/**
	 * Check if diff has any changes.
	 *
	 * @since 1.0.0
	 * @param array $diff Diff array.
	 * @return bool True if there are changes.
	 */
	private function has_differences( $diff ) {
		foreach ( $diff as $field_diff ) {
			if ( isset( $field_diff['changed'] ) && $field_diff['changed'] ) {
				return true;
			}
		}

		return false;
	}
}
