<?php
/**
 * Media Library Integration (Pro).
 *
 * Adds bulk alt tag generation capability to the WordPress Media Library.
 * Integrates with the existing alt tags generation system.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * TopRanker Media Library class.
 *
 * @since 1.0.0
 */
class TopRanker_Media_Library {

	/**
	 * Maximum attachments per bulk batch.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const MAX_BATCH_SIZE = 100;

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		$this->init_hooks();
	}

	/**
	 * Initialize hooks.
	 *
	 * @since 1.0.0
	 */
	private function init_hooks() {
		// Only add hooks in admin and if Pro is active.
		if ( ! is_admin() || ! topranker_is_pro() ) {
			return;
		}

		// Add bulk action to Media Library.
		add_filter( 'bulk_actions-upload', array( $this, 'add_bulk_action' ) );
		add_filter( 'handle_bulk_actions-upload', array( $this, 'handle_bulk_action' ), 10, 3 );

		// Add row action to individual attachments.
		add_filter( 'media_row_actions', array( $this, 'add_row_action' ), 10, 3 );

		// Add admin notices.
		add_action( 'admin_notices', array( $this, 'display_admin_notices' ) );

		// Enqueue scripts on Media Library pages.
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );

		// AJAX handlers for bulk alt tag generation.
		add_action( 'wp_ajax_topranker_media_bulk_alt', array( $this, 'ajax_bulk_generate_alt' ) );
		add_action( 'wp_ajax_topranker_media_single_alt', array( $this, 'ajax_single_generate_alt' ) );
		add_action( 'wp_ajax_topranker_media_apply_alt', array( $this, 'ajax_apply_alt' ) );
	}

	/**
	 * Add bulk action to Media Library dropdown.
	 *
	 * @since  1.0.0
	 * @param  array $bulk_actions Existing bulk actions.
	 * @return array Modified bulk actions.
	 */
	public function add_bulk_action( $bulk_actions ) {
		$bulk_actions['topranker_generate_alt'] = __( 'Generate Alt Tags (TopRanker)', 'topranker-ai' );
		return $bulk_actions;
	}

	/**
	 * Handle the bulk action.
	 *
	 * @since  1.0.0
	 * @param  string $redirect_to The redirect URL.
	 * @param  string $doaction    The action being taken.
	 * @param  array  $post_ids    The attachment IDs to process.
	 * @return string Modified redirect URL.
	 */
	public function handle_bulk_action( $redirect_to, $doaction, $post_ids ) {
		if ( 'topranker_generate_alt' !== $doaction ) {
			return $redirect_to;
		}

		// Verify the action is allowed.
		if ( ! current_user_can( 'upload_files' ) ) {
			return add_query_arg( 'topranker_error', 'no_permission', $redirect_to );
		}

		// Filter to only image attachments.
		$image_ids = array();
		foreach ( $post_ids as $post_id ) {
			if ( wp_attachment_is_image( $post_id ) ) {
				$image_ids[] = absint( $post_id );
			}
		}

		if ( empty( $image_ids ) ) {
			return add_query_arg( 'topranker_error', 'no_images', $redirect_to );
		}

		// Limit batch size.
		if ( count( $image_ids ) > self::MAX_BATCH_SIZE ) {
			$image_ids = array_slice( $image_ids, 0, self::MAX_BATCH_SIZE );
		}

		// Store the IDs in a transient for the processing page.
		$batch_key = 'topranker_media_batch_' . get_current_user_id();
		set_transient( $batch_key, $image_ids, HOUR_IN_SECONDS );

		// Redirect to the processing page.
		$redirect_to = add_query_arg(
			array(
				'topranker_media_alt'  => '1',
				'topranker_batch_count' => count( $image_ids ),
			),
			admin_url( 'upload.php' )
		);

		return $redirect_to;
	}

	/**
	 * Add row action for individual attachments.
	 *
	 * @since  1.0.0
	 * @param  array   $actions    Existing row actions.
	 * @param  WP_Post $post       The attachment post object.
	 * @param  bool    $detached   Whether the attachment is detached.
	 * @return array   Modified row actions.
	 */
	public function add_row_action( $actions, $post, $detached ) {
		// Only add for images.
		if ( ! wp_attachment_is_image( $post->ID ) ) {
			return $actions;
		}

		// Check user can edit.
		if ( ! current_user_can( 'edit_post', $post->ID ) ) {
			return $actions;
		}

		// Get current alt text.
		$current_alt = get_post_meta( $post->ID, '_wp_attachment_image_alt', true );
		$alt_status  = empty( $current_alt ) ? __( 'Generate Alt', 'topranker-ai' ) : __( 'Regenerate Alt', 'topranker-ai' );

		$actions['topranker_alt'] = sprintf(
			'<a href="#" class="topranker-generate-alt" data-attachment-id="%d" data-nonce="%s" aria-label="%s">%s</a>',
			esc_attr( $post->ID ),
			esc_attr( wp_create_nonce( 'topranker_media_alt_' . $post->ID ) ),
			/* translators: %s: Attachment title */
			esc_attr( sprintf( __( 'Generate alt text for "%s"', 'topranker-ai' ), $post->post_title ) ),
			esc_html( $alt_status )
		);

		return $actions;
	}

	/**
	 * Display admin notices.
	 *
	 * @since 1.0.0
	 */
	public function display_admin_notices() {
		$screen = get_current_screen();
		if ( ! $screen || 'upload' !== $screen->id ) {
			return;
		}

		// Check for error messages.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Display only.
		$error = isset( $_GET['topranker_error'] ) ? sanitize_key( $_GET['topranker_error'] ) : '';
		if ( $error ) {
			$message = '';
			switch ( $error ) {
				case 'no_permission':
					$message = __( 'You do not have permission to generate alt tags.', 'topranker-ai' );
					break;
				case 'no_images':
					$message = __( 'No image files were selected. Alt tags can only be generated for images.', 'topranker-ai' );
					break;
				default:
					$message = __( 'An error occurred. Please try again.', 'topranker-ai' );
			}
			?>
			<div class="notice notice-error is-dismissible">
				<p><strong><?php esc_html_e( 'TopRanker AI:', 'topranker-ai' ); ?></strong> <?php echo esc_html( $message ); ?></p>
			</div>
			<?php
		}

		// Check for success message.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Display only.
		$success_count = isset( $_GET['topranker_success'] ) ? absint( $_GET['topranker_success'] ) : 0;
		if ( $success_count > 0 ) {
			?>
			<div class="notice notice-success is-dismissible">
				<p>
					<strong><?php esc_html_e( 'TopRanker AI:', 'topranker-ai' ); ?></strong>
					<?php
					printf(
						/* translators: %d: Number of images processed */
						esc_html( _n(
							'Alt text generated for %d image.',
							'Alt text generated for %d images.',
							$success_count,
							'topranker-ai'
						) ),
						esc_html( $success_count )
					);
					?>
				</p>
			</div>
			<?php
		}
	}

	/**
	 * Enqueue scripts on Media Library pages.
	 *
	 * @since 1.0.0
	 * @param string $hook_suffix Current admin page.
	 */
	public function enqueue_scripts( $hook_suffix ) {
		if ( 'upload.php' !== $hook_suffix ) {
			return;
		}

		// Check if we should show the bulk processing modal.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Display only.
		$show_modal = isset( $_GET['topranker_media_alt'] ) && '1' === $_GET['topranker_media_alt'];

		// Get batch data if modal should be shown.
		$batch_ids = array();
		if ( $show_modal ) {
			$batch_key  = 'topranker_media_batch_' . get_current_user_id();
			$batch_ids  = get_transient( $batch_key );
			if ( ! is_array( $batch_ids ) ) {
				$batch_ids = array();
			}
		}

		// Enqueue the script.
		wp_enqueue_script(
			'topranker-media-library',
			TOPRANKER_URL . 'admin/js/topranker-media-library.js',
			array( 'jquery' ),
			TOPRANKER_VERSION,
			true
		);

		// Localize script data.
		wp_localize_script(
			'topranker-media-library',
			'toprankerMedia',
			array(
				'ajaxUrl'    => admin_url( 'admin-ajax.php' ),
				'nonce'      => wp_create_nonce( 'topranker_media_nonce' ),
				'showModal'  => $show_modal,
				'batchIds'   => $batch_ids,
				'isPro'      => topranker_is_pro(),
				'upgradeUrl' => topranker_get_upgrade_url(),
				'i18n'       => array(
					'modalTitle'       => __( 'Generate Alt Tags', 'topranker-ai' ),
					'processing'       => __( 'Processing...', 'topranker-ai' ),
					'generating'       => __( 'Generating alt text...', 'topranker-ai' ),
					'complete'         => __( 'Complete!', 'topranker-ai' ),
					'cancelled'        => __( 'Cancelled', 'topranker-ai' ),
					'error'            => __( 'Error', 'topranker-ai' ),
					'cancel'           => __( 'Cancel', 'topranker-ai' ),
					'close'            => __( 'Close', 'topranker-ai' ),
					'done'             => __( 'Done', 'topranker-ai' ),
					'optimized'        => __( 'optimized', 'topranker-ai' ),
					'failed'           => __( 'failed', 'topranker-ai' ),
					'of'               => __( 'of', 'topranker-ai' ),
					'imageAlt'         => __( 'image', 'topranker-ai' ),
					'imagesAlt'        => __( 'images', 'topranker-ai' ),
					'applyAlt'         => __( 'Apply', 'topranker-ai' ),
					'applied'          => __( 'Applied!', 'topranker-ai' ),
					'errorApply'       => __( 'Error applying alt text', 'topranker-ai' ),
					'networkError'     => __( 'Network error. Please try again.', 'topranker-ai' ),
					'proRequired'      => __( 'This feature requires TopRanker Pro.', 'topranker-ai' ),
					'estimatedCalls'   => __( 'Estimated API Calls:', 'topranker-ai' ),
					'estimatedCost'    => __( 'Estimated Cost:', 'topranker-ai' ),
					'confirmProceed'   => __( 'Proceed', 'topranker-ai' ),
					'confirmCancel'    => __( 'Cancel', 'topranker-ai' ),
				),
			)
		);

		// Add inline CSS for the modal.
		wp_add_inline_style(
			'media',
			$this->get_inline_styles()
		);
	}

	/**
	 * Get inline styles for the Media Library integration.
	 *
	 * @since  1.0.0
	 * @return string CSS styles.
	 */
	private function get_inline_styles() {
		return '
			/* TopRanker Media Library Styles */
			.topranker-media-modal {
				position: fixed;
				inset: 0;
				z-index: 160000;
				display: flex;
				align-items: center;
				justify-content: center;
			}

			.topranker-media-modal-backdrop {
				position: absolute;
				inset: 0;
				background: rgba(0, 0, 0, 0.6);
			}

			.topranker-media-modal-content {
				position: relative;
				background: #fff;
				border-radius: 4px;
				box-shadow: 0 3px 30px rgba(0, 0, 0, 0.2);
				max-width: 600px;
				width: 90%;
				max-height: 80vh;
				display: flex;
				flex-direction: column;
			}

			.topranker-media-modal-header {
				display: flex;
				justify-content: space-between;
				align-items: center;
				padding: 15px 20px;
				border-bottom: 1px solid #dcdcde;
			}

			.topranker-media-modal-header h2 {
				margin: 0;
				font-size: 18px;
			}

			.topranker-media-modal-close {
				background: none;
				border: none;
				cursor: pointer;
				padding: 5px;
				font-size: 20px;
				line-height: 1;
				color: #646970;
			}

			.topranker-media-modal-close:hover {
				color: #1d2327;
			}

			.topranker-media-modal-body {
				padding: 20px;
				overflow-y: auto;
				flex: 1;
			}

			.topranker-media-modal-footer {
				display: flex;
				justify-content: flex-end;
				gap: 10px;
				padding: 15px 20px;
				border-top: 1px solid #dcdcde;
				background: #f6f7f7;
			}

			/* Progress bar */
			.topranker-media-progress {
				margin: 20px 0;
			}

			.topranker-media-progress-header {
				display: flex;
				justify-content: space-between;
				margin-bottom: 10px;
			}

			.topranker-media-progress-bar {
				height: 20px;
				background: #dcdcde;
				border-radius: 10px;
				overflow: hidden;
			}

			.topranker-media-progress-fill {
				height: 100%;
				background: linear-gradient(90deg, #2271b1, #135e96);
				transition: width 0.3s ease;
				border-radius: 10px;
			}

			.topranker-media-progress-text {
				margin-top: 10px;
				font-size: 13px;
				color: #646970;
			}

			/* Results list */
			.topranker-media-results {
				max-height: 250px;
				overflow-y: auto;
				border: 1px solid #dcdcde;
				border-radius: 4px;
				margin-top: 15px;
			}

			.topranker-media-result {
				display: flex;
				align-items: center;
				gap: 10px;
				padding: 10px 12px;
				border-bottom: 1px solid #f0f0f1;
			}

			.topranker-media-result:last-child {
				border-bottom: none;
			}

			.topranker-media-result-thumb {
				width: 40px;
				height: 40px;
				object-fit: cover;
				border-radius: 3px;
				flex-shrink: 0;
			}

			.topranker-media-result-info {
				flex: 1;
				min-width: 0;
			}

			.topranker-media-result-title {
				font-weight: 600;
				white-space: nowrap;
				overflow: hidden;
				text-overflow: ellipsis;
			}

			.topranker-media-result-alt {
				font-size: 12px;
				color: #646970;
				white-space: nowrap;
				overflow: hidden;
				text-overflow: ellipsis;
			}

			.topranker-media-result-status {
				flex-shrink: 0;
			}

			.topranker-media-result.is-processing .topranker-media-result-status {
				color: #2271b1;
			}

			.topranker-media-result.is-success .topranker-media-result-status {
				color: #00a32a;
			}

			.topranker-media-result.is-error .topranker-media-result-status {
				color: #d63638;
			}

			/* Confirmation panel */
			.topranker-media-confirm {
				background: #f0f6fc;
				padding: 15px;
				border-radius: 4px;
				margin-bottom: 20px;
			}

			.topranker-media-confirm p {
				margin: 5px 0;
			}

			.topranker-media-confirm strong {
				display: inline-block;
				min-width: 150px;
			}

			/* Row action link styles */
			.topranker-generate-alt.is-loading {
				pointer-events: none;
				opacity: 0.6;
			}

			.topranker-alt-result {
				display: inline-block;
				margin-left: 10px;
				font-size: 12px;
			}

			.topranker-alt-result.is-success {
				color: #00a32a;
			}

			.topranker-alt-result.is-error {
				color: #d63638;
			}
		';
	}

	/**
	 * AJAX handler for bulk alt tag generation.
	 *
	 * Processes a single attachment in the bulk queue.
	 *
	 * @since 1.0.0
	 */
	public function ajax_bulk_generate_alt() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'topranker_media_nonce', 'nonce', false ) ) {
			wp_send_json_error(
				array( 'message' => __( 'Security check failed.', 'topranker-ai' ) ),
				403
			);
		}

		// Check Pro status.
		if ( ! topranker_is_pro() ) {
			wp_send_json_error(
				array( 'message' => __( 'This feature requires TopRanker Pro.', 'topranker-ai' ) ),
				403
			);
		}

		// Check permissions.
		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error(
				array( 'message' => __( 'You do not have permission to edit media.', 'topranker-ai' ) ),
				403
			);
		}

		// Get attachment ID.
		$attachment_id = isset( $_POST['attachment_id'] ) ? absint( $_POST['attachment_id'] ) : 0;
		if ( ! $attachment_id ) {
			wp_send_json_error(
				array( 'message' => __( 'Invalid attachment ID.', 'topranker-ai' ) ),
				400
			);
		}

		// Verify it's an image.
		if ( ! wp_attachment_is_image( $attachment_id ) ) {
			wp_send_json_error(
				array( 'message' => __( 'Attachment is not an image.', 'topranker-ai' ) ),
				400
			);
		}

		// Check user can edit this attachment.
		if ( ! current_user_can( 'edit_post', $attachment_id ) ) {
			wp_send_json_error(
				array( 'message' => __( 'You do not have permission to edit this image.', 'topranker-ai' ) ),
				403
			);
		}

		// Generate alt tag.
		if ( ! class_exists( 'TopRanker_Alt_Tags' ) ) {
			wp_send_json_error(
				array( 'message' => __( 'Alt tag generation is not available.', 'topranker-ai' ) ),
				500
			);
		}

		$alt_tags = new TopRanker_Alt_Tags();
		$result   = $alt_tags->generate_for_attachment( $attachment_id );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error(
				array( 'message' => $result->get_error_message() ),
				400
			);
		}

		wp_send_json_success( $result );
	}

	/**
	 * AJAX handler for single image alt tag generation (from row action).
	 *
	 * @since 1.0.0
	 */
	public function ajax_single_generate_alt() {
		// Get attachment ID for nonce verification.
		$attachment_id = isset( $_POST['attachment_id'] ) ? absint( $_POST['attachment_id'] ) : 0;

		// Verify nonce.
		if ( ! check_ajax_referer( 'topranker_media_alt_' . $attachment_id, 'nonce', false ) ) {
			wp_send_json_error(
				array( 'message' => __( 'Security check failed.', 'topranker-ai' ) ),
				403
			);
		}

		// Check Pro status.
		if ( ! topranker_is_pro() ) {
			wp_send_json_error(
				array( 'message' => __( 'This feature requires TopRanker Pro.', 'topranker-ai' ) ),
				403
			);
		}

		// Check permissions.
		if ( ! $attachment_id || ! current_user_can( 'edit_post', $attachment_id ) ) {
			wp_send_json_error(
				array( 'message' => __( 'You do not have permission to edit this image.', 'topranker-ai' ) ),
				403
			);
		}

		// Verify it's an image.
		if ( ! wp_attachment_is_image( $attachment_id ) ) {
			wp_send_json_error(
				array( 'message' => __( 'Attachment is not an image.', 'topranker-ai' ) ),
				400
			);
		}

		// Generate alt tag.
		if ( ! class_exists( 'TopRanker_Alt_Tags' ) ) {
			wp_send_json_error(
				array( 'message' => __( 'Alt tag generation is not available.', 'topranker-ai' ) ),
				500
			);
		}

		$alt_tags = new TopRanker_Alt_Tags();
		$result   = $alt_tags->generate_for_attachment( $attachment_id );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error(
				array( 'message' => $result->get_error_message() ),
				400
			);
		}

		wp_send_json_success( $result );
	}

	/**
	 * AJAX handler for applying generated alt text to attachment.
	 *
	 * @since 1.0.0
	 */
	public function ajax_apply_alt() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'topranker_media_nonce', 'nonce', false ) ) {
			wp_send_json_error(
				array( 'message' => __( 'Security check failed.', 'topranker-ai' ) ),
				403
			);
		}

		// Check Pro status.
		if ( ! topranker_is_pro() ) {
			wp_send_json_error(
				array( 'message' => __( 'This feature requires TopRanker Pro.', 'topranker-ai' ) ),
				403
			);
		}

		// Get data.
		$attachment_id = isset( $_POST['attachment_id'] ) ? absint( $_POST['attachment_id'] ) : 0;
		$alt_text      = isset( $_POST['alt_text'] ) ? sanitize_text_field( wp_unslash( $_POST['alt_text'] ) ) : '';

		if ( ! $attachment_id ) {
			wp_send_json_error(
				array( 'message' => __( 'Invalid attachment ID.', 'topranker-ai' ) ),
				400
			);
		}

		if ( empty( $alt_text ) ) {
			wp_send_json_error(
				array( 'message' => __( 'Alt text is empty.', 'topranker-ai' ) ),
				400
			);
		}

		// Check permissions.
		if ( ! current_user_can( 'edit_post', $attachment_id ) ) {
			wp_send_json_error(
				array( 'message' => __( 'You do not have permission to edit this image.', 'topranker-ai' ) ),
				403
			);
		}

		// Apply alt text.
		$result = update_post_meta( $attachment_id, '_wp_attachment_image_alt', $alt_text );

		if ( false === $result ) {
			// Check if the value is already the same.
			$current = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
			if ( $current !== $alt_text ) {
				wp_send_json_error(
					array( 'message' => __( 'Failed to save alt text.', 'topranker-ai' ) ),
					500
				);
			}
		}

		wp_send_json_success(
			array(
				'attachment_id' => $attachment_id,
				'alt_text'      => $alt_text,
			)
		);
	}

	/**
	 * Get images missing alt tags from the Media Library.
	 *
	 * @since  1.0.0
	 * @param  int $limit Maximum number of images to return.
	 * @return array Array of attachment data.
	 */
	public function get_images_missing_alt( $limit = 100 ) {
		global $wpdb;

		$results = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT p.ID, p.post_title, p.guid
				FROM {$wpdb->posts} p
				LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_wp_attachment_image_alt'
				WHERE p.post_type = 'attachment'
				AND p.post_mime_type LIKE 'image/%'
				AND (pm.meta_value IS NULL OR pm.meta_value = '')
				ORDER BY p.ID DESC
				LIMIT %d",
				$limit
			)
		);

		$images = array();
		foreach ( $results as $row ) {
			$images[] = array(
				'ID'        => $row->ID,
				'title'     => $row->post_title,
				'url'       => wp_get_attachment_url( $row->ID ),
				'thumb_url' => wp_get_attachment_image_url( $row->ID, 'thumbnail' ),
			);
		}

		return $images;
	}

	/**
	 * Count images missing alt tags.
	 *
	 * @since  1.0.0
	 * @return int Count of images without alt tags.
	 */
	public function count_images_missing_alt() {
		global $wpdb;

		return (int) $wpdb->get_var(
			"SELECT COUNT(*)
			FROM {$wpdb->posts} p
			LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_wp_attachment_image_alt'
			WHERE p.post_type = 'attachment'
			AND p.post_mime_type LIKE 'image/%'
			AND (pm.meta_value IS NULL OR pm.meta_value = '')"
		);
	}
}
