<?php
/**
 * Schema markup generation using OpenAI.
 *
 * Analyzes post content to detect content type and generates valid JSON-LD
 * schema markup. Supports Article, FAQ, HowTo, and Product (WooCommerce) schemas.
 * Pro feature only.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * TopRanker Schema class.
 *
 * @since 1.0.0
 */
class TopRanker_Schema {

	/**
	 * Schema type for Article.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	const TYPE_ARTICLE = 'Article';

	/**
	 * Schema type for FAQ.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	const TYPE_FAQ = 'FAQPage';

	/**
	 * Schema type for HowTo.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	const TYPE_HOWTO = 'HowTo';

	/**
	 * Schema type for Product.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	const TYPE_PRODUCT = 'Product';

	/**
	 * API instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_API|null
	 */
	private $api = null;

	/**
	 * Optimizer instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_Optimizer|null
	 */
	private $optimizer = null;

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		// Hook into wp_head for schema output.
		add_action( 'wp_head', array( $this, 'output_schema' ), 99 );
	}

	/**
	 * Get the API instance.
	 *
	 * @since  1.0.0
	 * @return TopRanker_API
	 */
	private function get_api() {
		if ( null === $this->api ) {
			$this->api = new TopRanker_API();
		}
		return $this->api;
	}

	/**
	 * Get the optimizer instance.
	 *
	 * @since  1.0.0
	 * @return TopRanker_Optimizer
	 */
	private function get_optimizer() {
		if ( null === $this->optimizer ) {
			$this->optimizer = new TopRanker_Optimizer();
		}
		return $this->optimizer;
	}

	/**
	 * Generate schema markup for a post.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post           Post ID or WP_Post object.
	 * @param string      $focus_keyphrase Optional. Focus keyphrase.
	 * @return array|WP_Error Schema data array or WP_Error on failure.
	 */
	public function generate_schema( $post, $focus_keyphrase = '' ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return new WP_Error(
				'invalid_post',
				__( 'Invalid post.', 'topranker-ai' )
			);
		}

		// Detect content type.
		$schema_type = $this->detect_schema_type( $post );

		// Get prepared content.
		$optimizer      = $this->get_optimizer();
		$content        = $optimizer->prepare_content( $post );
		$context_prefix = $optimizer->build_context_prefix( $post );

		if ( empty( $content ) && empty( $post->post_title ) ) {
			return new WP_Error(
				'no_content',
				__( 'Post has no content to analyze.', 'topranker-ai' )
			);
		}

		// Get additional context based on post type.
		$post_context = $this->get_post_context( $post );

		// Build the prompt based on schema type.
		$prompt = $this->build_prompt( $post, $content, $schema_type, $focus_keyphrase, $post_context );

		// Call the API.
		$api      = $this->get_api();
		$messages = array(
			array(
				'role'    => 'system',
				'content' => $context_prefix,
			),
			array(
				'role'    => 'user',
				'content' => $prompt,
			),
		);

		$response = $api->chat_completion( $messages );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		// Parse and validate the response.
		$schema = $this->parse_and_validate( $response['content'], $schema_type );

		if ( is_wp_error( $schema ) ) {
			// Retry once.
			$response = $api->chat_completion( $messages );

			if ( is_wp_error( $response ) ) {
				return $response;
			}

			$schema = $this->parse_and_validate( $response['content'], $schema_type );

			if ( is_wp_error( $schema ) ) {
				return $schema;
			}
		}

		// Cache the schema.
		$this->save_schema( $post->ID, $schema );

		return array(
			'post_id'     => $post->ID,
			'schema_type' => $schema_type,
			'schema'      => $schema,
			'json_ld'     => $this->to_json_ld( $schema ),
		);
	}

	/**
	 * Detect the appropriate schema type for a post.
	 *
	 * @since 1.0.0
	 * @param WP_Post $post Post object.
	 * @return string Schema type constant.
	 */
	public function detect_schema_type( $post ) {
		// Check for WooCommerce product.
		if ( 'product' === $post->post_type && class_exists( 'WooCommerce' ) ) {
			return self::TYPE_PRODUCT;
		}

		$content = $post->post_content;

		// Check for FAQ pattern (Q&A structure).
		if ( $this->has_faq_pattern( $content ) ) {
			return self::TYPE_FAQ;
		}

		// Check for HowTo pattern (numbered steps).
		if ( $this->has_howto_pattern( $content ) ) {
			return self::TYPE_HOWTO;
		}

		// Default to Article.
		return self::TYPE_ARTICLE;
	}

	/**
	 * Check if content has FAQ pattern.
	 *
	 * Looks for Q&A structures, question headings, or FAQ blocks.
	 *
	 * @since 1.0.0
	 * @param string $content Post content.
	 * @return bool True if FAQ pattern detected.
	 */
	private function has_faq_pattern( $content ) {
		// Check for "Q:" or "Question:" patterns.
		if ( preg_match_all( '/\b(Q:|Question\s*:)/i', $content, $matches ) && count( $matches[0] ) >= 3 ) {
			return true;
		}

		// Check for multiple question mark headings.
		if ( preg_match_all( '/<h[2-4][^>]*>[^<]*\?[^<]*<\/h[2-4]>/i', $content, $matches ) && count( $matches[0] ) >= 3 ) {
			return true;
		}

		// Check for FAQ block (Gutenberg or other).
		if ( strpos( $content, 'wp-block-faq' ) !== false || strpos( $content, 'faq-section' ) !== false ) {
			return true;
		}

		// Check for "Frequently Asked Questions" heading.
		if ( preg_match( '/<h[1-3][^>]*>[^<]*frequently\s+asked\s+questions[^<]*<\/h[1-3]>/i', $content ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Check if content has HowTo pattern.
	 *
	 * Looks for numbered steps, step-by-step instructions, or "How to" titles.
	 *
	 * @since 1.0.0
	 * @param string $content Post content.
	 * @return bool True if HowTo pattern detected.
	 */
	private function has_howto_pattern( $content ) {
		// Check for "Step 1", "Step 2" pattern.
		if ( preg_match_all( '/\bStep\s*[1-9]\d*\b/i', $content, $matches ) && count( $matches[0] ) >= 3 ) {
			return true;
		}

		// Check for numbered list items (1., 2., 3.).
		if ( preg_match_all( '/<li[^>]*>\s*\d+\.\s*/i', $content, $matches ) && count( $matches[0] ) >= 3 ) {
			return true;
		}

		// Check for ordered list with multiple items.
		if ( preg_match_all( '/<ol[^>]*>.*?<\/ol>/is', $content, $ols ) ) {
			foreach ( $ols[0] as $ol ) {
				if ( preg_match_all( '/<li/i', $ol, $lis ) && count( $lis[0] ) >= 3 ) {
					return true;
				}
			}
		}

		// Check for "How to" in title or first heading.
		if ( preg_match( '/<h[1-2][^>]*>[^<]*how\s+to\b[^<]*<\/h[1-2]>/i', $content ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Get additional context for the post.
	 *
	 * @since 1.0.0
	 * @param WP_Post $post Post object.
	 * @return array Context data array.
	 */
	private function get_post_context( $post ) {
		$context = array(
			'author'     => '',
			'date'       => '',
			'modified'   => '',
			'categories' => array(),
			'image'      => '',
		);

		// Get author.
		$author = get_userdata( $post->post_author );
		if ( $author ) {
			$context['author'] = $author->display_name;
		}

		// Get dates.
		$context['date']     = get_the_date( 'c', $post );
		$context['modified'] = get_the_modified_date( 'c', $post );

		// Get categories.
		$categories = get_the_category( $post->ID );
		if ( $categories ) {
			foreach ( $categories as $category ) {
				$context['categories'][] = $category->name;
			}
		}

		// Get featured image.
		$thumbnail_id = get_post_thumbnail_id( $post->ID );
		if ( $thumbnail_id ) {
			$context['image'] = wp_get_attachment_url( $thumbnail_id );
		}

		return $context;
	}

	/**
	 * Build the AI prompt for schema generation.
	 *
	 * @since 1.0.0
	 * @param WP_Post $post            Post object.
	 * @param string  $content         Prepared content.
	 * @param string  $schema_type     Schema type.
	 * @param string  $focus_keyphrase Focus keyphrase if available.
	 * @param array   $post_context    Additional context data.
	 * @return string The formatted prompt.
	 */
	private function build_prompt( $post, $content, $schema_type, $focus_keyphrase, $post_context ) {
		$keyphrase_instruction = '';
		if ( ! empty( $focus_keyphrase ) ) {
			$keyphrase_instruction = sprintf(
				/* translators: %s: focus keyphrase */
				__( 'Focus keyphrase: "%s"', 'topranker-ai' ),
				$focus_keyphrase
			);
		}

		$context_text = '';
		if ( ! empty( $post_context['author'] ) ) {
			$context_text .= sprintf( __( 'Author: %s', 'topranker-ai' ), $post_context['author'] ) . "\n";
		}
		if ( ! empty( $post_context['date'] ) ) {
			$context_text .= sprintf( __( 'Published: %s', 'topranker-ai' ), $post_context['date'] ) . "\n";
		}
		if ( ! empty( $post_context['image'] ) ) {
			$context_text .= sprintf( __( 'Featured Image: %s', 'topranker-ai' ), $post_context['image'] ) . "\n";
		}
		if ( ! empty( $post_context['categories'] ) ) {
			$context_text .= sprintf( __( 'Categories: %s', 'topranker-ai' ), implode( ', ', $post_context['categories'] ) ) . "\n";
		}

		$schema_instructions = $this->get_schema_instructions( $schema_type );

		$prompt = sprintf(
			/* translators: 1: Schema type, 2: Post title, 3: URL, 4: Context text, 5: Keyphrase instruction, 6: Content, 7: Schema instructions */
			__(
				'Generate valid JSON-LD schema markup of type "%1$s" for this article.

ARTICLE DETAILS:
Title: %2$s
URL: %3$s
%4$s
%5$s

CONTENT:
%6$s

%7$s

REQUIREMENTS:
1. Generate valid JSON-LD that can be directly embedded in a script tag
2. Include ALL required fields for the schema type
3. Use proper @context and @type
4. Ensure all string values are properly escaped
5. Do NOT include any markdown formatting or code block markers
6. The response must be ONLY valid JSON, nothing else

Respond with the JSON-LD object only, no explanation or markdown.',
				'topranker-ai'
			),
			$schema_type,
			$post->post_title,
			get_permalink( $post->ID ),
			$context_text,
			$keyphrase_instruction,
			mb_substr( $content, 0, 4000 ),
			$schema_instructions
		);

		return $prompt;
	}

	/**
	 * Get schema-specific instructions.
	 *
	 * @since 1.0.0
	 * @param string $schema_type Schema type.
	 * @return string Instructions text.
	 */
	private function get_schema_instructions( $schema_type ) {
		switch ( $schema_type ) {
			case self::TYPE_FAQ:
				return __(
					'SCHEMA SPECIFIC REQUIREMENTS (FAQPage):
- Extract all Q&A pairs from the content
- Each question should be in "mainEntity" as a Question type
- Each answer should be the "acceptedAnswer" as an Answer type
- Minimum 3 Q&A pairs required
- Format:
{
  "@context": "https://schema.org",
  "@type": "FAQPage",
  "mainEntity": [
    {
      "@type": "Question",
      "name": "question text",
      "acceptedAnswer": {
        "@type": "Answer",
        "text": "answer text"
      }
    }
  ]
}',
					'topranker-ai'
				);

			case self::TYPE_HOWTO:
				return __(
					'SCHEMA SPECIFIC REQUIREMENTS (HowTo):
- Extract the step-by-step instructions
- Include name, description, totalTime (estimate), and step array
- Each step should have @type "HowToStep" with name and text
- Include image if featured image is available
- Format:
{
  "@context": "https://schema.org",
  "@type": "HowTo",
  "name": "How to title",
  "description": "Brief description",
  "totalTime": "PT30M",
  "image": "image URL if available",
  "step": [
    {
      "@type": "HowToStep",
      "name": "Step title",
      "text": "Step instructions"
    }
  ]
}',
					'topranker-ai'
				);

			case self::TYPE_PRODUCT:
				return __(
					'SCHEMA SPECIFIC REQUIREMENTS (Product):
- Include name, description, image, offers
- The offers should include price, priceCurrency, availability, url
- Include brand if identifiable
- Include sku if available
- Format:
{
  "@context": "https://schema.org",
  "@type": "Product",
  "name": "Product name",
  "description": "Product description",
  "image": "image URL",
  "brand": {"@type": "Brand", "name": "brand name"},
  "sku": "SKU if available",
  "offers": {
    "@type": "Offer",
    "price": "99.99",
    "priceCurrency": "USD",
    "availability": "https://schema.org/InStock",
    "url": "product URL"
  }
}',
					'topranker-ai'
				);

			case self::TYPE_ARTICLE:
			default:
				return __(
					'SCHEMA SPECIFIC REQUIREMENTS (Article):
- Include headline, description, author, datePublished, dateModified
- Include image if featured image is available
- Include publisher with logo
- Format:
{
  "@context": "https://schema.org",
  "@type": "Article",
  "headline": "article title (max 110 chars)",
  "description": "brief description",
  "author": {"@type": "Person", "name": "author name"},
  "publisher": {"@type": "Organization", "name": "site name", "logo": {"@type": "ImageObject", "url": "logo URL"}},
  "datePublished": "ISO date",
  "dateModified": "ISO date",
  "image": "featured image URL"
}',
					'topranker-ai'
				);
		}
	}

	/**
	 * Parse and validate the schema response.
	 *
	 * @since 1.0.0
	 * @param string $content     Raw response content.
	 * @param string $schema_type Expected schema type.
	 * @return array|WP_Error Validated schema array or WP_Error.
	 */
	private function parse_and_validate( $content, $schema_type ) {
		$api    = $this->get_api();
		$parsed = $api->parse_json_response( $content );

		if ( is_wp_error( $parsed ) ) {
			return $parsed;
		}

		// Validate basic structure.
		if ( ! isset( $parsed['@context'] ) || ! isset( $parsed['@type'] ) ) {
			return new WP_Error(
				'invalid_schema',
				__( 'Schema is missing required @context or @type.', 'topranker-ai' )
			);
		}

		// Validate @context.
		if ( strpos( $parsed['@context'], 'schema.org' ) === false ) {
			$parsed['@context'] = 'https://schema.org';
		}

		// Validate schema type matches expected.
		$actual_type = $parsed['@type'];
		if ( $actual_type !== $schema_type ) {
			// Allow NewsArticle or BlogPosting as Article variants.
			$article_variants = array( 'Article', 'NewsArticle', 'BlogPosting' );
			if ( self::TYPE_ARTICLE === $schema_type && in_array( $actual_type, $article_variants, true ) ) {
				// Acceptable variant.
			} else {
				// Force correct type.
				$parsed['@type'] = $schema_type;
			}
		}

		// Validate type-specific requirements.
		$validation = $this->validate_type_requirements( $parsed, $schema_type );

		if ( is_wp_error( $validation ) ) {
			return $validation;
		}

		return $parsed;
	}

	/**
	 * Validate type-specific schema requirements.
	 *
	 * @since 1.0.0
	 * @param array  $schema      Parsed schema array.
	 * @param string $schema_type Schema type.
	 * @return bool|WP_Error True on success, WP_Error on failure.
	 */
	private function validate_type_requirements( $schema, $schema_type ) {
		switch ( $schema_type ) {
			case self::TYPE_FAQ:
				if ( ! isset( $schema['mainEntity'] ) || ! is_array( $schema['mainEntity'] ) ) {
					return new WP_Error(
						'invalid_faq',
						__( 'FAQ schema is missing mainEntity array.', 'topranker-ai' )
					);
				}
				if ( count( $schema['mainEntity'] ) < 1 ) {
					return new WP_Error(
						'invalid_faq',
						__( 'FAQ schema must have at least one question.', 'topranker-ai' )
					);
				}
				foreach ( $schema['mainEntity'] as $entity ) {
					if ( ! isset( $entity['name'] ) || ! isset( $entity['acceptedAnswer'] ) ) {
						return new WP_Error(
							'invalid_faq',
							__( 'FAQ question is missing name or acceptedAnswer.', 'topranker-ai' )
						);
					}
				}
				break;

			case self::TYPE_HOWTO:
				if ( ! isset( $schema['name'] ) ) {
					return new WP_Error(
						'invalid_howto',
						__( 'HowTo schema is missing name.', 'topranker-ai' )
					);
				}
				if ( ! isset( $schema['step'] ) || ! is_array( $schema['step'] ) ) {
					return new WP_Error(
						'invalid_howto',
						__( 'HowTo schema is missing step array.', 'topranker-ai' )
					);
				}
				if ( count( $schema['step'] ) < 2 ) {
					return new WP_Error(
						'invalid_howto',
						__( 'HowTo schema must have at least 2 steps.', 'topranker-ai' )
					);
				}
				break;

			case self::TYPE_PRODUCT:
				if ( ! isset( $schema['name'] ) ) {
					return new WP_Error(
						'invalid_product',
						__( 'Product schema is missing name.', 'topranker-ai' )
					);
				}
				if ( ! isset( $schema['offers'] ) ) {
					return new WP_Error(
						'invalid_product',
						__( 'Product schema is missing offers.', 'topranker-ai' )
					);
				}
				break;

			case self::TYPE_ARTICLE:
			default:
				if ( ! isset( $schema['headline'] ) ) {
					return new WP_Error(
						'invalid_article',
						__( 'Article schema is missing headline.', 'topranker-ai' )
					);
				}
				break;
		}

		return true;
	}

	/**
	 * Convert schema array to JSON-LD script tag content.
	 *
	 * @since 1.0.0
	 * @param array $schema Schema array.
	 * @return string JSON-LD string.
	 */
	public function to_json_ld( $schema ) {
		return wp_json_encode( $schema, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT );
	}

	/**
	 * Save schema to post meta.
	 *
	 * @since 1.0.0
	 * @param int   $post_id Post ID.
	 * @param array $schema  Schema array.
	 * @return bool True on success.
	 */
	public function save_schema( $post_id, $schema ) {
		$json_ld = $this->to_json_ld( $schema );
		return (bool) update_post_meta( $post_id, '_topranker_schema', $json_ld );
	}

	/**
	 * Get saved schema for a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return array|null Schema array or null if not found.
	 */
	public function get_schema( $post_id ) {
		$json_ld = get_post_meta( $post_id, '_topranker_schema', true );

		if ( empty( $json_ld ) ) {
			return null;
		}

		$schema = json_decode( $json_ld, true );

		if ( JSON_ERROR_NONE !== json_last_error() ) {
			return null;
		}

		return $schema;
	}

	/**
	 * Check if post has schema.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True if schema exists.
	 */
	public function has_schema( $post_id ) {
		$schema = get_post_meta( $post_id, '_topranker_schema', true );
		return ! empty( $schema );
	}

	/**
	 * Delete schema for a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True on success.
	 */
	public function delete_schema( $post_id ) {
		return delete_post_meta( $post_id, '_topranker_schema' );
	}

	/**
	 * Output schema markup in wp_head.
	 *
	 * Only outputs if:
	 * - We're on a singular post
	 * - The post has TopRanker schema
	 * - SEO mode allows output
	 *
	 * @since 1.0.0
	 */
	public function output_schema() {
		if ( ! is_singular() ) {
			return;
		}

		$post_id = get_the_ID();

		if ( ! $post_id ) {
			return;
		}

		// Check if we should output meta tags.
		$seo_mode = get_option( 'topranker_seo_mode', 'standalone' );

		// In "suggest" mode, don't output anything.
		if ( 'suggest' === $seo_mode ) {
			return;
		}

		// Get the schema.
		$json_ld = get_post_meta( $post_id, '_topranker_schema', true );

		if ( empty( $json_ld ) ) {
			return;
		}

		// Validate it's proper JSON.
		$test = json_decode( $json_ld, true );
		if ( JSON_ERROR_NONE !== json_last_error() ) {
			return;
		}

		// Check for existing schema from other plugins.
		if ( $this->has_existing_schema() ) {
			return;
		}

		// Output the schema.
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON-LD should not be escaped.
		printf(
			"\n<!-- TopRanker AI Schema Markup -->\n<script type=\"application/ld+json\">\n%s\n</script>\n",
			$json_ld
		);
	}

	/**
	 * Check if another plugin is outputting schema.
	 *
	 * @since 1.0.0
	 * @return bool True if schema already exists.
	 */
	private function has_existing_schema() {
		// Check for Yoast SEO schema.
		if ( class_exists( 'WPSEO_Schema' ) || has_action( 'wpseo_json_ld' ) ) {
			return true;
		}

		// Check for RankMath schema.
		if ( class_exists( 'RankMath' ) && has_action( 'rank_math/json_ld' ) ) {
			return true;
		}

		// Check for SEOPress schema.
		if ( function_exists( 'seopress_init' ) ) {
			$seopress_schema = get_option( 'seopress_pro_option_name' );
			if ( isset( $seopress_schema['seopress_rich_snippets_enable'] ) && '1' === $seopress_schema['seopress_rich_snippets_enable'] ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Generate Product schema for WooCommerce products.
	 *
	 * This is called from class-topranker-woocommerce.php with product-specific data.
	 *
	 * @since 1.0.0
	 * @param int   $product_id   Product ID.
	 * @param array $product_data Product data array from WooCommerce integration.
	 * @return array|WP_Error Schema data or WP_Error.
	 */
	public function generate_product_schema( $product_id, $product_data ) {
		if ( ! class_exists( 'WooCommerce' ) ) {
			return new WP_Error(
				'no_woocommerce',
				__( 'WooCommerce is not active.', 'topranker-ai' )
			);
		}

		$product = wc_get_product( $product_id );

		if ( ! $product ) {
			return new WP_Error(
				'invalid_product',
				__( 'Invalid product.', 'topranker-ai' )
			);
		}

		// Build product schema directly.
		$schema = array(
			'@context'    => 'https://schema.org',
			'@type'       => 'Product',
			'name'        => $product->get_name(),
			'description' => wp_strip_all_tags( $product->get_description() ),
			'url'         => get_permalink( $product_id ),
		);

		// Add image.
		$image_id = $product->get_image_id();
		if ( $image_id ) {
			$schema['image'] = wp_get_attachment_url( $image_id );
		}

		// Add SKU.
		$sku = $product->get_sku();
		if ( $sku ) {
			$schema['sku'] = $sku;
		}

		// Add brand if available.
		if ( ! empty( $product_data['brand'] ) ) {
			$schema['brand'] = array(
				'@type' => 'Brand',
				'name'  => $product_data['brand'],
			);
		}

		// Add offers.
		$schema['offers'] = array(
			'@type'         => 'Offer',
			'url'           => get_permalink( $product_id ),
			'priceCurrency' => get_woocommerce_currency(),
			'price'         => $product->get_price(),
			'availability'  => $product->is_in_stock() ? 'https://schema.org/InStock' : 'https://schema.org/OutOfStock',
		);

		// Add aggregate rating if reviews exist.
		$review_count = $product->get_review_count();
		if ( $review_count > 0 ) {
			$schema['aggregateRating'] = array(
				'@type'       => 'AggregateRating',
				'ratingValue' => $product->get_average_rating(),
				'reviewCount' => $review_count,
			);
		}

		// Save and return.
		$this->save_schema( $product_id, $schema );

		return array(
			'post_id'     => $product_id,
			'schema_type' => self::TYPE_PRODUCT,
			'schema'      => $schema,
			'json_ld'     => $this->to_json_ld( $schema ),
		);
	}

	/**
	 * Get all supported schema types.
	 *
	 * @since  1.0.0
	 * @return array Array of schema type labels.
	 */
	public function get_schema_types() {
		return array(
			self::TYPE_ARTICLE => __( 'Article', 'topranker-ai' ),
			self::TYPE_FAQ     => __( 'FAQ Page', 'topranker-ai' ),
			self::TYPE_HOWTO   => __( 'How-To', 'topranker-ai' ),
			self::TYPE_PRODUCT => __( 'Product', 'topranker-ai' ),
		);
	}

	/**
	 * Get human-readable schema type name.
	 *
	 * @since 1.0.0
	 * @param string $type Schema type constant.
	 * @return string Human-readable name.
	 */
	public function get_schema_type_name( $type ) {
		$types = $this->get_schema_types();
		return isset( $types[ $type ] ) ? $types[ $type ] : $type;
	}
}
