<?php
/**
 * SEO Audit/Score System (Pro).
 *
 * Scores each post 0-100 based on 14 SEO criteria including meta data,
 * content quality, internal links, and schema markup.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * TopRanker SEO Audit class.
 *
 * @since 1.0.0
 */
class TopRanker_SEO_Audit {

	/**
	 * Maximum possible score.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const MAX_SCORE = 100;

	/**
	 * Score thresholds.
	 *
	 * @since 1.0.0
	 * @var   array
	 */
	const THRESHOLDS = array(
		'good'    => 80,
		'warning' => 50,
	);

	/**
	 * Scoring criteria with point values.
	 *
	 * @since 1.0.0
	 * @var   array
	 */
	private $criteria = array(
		'has_meta_title'             => 10,
		'meta_title_length'          => 5,
		'meta_title_has_keyphrase'   => 5,
		'has_meta_description'       => 10,
		'meta_description_length'    => 5,
		'meta_description_keyphrase' => 5,
		'has_focus_keyphrase'        => 10,
		'has_excerpt'                => 5,
		'images_have_alt'            => 10,
		'has_internal_links'         => 10,
		'content_length'             => 5,
		'has_headings'               => 5,
		'keyphrase_in_first_para'    => 5,
		'has_schema'                 => 10,
	);

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		// Class is loaded but hooks are initialized elsewhere.
	}

	/**
	 * Calculate SEO score for a post.
	 *
	 * @since  1.0.0
	 * @param  int|WP_Post $post Post ID or WP_Post object.
	 * @return array Score data including total score and individual checks.
	 */
	public function calculate_score( $post ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return array(
				'score'  => 0,
				'level'  => 'missing',
				'checks' => array(),
			);
		}

		$checks = array();
		$score  = 0;

		// Get focus keyphrase for checks that need it.
		$focus_keyphrase = get_post_meta( $post->ID, '_topranker_focus_keyphrase', true );
		$focus_keyphrase = strtolower( trim( $focus_keyphrase ) );

		// 1. Has meta title (10 points).
		$meta_title          = get_post_meta( $post->ID, '_topranker_meta_title', true );
		$has_meta_title      = ! empty( $meta_title );
		$checks['has_meta_title'] = array(
			'passed' => $has_meta_title,
			'points' => $has_meta_title ? $this->criteria['has_meta_title'] : 0,
			'max'    => $this->criteria['has_meta_title'],
			'label'  => __( 'Meta title exists', 'topranker-ai' ),
			'tip'    => $has_meta_title
				? __( 'Great! You have a meta title set.', 'topranker-ai' )
				: __( 'Add a meta title to improve SEO.', 'topranker-ai' ),
		);
		$score += $checks['has_meta_title']['points'];

		// 2. Meta title length 50-60 chars (5 points).
		$meta_title_length       = mb_strlen( $meta_title );
		$meta_title_length_ok    = $meta_title_length >= 50 && $meta_title_length <= 60;
		$checks['meta_title_length'] = array(
			'passed' => $meta_title_length_ok,
			'points' => $meta_title_length_ok ? $this->criteria['meta_title_length'] : 0,
			'max'    => $this->criteria['meta_title_length'],
			'label'  => __( 'Meta title length is optimal (50-60 chars)', 'topranker-ai' ),
			'value'  => $meta_title_length,
			'tip'    => $meta_title_length_ok
				? sprintf(
					/* translators: %d: Character count */
					__( 'Perfect! Your meta title is %d characters.', 'topranker-ai' ),
					$meta_title_length
				)
				: sprintf(
					/* translators: %d: Character count */
					__( 'Your meta title is %d characters. Aim for 50-60 characters.', 'topranker-ai' ),
					$meta_title_length
				),
		);
		$score += $checks['meta_title_length']['points'];

		// 3. Meta title contains keyphrase (5 points).
		$meta_title_has_keyphrase = false;
		if ( $has_meta_title && ! empty( $focus_keyphrase ) ) {
			$meta_title_has_keyphrase = false !== mb_stripos( strtolower( $meta_title ), $focus_keyphrase );
		}
		$checks['meta_title_has_keyphrase'] = array(
			'passed' => $meta_title_has_keyphrase,
			'points' => $meta_title_has_keyphrase ? $this->criteria['meta_title_has_keyphrase'] : 0,
			'max'    => $this->criteria['meta_title_has_keyphrase'],
			'label'  => __( 'Meta title contains focus keyphrase', 'topranker-ai' ),
			'tip'    => $meta_title_has_keyphrase
				? __( 'Great! Your keyphrase is in the meta title.', 'topranker-ai' )
				: __( 'Include your focus keyphrase in the meta title.', 'topranker-ai' ),
		);
		$score += $checks['meta_title_has_keyphrase']['points'];

		// 4. Has meta description (10 points).
		$meta_description     = get_post_meta( $post->ID, '_topranker_meta_description', true );
		$has_meta_description = ! empty( $meta_description );
		$checks['has_meta_description'] = array(
			'passed' => $has_meta_description,
			'points' => $has_meta_description ? $this->criteria['has_meta_description'] : 0,
			'max'    => $this->criteria['has_meta_description'],
			'label'  => __( 'Meta description exists', 'topranker-ai' ),
			'tip'    => $has_meta_description
				? __( 'Great! You have a meta description set.', 'topranker-ai' )
				: __( 'Add a meta description to improve click-through rates.', 'topranker-ai' ),
		);
		$score += $checks['has_meta_description']['points'];

		// 5. Meta description length 120-155 chars (5 points).
		$meta_desc_length    = mb_strlen( $meta_description );
		$meta_desc_length_ok = $meta_desc_length >= 120 && $meta_desc_length <= 155;
		$checks['meta_description_length'] = array(
			'passed' => $meta_desc_length_ok,
			'points' => $meta_desc_length_ok ? $this->criteria['meta_description_length'] : 0,
			'max'    => $this->criteria['meta_description_length'],
			'label'  => __( 'Meta description length is optimal (120-155 chars)', 'topranker-ai' ),
			'value'  => $meta_desc_length,
			'tip'    => $meta_desc_length_ok
				? sprintf(
					/* translators: %d: Character count */
					__( 'Perfect! Your meta description is %d characters.', 'topranker-ai' ),
					$meta_desc_length
				)
				: sprintf(
					/* translators: %d: Character count */
					__( 'Your meta description is %d characters. Aim for 120-155 characters.', 'topranker-ai' ),
					$meta_desc_length
				),
		);
		$score += $checks['meta_description_length']['points'];

		// 6. Meta description contains keyphrase (5 points).
		$meta_desc_has_keyphrase = false;
		if ( $has_meta_description && ! empty( $focus_keyphrase ) ) {
			$meta_desc_has_keyphrase = false !== mb_stripos( strtolower( $meta_description ), $focus_keyphrase );
		}
		$checks['meta_description_keyphrase'] = array(
			'passed' => $meta_desc_has_keyphrase,
			'points' => $meta_desc_has_keyphrase ? $this->criteria['meta_description_keyphrase'] : 0,
			'max'    => $this->criteria['meta_description_keyphrase'],
			'label'  => __( 'Meta description contains focus keyphrase', 'topranker-ai' ),
			'tip'    => $meta_desc_has_keyphrase
				? __( 'Great! Your keyphrase is in the meta description.', 'topranker-ai' )
				: __( 'Include your focus keyphrase in the meta description.', 'topranker-ai' ),
		);
		$score += $checks['meta_description_keyphrase']['points'];

		// 7. Has focus keyphrase (10 points).
		$has_focus_keyphrase = ! empty( $focus_keyphrase );
		$checks['has_focus_keyphrase'] = array(
			'passed' => $has_focus_keyphrase,
			'points' => $has_focus_keyphrase ? $this->criteria['has_focus_keyphrase'] : 0,
			'max'    => $this->criteria['has_focus_keyphrase'],
			'label'  => __( 'Focus keyphrase is set', 'topranker-ai' ),
			'tip'    => $has_focus_keyphrase
				? __( 'Great! You have a focus keyphrase.', 'topranker-ai' )
				: __( 'Set a focus keyphrase to guide your SEO optimization.', 'topranker-ai' ),
		);
		$score += $checks['has_focus_keyphrase']['points'];

		// 8. Has excerpt (5 points).
		$has_excerpt = ! empty( $post->post_excerpt );
		$checks['has_excerpt'] = array(
			'passed' => $has_excerpt,
			'points' => $has_excerpt ? $this->criteria['has_excerpt'] : 0,
			'max'    => $this->criteria['has_excerpt'],
			'label'  => __( 'Excerpt is set', 'topranker-ai' ),
			'tip'    => $has_excerpt
				? __( 'Great! You have an excerpt.', 'topranker-ai' )
				: __( 'Add an excerpt for better previews and SEO.', 'topranker-ai' ),
		);
		$score += $checks['has_excerpt']['points'];

		// 9. All images have alt tags (10 points).
		$alt_tag_result         = $this->check_image_alt_tags( $post );
		$images_have_alt        = $alt_tag_result['all_have_alt'];
		$checks['images_have_alt'] = array(
			'passed' => $images_have_alt,
			'points' => $images_have_alt ? $this->criteria['images_have_alt'] : 0,
			'max'    => $this->criteria['images_have_alt'],
			'label'  => __( 'All images have alt tags', 'topranker-ai' ),
			'value'  => array(
				'total'   => $alt_tag_result['total'],
				'with_alt' => $alt_tag_result['with_alt'],
			),
			'tip'    => $images_have_alt
				? __( 'Great! All images have alt text.', 'topranker-ai' )
				: sprintf(
					/* translators: 1: Number with alt, 2: Total images */
					__( '%1$d of %2$d images have alt text. Add alt text to improve accessibility and SEO.', 'topranker-ai' ),
					$alt_tag_result['with_alt'],
					$alt_tag_result['total']
				),
		);
		$score += $checks['images_have_alt']['points'];

		// 10. Has at least 2 internal links (10 points).
		$internal_links_result = $this->count_internal_links( $post );
		$has_internal_links    = $internal_links_result >= 2;
		$checks['has_internal_links'] = array(
			'passed' => $has_internal_links,
			'points' => $has_internal_links ? $this->criteria['has_internal_links'] : 0,
			'max'    => $this->criteria['has_internal_links'],
			'label'  => __( 'Has at least 2 internal links', 'topranker-ai' ),
			'value'  => $internal_links_result,
			'tip'    => $has_internal_links
				? sprintf(
					/* translators: %d: Number of internal links */
					__( 'Great! You have %d internal links.', 'topranker-ai' ),
					$internal_links_result
				)
				: sprintf(
					/* translators: %d: Number of internal links */
					__( 'You have %d internal links. Add at least 2 to improve site navigation.', 'topranker-ai' ),
					$internal_links_result
				),
		);
		$score += $checks['has_internal_links']['points'];

		// 11. Content length > 300 words (5 points).
		$word_count        = $this->get_word_count( $post );
		$content_length_ok = $word_count > 300;
		$checks['content_length'] = array(
			'passed' => $content_length_ok,
			'points' => $content_length_ok ? $this->criteria['content_length'] : 0,
			'max'    => $this->criteria['content_length'],
			'label'  => __( 'Content has more than 300 words', 'topranker-ai' ),
			'value'  => $word_count,
			'tip'    => $content_length_ok
				? sprintf(
					/* translators: %d: Word count */
					__( 'Great! Your content has %d words.', 'topranker-ai' ),
					$word_count
				)
				: sprintf(
					/* translators: %d: Word count */
					__( 'Your content has %d words. Aim for at least 300 words.', 'topranker-ai' ),
					$word_count
				),
		);
		$score += $checks['content_length']['points'];

		// 12. Has heading tags (H2, H3) (5 points).
		$headings_result = $this->check_headings( $post );
		$has_headings    = $headings_result['has_h2'] || $headings_result['has_h3'];
		$checks['has_headings'] = array(
			'passed' => $has_headings,
			'points' => $has_headings ? $this->criteria['has_headings'] : 0,
			'max'    => $this->criteria['has_headings'],
			'label'  => __( 'Content has subheadings (H2, H3)', 'topranker-ai' ),
			'value'  => array(
				'h2_count' => $headings_result['h2_count'],
				'h3_count' => $headings_result['h3_count'],
			),
			'tip'    => $has_headings
				? __( 'Great! Your content has subheadings.', 'topranker-ai' )
				: __( 'Add H2 or H3 headings to structure your content.', 'topranker-ai' ),
		);
		$score += $checks['has_headings']['points'];

		// 13. Keyphrase in first paragraph (5 points).
		$keyphrase_in_first_para = false;
		if ( ! empty( $focus_keyphrase ) ) {
			$first_paragraph         = $this->get_first_paragraph( $post );
			$keyphrase_in_first_para = false !== mb_stripos( strtolower( $first_paragraph ), $focus_keyphrase );
		}
		$checks['keyphrase_in_first_para'] = array(
			'passed' => $keyphrase_in_first_para,
			'points' => $keyphrase_in_first_para ? $this->criteria['keyphrase_in_first_para'] : 0,
			'max'    => $this->criteria['keyphrase_in_first_para'],
			'label'  => __( 'Keyphrase appears in first paragraph', 'topranker-ai' ),
			'tip'    => $keyphrase_in_first_para
				? __( 'Great! Your keyphrase is in the first paragraph.', 'topranker-ai' )
				: __( 'Include your focus keyphrase in the first paragraph.', 'topranker-ai' ),
		);
		$score += $checks['keyphrase_in_first_para']['points'];

		// 14. Has schema markup (10 points).
		$schema     = get_post_meta( $post->ID, '_topranker_schema', true );
		$has_schema = ! empty( $schema );
		$checks['has_schema'] = array(
			'passed' => $has_schema,
			'points' => $has_schema ? $this->criteria['has_schema'] : 0,
			'max'    => $this->criteria['has_schema'],
			'label'  => __( 'Has schema markup', 'topranker-ai' ),
			'tip'    => $has_schema
				? __( 'Great! You have schema markup.', 'topranker-ai' )
				: __( 'Add schema markup to enhance search results.', 'topranker-ai' ),
		);
		$score += $checks['has_schema']['points'];

		// Determine level based on score.
		$level = $this->get_score_level( $score );

		return array(
			'score'      => $score,
			'max_score'  => self::MAX_SCORE,
			'percentage' => round( ( $score / self::MAX_SCORE ) * 100 ),
			'level'      => $level,
			'checks'     => $checks,
			'passed'     => $this->count_passed_checks( $checks ),
			'total'      => count( $checks ),
		);
	}

	/**
	 * Calculate and store SEO score for a post.
	 *
	 * @since  1.0.0
	 * @param  int|WP_Post $post Post ID or WP_Post object.
	 * @return int The calculated score.
	 */
	public function update_score( $post ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return 0;
		}

		$result = $this->calculate_score( $post );
		$score  = $result['score'];

		update_post_meta( $post->ID, '_topranker_seo_score', $score );

		return $score;
	}

	/**
	 * Get the stored SEO score for a post.
	 *
	 * @since  1.0.0
	 * @param  int|WP_Post $post Post ID or WP_Post object.
	 * @return int|null The stored score or null if not set.
	 */
	public function get_score( $post ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return null;
		}

		$score = get_post_meta( $post->ID, '_topranker_seo_score', true );

		if ( '' === $score ) {
			return null;
		}

		return (int) $score;
	}

	/**
	 * Get the score level (good, warning, bad).
	 *
	 * @since  1.0.0
	 * @param  int $score The SEO score.
	 * @return string Level: 'good', 'warning', or 'bad'.
	 */
	public function get_score_level( $score ) {
		if ( $score >= self::THRESHOLDS['good'] ) {
			return 'good';
		} elseif ( $score >= self::THRESHOLDS['warning'] ) {
			return 'warning';
		}
		return 'bad';
	}

	/**
	 * Get the score indicator for display.
	 *
	 * @since  1.0.0
	 * @param  int $score The SEO score.
	 * @return array Indicator data with class, icon, and label.
	 */
	public function get_score_indicator( $score ) {
		$level = $this->get_score_level( $score );

		switch ( $level ) {
			case 'good':
				return array(
					'level'      => $level,
					'class'      => 'topranker-score-good',
					'icon'       => 'dashicons-yes-alt',
					'icon_class' => 'topranker-icon-good',
					'label'      => __( 'Good', 'topranker-ai' ),
					'aria_label' => sprintf(
						/* translators: %d: SEO score */
						__( 'Good SEO score: %d out of 100', 'topranker-ai' ),
						$score
					),
				);
			case 'warning':
				return array(
					'level'      => $level,
					'class'      => 'topranker-score-warning',
					'icon'       => 'dashicons-warning',
					'icon_class' => 'topranker-icon-warning',
					'label'      => __( 'Needs improvement', 'topranker-ai' ),
					'aria_label' => sprintf(
						/* translators: %d: SEO score */
						__( 'SEO score needs improvement: %d out of 100', 'topranker-ai' ),
						$score
					),
				);
			case 'bad':
			default:
				return array(
					'level'      => $level,
					'class'      => 'topranker-score-bad',
					'icon'       => 'dashicons-dismiss',
					'icon_class' => 'topranker-icon-bad',
					'label'      => __( 'Poor', 'topranker-ai' ),
					'aria_label' => sprintf(
						/* translators: %d: SEO score */
						__( 'Poor SEO score: %d out of 100', 'topranker-ai' ),
						$score
					),
				);
		}
	}

	/**
	 * Check image alt tags in a post.
	 *
	 * @since  1.0.0
	 * @param  WP_Post $post Post object.
	 * @return array Alt tag check result.
	 */
	private function check_image_alt_tags( $post ) {
		$content = $post->post_content;
		$total   = 0;
		$with_alt = 0;

		// Find all img tags.
		if ( preg_match_all( '/<img[^>]+>/i', $content, $matches ) ) {
			$total = count( $matches[0] );

			foreach ( $matches[0] as $img_tag ) {
				// Check if alt attribute exists and is not empty.
				if ( preg_match( '/alt=["\']([^"\']+)["\']/', $img_tag, $alt_match ) ) {
					if ( ! empty( trim( $alt_match[1] ) ) ) {
						++$with_alt;
					}
				}
			}
		}

		// Also check featured image.
		$featured_id = get_post_thumbnail_id( $post->ID );
		if ( $featured_id ) {
			++$total;
			$featured_alt = get_post_meta( $featured_id, '_wp_attachment_image_alt', true );
			if ( ! empty( $featured_alt ) ) {
				++$with_alt;
			}
		}

		// If no images, consider it passed.
		if ( 0 === $total ) {
			return array(
				'all_have_alt' => true,
				'total'        => 0,
				'with_alt'     => 0,
			);
		}

		return array(
			'all_have_alt' => $total === $with_alt,
			'total'        => $total,
			'with_alt'     => $with_alt,
		);
	}

	/**
	 * Count internal links in a post.
	 *
	 * @since  1.0.0
	 * @param  WP_Post $post Post object.
	 * @return int Number of internal links.
	 */
	private function count_internal_links( $post ) {
		$content  = $post->post_content;
		$site_url = home_url();
		$count    = 0;

		// Find all anchor tags.
		if ( preg_match_all( '/<a[^>]+href=["\']([^"\']+)["\'][^>]*>/i', $content, $matches ) ) {
			foreach ( $matches[1] as $url ) {
				// Check if it's an internal link.
				if ( 0 === strpos( $url, $site_url ) || 0 === strpos( $url, '/' ) ) {
					// Exclude anchors and self-links.
					if ( 0 !== strpos( $url, '#' ) && $url !== get_permalink( $post->ID ) ) {
						++$count;
					}
				}
			}
		}

		return $count;
	}

	/**
	 * Get word count for a post.
	 *
	 * @since  1.0.0
	 * @param  WP_Post $post Post object.
	 * @return int Word count.
	 */
	private function get_word_count( $post ) {
		$content = $post->post_content;

		// Strip shortcodes.
		$content = strip_shortcodes( $content );

		// Strip HTML tags.
		$content = wp_strip_all_tags( $content );

		// Decode HTML entities.
		$content = html_entity_decode( $content, ENT_QUOTES | ENT_HTML5, 'UTF-8' );

		// Trim and get word count.
		$content = trim( $content );

		if ( empty( $content ) ) {
			return 0;
		}

		return str_word_count( $content );
	}

	/**
	 * Check for heading tags in a post.
	 *
	 * @since  1.0.0
	 * @param  WP_Post $post Post object.
	 * @return array Headings check result.
	 */
	private function check_headings( $post ) {
		$content = $post->post_content;

		// Count H2 tags.
		preg_match_all( '/<h2[^>]*>/i', $content, $h2_matches );
		$h2_count = count( $h2_matches[0] );

		// Count H3 tags.
		preg_match_all( '/<h3[^>]*>/i', $content, $h3_matches );
		$h3_count = count( $h3_matches[0] );

		return array(
			'has_h2'   => $h2_count > 0,
			'has_h3'   => $h3_count > 0,
			'h2_count' => $h2_count,
			'h3_count' => $h3_count,
		);
	}

	/**
	 * Get the first paragraph of a post.
	 *
	 * @since  1.0.0
	 * @param  WP_Post $post Post object.
	 * @return string First paragraph text.
	 */
	private function get_first_paragraph( $post ) {
		$content = $post->post_content;

		// Strip shortcodes.
		$content = strip_shortcodes( $content );

		// Strip HTML tags.
		$content = wp_strip_all_tags( $content );

		// Decode HTML entities.
		$content = html_entity_decode( $content, ENT_QUOTES | ENT_HTML5, 'UTF-8' );

		// Collapse whitespace.
		$content = preg_replace( '/\n\s*\n/', "\n\n", $content );
		$content = preg_replace( '/[ \t]+/', ' ', $content );
		$content = trim( $content );

		// Split by double newlines.
		$paragraphs = preg_split( '/\n\s*\n/', $content );

		if ( ! empty( $paragraphs ) ) {
			return trim( $paragraphs[0] );
		}

		// Fallback: return first 500 characters.
		return mb_substr( $content, 0, 500 );
	}

	/**
	 * Count passed checks.
	 *
	 * @since  1.0.0
	 * @param  array $checks Array of check results.
	 * @return int Number of passed checks.
	 */
	private function count_passed_checks( $checks ) {
		$passed = 0;

		foreach ( $checks as $check ) {
			if ( $check['passed'] ) {
				++$passed;
			}
		}

		return $passed;
	}

	/**
	 * Get failed checks for a post.
	 *
	 * @since  1.0.0
	 * @param  int|WP_Post $post Post ID or WP_Post object.
	 * @return array Array of failed checks with tips.
	 */
	public function get_failed_checks( $post ) {
		$result = $this->calculate_score( $post );
		$failed = array();

		foreach ( $result['checks'] as $key => $check ) {
			if ( ! $check['passed'] ) {
				$failed[ $key ] = array(
					'label'  => $check['label'],
					'tip'    => $check['tip'],
					'points' => $check['max'],
				);
			}
		}

		return $failed;
	}

	/**
	 * Get improvement suggestions for a post.
	 *
	 * Returns a prioritized list of what to fix to improve the score.
	 *
	 * @since  1.0.0
	 * @param  int|WP_Post $post Post ID or WP_Post object.
	 * @return array Array of suggestions ordered by impact.
	 */
	public function get_suggestions( $post ) {
		$failed = $this->get_failed_checks( $post );

		// Sort by points (highest impact first).
		uasort(
			$failed,
			function( $a, $b ) {
				return $b['points'] - $a['points'];
			}
		);

		$suggestions = array();

		foreach ( $failed as $key => $check ) {
			$suggestions[] = array(
				'id'      => $key,
				'label'   => $check['label'],
				'tip'     => $check['tip'],
				'impact'  => $check['points'],
			);
		}

		return $suggestions;
	}

	/**
	 * Render SEO score column content for post list.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 */
	public function render_score_column( $post_id ) {
		$score = $this->get_score( $post_id );

		// If no score stored, calculate it.
		if ( null === $score ) {
			$score = $this->update_score( $post_id );
		}

		$indicator = $this->get_score_indicator( $score );

		printf(
			'<span class="topranker-seo-score %s" role="img" aria-label="%s" title="%s">
				<span class="dashicons %s %s"></span>
				<span class="topranker-score-value">%d</span>
			</span>',
			esc_attr( $indicator['class'] ),
			esc_attr( $indicator['aria_label'] ),
			esc_attr( $indicator['aria_label'] ),
			esc_attr( $indicator['icon'] ),
			esc_attr( $indicator['icon_class'] ),
			intval( $score )
		);
	}

	/**
	 * Get score breakdown HTML for editor display.
	 *
	 * @since  1.0.0
	 * @param  int|WP_Post $post Post ID or WP_Post object.
	 * @return string HTML for score breakdown.
	 */
	public function get_score_breakdown_html( $post ) {
		$result = $this->calculate_score( $post );

		ob_start();
		?>
		<div class="topranker-seo-audit">
			<div class="topranker-audit-header">
				<div class="topranker-audit-score <?php echo esc_attr( 'is-' . $result['level'] ); ?>">
					<span class="topranker-score-number"><?php echo intval( $result['score'] ); ?></span>
					<span class="topranker-score-max">/<?php echo intval( $result['max_score'] ); ?></span>
				</div>
				<div class="topranker-audit-summary">
					<span class="topranker-checks-passed">
						<?php
						printf(
							/* translators: 1: Passed checks, 2: Total checks */
							esc_html__( '%1$d of %2$d checks passed', 'topranker-ai' ),
							intval( $result['passed'] ),
							intval( $result['total'] )
						);
						?>
					</span>
				</div>
			</div>

			<div class="topranker-audit-checklist">
				<?php foreach ( $result['checks'] as $key => $check ) : ?>
					<div class="topranker-check-item <?php echo $check['passed'] ? 'is-passed' : 'is-failed'; ?>">
						<span class="topranker-check-icon dashicons <?php echo $check['passed'] ? 'dashicons-yes' : 'dashicons-no'; ?>"></span>
						<span class="topranker-check-label"><?php echo esc_html( $check['label'] ); ?></span>
						<span class="topranker-check-points">
							<?php
							printf(
								'+%d',
								$check['passed'] ? intval( $check['points'] ) : 0
							);
							?>
						</span>
						<span class="topranker-check-tip"><?php echo esc_html( $check['tip'] ); ?></span>
					</div>
				<?php endforeach; ?>
			</div>
		</div>
		<?php
		return ob_get_clean();
	}

	/**
	 * Get data for REST API response.
	 *
	 * @since  1.0.0
	 * @param  int|WP_Post $post Post ID or WP_Post object.
	 * @return array SEO audit data for API.
	 */
	public function get_api_data( $post ) {
		$result      = $this->calculate_score( $post );
		$suggestions = $this->get_suggestions( $post );
		$indicator   = $this->get_score_indicator( $result['score'] );

		return array(
			'score'       => $result['score'],
			'max_score'   => $result['max_score'],
			'percentage'  => $result['percentage'],
			'level'       => $result['level'],
			'indicator'   => $indicator,
			'passed'      => $result['passed'],
			'total'       => $result['total'],
			'checks'      => $result['checks'],
			'suggestions' => $suggestions,
		);
	}
}
