<?php
/**
 * WooCommerce product optimization.
 *
 * Provides specialized SEO optimization for WooCommerce products including
 * purchase-intent meta titles and descriptions, product short descriptions,
 * and product schema markup. Pro feature only.
 *
 * @package TopRanker_AI
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * TopRanker WooCommerce class.
 *
 * @since 1.0.0
 */
class TopRanker_WooCommerce {

	/**
	 * Maximum short description length.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const SHORT_DESCRIPTION_MAX_LENGTH = 400;

	/**
	 * Maximum meta title length.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const META_TITLE_MAX_LENGTH = 60;

	/**
	 * Maximum meta description length.
	 *
	 * @since 1.0.0
	 * @var   int
	 */
	const META_DESCRIPTION_MAX_LENGTH = 155;

	/**
	 * API instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_API|null
	 */
	private $api = null;

	/**
	 * Optimizer instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_Optimizer|null
	 */
	private $optimizer = null;

	/**
	 * Schema instance.
	 *
	 * @since 1.0.0
	 * @var   TopRanker_Schema|null
	 */
	private $schema = null;

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		// Only initialize if WooCommerce is active.
		if ( ! class_exists( 'WooCommerce' ) ) {
			return;
		}

		$this->init_hooks();
	}

	/**
	 * Initialize hooks.
	 *
	 * @since 1.0.0
	 */
	private function init_hooks() {
		// Filter the optimizer to use WooCommerce-specific optimization for products.
		add_filter( 'topranker_optimize_product', array( $this, 'optimize_product' ), 10, 2 );

		// Add WooCommerce data to optimization context.
		add_filter( 'topranker_post_context', array( $this, 'add_product_context' ), 10, 2 );
	}

	/**
	 * Get the API instance.
	 *
	 * @since  1.0.0
	 * @return TopRanker_API
	 */
	private function get_api() {
		if ( null === $this->api ) {
			$this->api = new TopRanker_API();
		}
		return $this->api;
	}

	/**
	 * Get the optimizer instance.
	 *
	 * @since  1.0.0
	 * @return TopRanker_Optimizer
	 */
	private function get_optimizer() {
		if ( null === $this->optimizer ) {
			$this->optimizer = new TopRanker_Optimizer();
		}
		return $this->optimizer;
	}

	/**
	 * Get the schema instance.
	 *
	 * @since  1.0.0
	 * @return TopRanker_Schema|null
	 */
	private function get_schema() {
		if ( null === $this->schema && class_exists( 'TopRanker_Schema' ) ) {
			$this->schema = new TopRanker_Schema();
		}
		return $this->schema;
	}

	/**
	 * Check if a post is a WooCommerce product.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post Post ID or WP_Post object.
	 * @return bool True if this is a product.
	 */
	public function is_product( $post ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return false;
		}

		return 'product' === $post->post_type && class_exists( 'WooCommerce' );
	}

	/**
	 * Get product data for optimization.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post Post ID or WP_Post object.
	 * @return array|WP_Error Product data array or WP_Error.
	 */
	public function get_product_data( $post ) {
		$post = get_post( $post );

		if ( ! $post || 'product' !== $post->post_type ) {
			return new WP_Error(
				'not_a_product',
				__( 'The specified post is not a WooCommerce product.', 'topranker-ai' )
			);
		}

		$product = wc_get_product( $post->ID );

		if ( ! $product ) {
			return new WP_Error(
				'invalid_product',
				__( 'Could not load the WooCommerce product.', 'topranker-ai' )
			);
		}

		// Get categories.
		$categories   = array();
		$product_cats = get_the_terms( $post->ID, 'product_cat' );
		if ( $product_cats && ! is_wp_error( $product_cats ) ) {
			foreach ( $product_cats as $cat ) {
				$categories[] = $cat->name;
			}
		}

		// Get tags.
		$tags         = array();
		$product_tags = get_the_terms( $post->ID, 'product_tag' );
		if ( $product_tags && ! is_wp_error( $product_tags ) ) {
			foreach ( $product_tags as $tag ) {
				$tags[] = $tag->name;
			}
		}

		// Get attributes.
		$attributes      = array();
		$product_attrs   = $product->get_attributes();
		if ( $product_attrs ) {
			foreach ( $product_attrs as $attr ) {
				if ( is_object( $attr ) && method_exists( $attr, 'get_name' ) ) {
					$attr_name = $attr->get_name();
					if ( $attr->is_taxonomy() ) {
						$attr_values = wc_get_product_terms( $post->ID, $attr_name, array( 'fields' => 'names' ) );
					} else {
						$attr_values = $attr->get_options();
					}
					if ( ! empty( $attr_values ) ) {
						$attributes[ $attr_name ] = $attr_values;
					}
				}
			}
		}

		// Determine brand (try common attribute names).
		$brand = $this->get_product_brand( $product, $post->ID );

		// Build product data array.
		$data = array(
			'id'                => $post->ID,
			'name'              => $product->get_name(),
			'type'              => $product->get_type(),
			'sku'               => $product->get_sku(),
			'price'             => $product->get_price(),
			'regular_price'     => $product->get_regular_price(),
			'sale_price'        => $product->get_sale_price(),
			'currency'          => get_woocommerce_currency(),
			'currency_symbol'   => get_woocommerce_currency_symbol(),
			'in_stock'          => $product->is_in_stock(),
			'stock_status'      => $product->get_stock_status(),
			'categories'        => $categories,
			'tags'              => $tags,
			'attributes'        => $attributes,
			'brand'             => $brand,
			'description'       => $product->get_description(),
			'short_description' => $product->get_short_description(),
			'average_rating'    => $product->get_average_rating(),
			'review_count'      => $product->get_review_count(),
			'is_on_sale'        => $product->is_on_sale(),
			'is_virtual'        => $product->is_virtual(),
			'is_downloadable'   => $product->is_downloadable(),
			'weight'            => $product->get_weight(),
			'dimensions'        => array(
				'length' => $product->get_length(),
				'width'  => $product->get_width(),
				'height' => $product->get_height(),
			),
		);

		// For variable products, get parent data.
		if ( 'variation' === $product->get_type() ) {
			$parent_id = $product->get_parent_id();
			if ( $parent_id ) {
				$parent_product         = wc_get_product( $parent_id );
				$data['parent_id']      = $parent_id;
				$data['parent_name']    = $parent_product ? $parent_product->get_name() : '';
			}
		}

		return $data;
	}

	/**
	 * Get product brand from various sources.
	 *
	 * @since 1.0.0
	 * @param WC_Product $product    Product object.
	 * @param int        $product_id Product ID.
	 * @return string Brand name or empty string.
	 */
	private function get_product_brand( $product, $product_id ) {
		// Check common brand taxonomy names.
		$brand_taxonomies = array( 'pa_brand', 'product_brand', 'brand' );

		foreach ( $brand_taxonomies as $taxonomy ) {
			$terms = get_the_terms( $product_id, $taxonomy );
			if ( $terms && ! is_wp_error( $terms ) ) {
				return $terms[0]->name;
			}
		}

		// Check common brand attribute names.
		$brand_attrs = array( 'brand', 'manufacturer', 'make' );
		$attributes  = $product->get_attributes();

		foreach ( $brand_attrs as $attr_name ) {
			// Check both prefixed and unprefixed.
			$keys_to_check = array( $attr_name, 'pa_' . $attr_name );

			foreach ( $keys_to_check as $key ) {
				if ( isset( $attributes[ $key ] ) ) {
					$attr = $attributes[ $key ];
					if ( is_object( $attr ) && method_exists( $attr, 'get_options' ) ) {
						$options = $attr->get_options();
						if ( ! empty( $options ) ) {
							return is_array( $options ) ? $options[0] : $options;
						}
					}
				}
			}
		}

		// Check for WooCommerce Brands plugin.
		$brands = get_the_terms( $product_id, 'product_brand' );
		if ( $brands && ! is_wp_error( $brands ) ) {
			return $brands[0]->name;
		}

		return '';
	}

	/**
	 * Optimize a WooCommerce product.
	 *
	 * Generates purchase-intent optimized meta title, meta description,
	 * and short description for products.
	 *
	 * @since 1.0.0
	 * @param int|WP_Post $post           Post ID or WP_Post object.
	 * @param string      $focus_keyphrase Optional. Focus keyphrase.
	 * @return array|WP_Error Optimization results or WP_Error.
	 */
	public function optimize_product( $post, $focus_keyphrase = '' ) {
		$post = get_post( $post );

		if ( ! $post ) {
			return new WP_Error(
				'invalid_post',
				__( 'Invalid post.', 'topranker-ai' )
			);
		}

		if ( ! $this->is_product( $post ) ) {
			return new WP_Error(
				'not_a_product',
				__( 'This is not a WooCommerce product.', 'topranker-ai' )
			);
		}

		// Get product data.
		$product_data = $this->get_product_data( $post );

		if ( is_wp_error( $product_data ) ) {
			return $product_data;
		}

		$results = array(
			'meta_title'        => null,
			'meta_description'  => null,
			'short_description' => null,
			'schema'            => null,
		);

		$errors = array();

		// Generate meta title with purchase intent.
		$title_result = $this->generate_product_meta_title( $post, $product_data, $focus_keyphrase );

		if ( is_wp_error( $title_result ) ) {
			$errors['meta_title'] = $title_result->get_error_message();
		} else {
			$results['meta_title'] = $title_result;
		}

		// Generate meta description with purchase intent.
		$description_result = $this->generate_product_meta_description( $post, $product_data, $focus_keyphrase );

		if ( is_wp_error( $description_result ) ) {
			$errors['meta_description'] = $description_result->get_error_message();
		} else {
			$results['meta_description'] = $description_result;
		}

		// Generate optimized short description.
		$short_desc_result = $this->generate_short_description( $post, $product_data );

		if ( is_wp_error( $short_desc_result ) ) {
			$errors['short_description'] = $short_desc_result->get_error_message();
		} else {
			$results['short_description'] = $short_desc_result;
		}

		// Generate product schema.
		$schema = $this->get_schema();
		if ( $schema ) {
			$schema_result = $schema->generate_product_schema( $post->ID, $product_data );

			if ( is_wp_error( $schema_result ) ) {
				$errors['schema'] = $schema_result->get_error_message();
			} else {
				$results['schema'] = $schema_result;
			}
		}

		// Check if we got at least some results.
		$has_results = false;
		foreach ( $results as $value ) {
			if ( null !== $value ) {
				$has_results = true;
				break;
			}
		}

		if ( ! $has_results ) {
			$error_messages = implode( ' ', $errors );
			return new WP_Error(
				'optimization_failed',
				/* translators: %s: error messages */
				sprintf( __( 'Product optimization failed: %s', 'topranker-ai' ), $error_messages )
			);
		}

		return array(
			'post_id'      => $post->ID,
			'product_data' => $product_data,
			'results'      => $results,
			'errors'       => $errors,
		);
	}

	/**
	 * Generate meta title for a product with purchase intent.
	 *
	 * @since 1.0.0
	 * @param WP_Post $post           Post object.
	 * @param array   $product_data   Product data array.
	 * @param string  $focus_keyphrase Optional. Focus keyphrase.
	 * @return array|WP_Error Array with 'suggestions' key or WP_Error.
	 */
	public function generate_product_meta_title( $post, $product_data, $focus_keyphrase = '' ) {
		$api       = $this->get_api();
		$optimizer = $this->get_optimizer();
		$context   = $optimizer->build_context_prefix( $post );

		$prompt = $this->build_product_meta_title_prompt( $product_data, $focus_keyphrase );

		$messages = array(
			array(
				'role'    => 'system',
				'content' => $context,
			),
			array(
				'role'    => 'user',
				'content' => $prompt,
			),
		);

		$response = $api->chat_completion( $messages );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$parsed = $api->parse_json_response( $response['content'] );

		if ( is_wp_error( $parsed ) ) {
			// Retry once.
			$response = $api->chat_completion( $messages );

			if ( is_wp_error( $response ) ) {
				return $response;
			}

			$parsed = $api->parse_json_response( $response['content'] );

			if ( is_wp_error( $parsed ) ) {
				return $parsed;
			}
		}

		return $this->validate_title_response( $parsed );
	}

	/**
	 * Build the product meta title prompt.
	 *
	 * @since 1.0.0
	 * @param array  $product_data   Product data array.
	 * @param string $focus_keyphrase Focus keyphrase if provided.
	 * @return string The formatted prompt.
	 */
	private function build_product_meta_title_prompt( $product_data, $focus_keyphrase ) {
		$site_name = get_bloginfo( 'name' );

		$keyphrase_instruction = '';
		if ( ! empty( $focus_keyphrase ) ) {
			$keyphrase_instruction = sprintf(
				/* translators: %s: focus keyphrase */
				__( 'Include the focus keyphrase "%s" naturally.', 'topranker-ai' ),
				$focus_keyphrase
			);
		}

		$price_info = '';
		if ( ! empty( $product_data['price'] ) ) {
			$price_info = sprintf(
				__( 'Price: %s%s', 'topranker-ai' ),
				$product_data['currency_symbol'],
				$product_data['price']
			);
			if ( $product_data['is_on_sale'] && ! empty( $product_data['regular_price'] ) ) {
				$price_info .= sprintf(
					__( ' (On Sale, was %s%s)', 'topranker-ai' ),
					$product_data['currency_symbol'],
					$product_data['regular_price']
				);
			}
		}

		$categories_text = ! empty( $product_data['categories'] )
			? implode( ', ', $product_data['categories'] )
			: '';

		$brand_info = '';
		if ( ! empty( $product_data['brand'] ) ) {
			$brand_info = sprintf( __( 'Brand: %s', 'topranker-ai' ), $product_data['brand'] );
		}

		$prompt = sprintf(
			/* translators: 1: Product name, 2: Price info, 3: Brand info, 4: Categories, 5: Short description, 6: Keyphrase instruction, 7: Site name, 8: Max length */
			__(
				'Generate 3 SEO-optimized meta title suggestions for this product, optimized for purchase intent.

PRODUCT DETAILS:
Product Name: %1$s
%2$s
%3$s
Categories: %4$s
Short Description: %5$s

Requirements:
- Maximum %8$d characters per title (strict limit)
- Optimize for purchase intent (people ready to buy)
- Include compelling elements like price point, brand, or key benefit if space allows
- Make it click-worthy for shoppers, not just informative
%6$s
- Optionally append " | %7$s" if it fits within the character limit
- Each title should offer a different angle (e.g., brand focus, price focus, feature focus)

Respond with valid JSON only, in this exact format:
{
  "suggestions": [
    {"title": "First title suggestion", "rank": 1},
    {"title": "Second title suggestion", "rank": 2},
    {"title": "Third title suggestion", "rank": 3}
  ]
}

Rank them by quality for driving sales, with 1 being the best.',
				'topranker-ai'
			),
			$product_data['name'],
			$price_info,
			$brand_info,
			$categories_text,
			mb_substr( $product_data['short_description'], 0, 200 ),
			$keyphrase_instruction,
			$site_name,
			self::META_TITLE_MAX_LENGTH
		);

		return $prompt;
	}

	/**
	 * Validate and format title response.
	 *
	 * @since 1.0.0
	 * @param array $parsed Parsed JSON response.
	 * @return array|WP_Error Formatted suggestions or error.
	 */
	private function validate_title_response( $parsed ) {
		if ( ! isset( $parsed['suggestions'] ) || ! is_array( $parsed['suggestions'] ) ) {
			return new WP_Error(
				'invalid_response',
				__( 'Invalid response format from AI.', 'topranker-ai' )
			);
		}

		$suggestions = array();

		foreach ( $parsed['suggestions'] as $suggestion ) {
			if ( ! isset( $suggestion['title'] ) ) {
				continue;
			}

			$title = sanitize_text_field( $suggestion['title'] );

			// Trim if too long.
			if ( mb_strlen( $title ) > self::META_TITLE_MAX_LENGTH ) {
				$title = $this->smart_truncate( $title, self::META_TITLE_MAX_LENGTH );
			}

			$suggestions[] = array(
				'title'  => $title,
				'length' => mb_strlen( $title ),
				'rank'   => isset( $suggestion['rank'] ) ? (int) $suggestion['rank'] : count( $suggestions ) + 1,
			);
		}

		if ( empty( $suggestions ) ) {
			return new WP_Error(
				'no_suggestions',
				__( 'No valid suggestions generated.', 'topranker-ai' )
			);
		}

		// Sort by rank.
		usort(
			$suggestions,
			function( $a, $b ) {
				return $a['rank'] - $b['rank'];
			}
		);

		return array(
			'suggestions' => $suggestions,
		);
	}

	/**
	 * Generate meta description for a product with purchase intent.
	 *
	 * @since 1.0.0
	 * @param WP_Post $post           Post object.
	 * @param array   $product_data   Product data array.
	 * @param string  $focus_keyphrase Optional. Focus keyphrase.
	 * @return array|WP_Error Array with 'suggestions' key or WP_Error.
	 */
	public function generate_product_meta_description( $post, $product_data, $focus_keyphrase = '' ) {
		$api       = $this->get_api();
		$optimizer = $this->get_optimizer();
		$context   = $optimizer->build_context_prefix( $post );

		$prompt = $this->build_product_meta_description_prompt( $product_data, $focus_keyphrase );

		$messages = array(
			array(
				'role'    => 'system',
				'content' => $context,
			),
			array(
				'role'    => 'user',
				'content' => $prompt,
			),
		);

		$response = $api->chat_completion( $messages );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$parsed = $api->parse_json_response( $response['content'] );

		if ( is_wp_error( $parsed ) ) {
			// Retry once.
			$response = $api->chat_completion( $messages );

			if ( is_wp_error( $response ) ) {
				return $response;
			}

			$parsed = $api->parse_json_response( $response['content'] );

			if ( is_wp_error( $parsed ) ) {
				return $parsed;
			}
		}

		return $this->validate_description_response( $parsed );
	}

	/**
	 * Build the product meta description prompt.
	 *
	 * @since 1.0.0
	 * @param array  $product_data   Product data array.
	 * @param string $focus_keyphrase Focus keyphrase if provided.
	 * @return string The formatted prompt.
	 */
	private function build_product_meta_description_prompt( $product_data, $focus_keyphrase ) {
		$keyphrase_instruction = '';
		if ( ! empty( $focus_keyphrase ) ) {
			$keyphrase_instruction = sprintf(
				/* translators: %s: focus keyphrase */
				__( 'Include the focus keyphrase "%s" naturally.', 'topranker-ai' ),
				$focus_keyphrase
			);
		}

		$price_info = '';
		if ( ! empty( $product_data['price'] ) ) {
			$price_info = sprintf(
				__( 'Price: %s%s', 'topranker-ai' ),
				$product_data['currency_symbol'],
				$product_data['price']
			);
			if ( $product_data['is_on_sale'] ) {
				$price_info .= __( ' (On Sale)', 'topranker-ai' );
			}
		}

		$attributes_text = '';
		if ( ! empty( $product_data['attributes'] ) ) {
			$attr_parts = array();
			foreach ( $product_data['attributes'] as $name => $values ) {
				$display_name  = str_replace( array( 'pa_', '-', '_' ), array( '', ' ', ' ' ), $name );
				$display_name  = ucwords( $display_name );
				$values_string = is_array( $values ) ? implode( ', ', $values ) : $values;
				$attr_parts[]  = $display_name . ': ' . $values_string;
			}
			$attributes_text = implode( '; ', array_slice( $attr_parts, 0, 3 ) );
		}

		$rating_info = '';
		if ( ! empty( $product_data['average_rating'] ) && $product_data['average_rating'] > 0 ) {
			$rating_info = sprintf(
				/* translators: 1: Average rating, 2: Number of reviews */
				__( 'Rating: %1$s/5 (%2$d reviews)', 'topranker-ai' ),
				$product_data['average_rating'],
				$product_data['review_count']
			);
		}

		$stock_info = $product_data['in_stock']
			? __( 'In Stock', 'topranker-ai' )
			: __( 'Out of Stock', 'topranker-ai' );

		$prompt = sprintf(
			/* translators: 1: Product name, 2: Price info, 3: Rating info, 4: Stock info, 5: Attributes, 6: Description, 7: Keyphrase instruction, 8: Max length */
			__(
				'Generate 3 SEO-optimized meta description suggestions for this product, optimized for purchase intent.

PRODUCT DETAILS:
Product Name: %1$s
%2$s
%3$s
Availability: %4$s
Key Features: %5$s
Description: %6$s

Requirements:
- Maximum %8$d characters per description (strict limit)
- Optimize for purchase intent - convince shoppers to click and buy
- Include compelling elements: unique selling points, price advantage, key benefits
- Use action-oriented language (e.g., "Shop now", "Get yours today", "Free shipping")
%7$s
- Make each description unique with a different angle

Respond with valid JSON only, in this exact format:
{
  "suggestions": [
    {"description": "First description suggestion", "rank": 1},
    {"description": "Second description suggestion", "rank": 2},
    {"description": "Third description suggestion", "rank": 3}
  ]
}

Rank them by effectiveness for driving sales, with 1 being the best.',
				'topranker-ai'
			),
			$product_data['name'],
			$price_info,
			$rating_info,
			$stock_info,
			$attributes_text,
			mb_substr( wp_strip_all_tags( $product_data['description'] ), 0, 500 ),
			$keyphrase_instruction,
			self::META_DESCRIPTION_MAX_LENGTH
		);

		return $prompt;
	}

	/**
	 * Validate and format description response.
	 *
	 * @since 1.0.0
	 * @param array $parsed Parsed JSON response.
	 * @return array|WP_Error Formatted suggestions or error.
	 */
	private function validate_description_response( $parsed ) {
		if ( ! isset( $parsed['suggestions'] ) || ! is_array( $parsed['suggestions'] ) ) {
			return new WP_Error(
				'invalid_response',
				__( 'Invalid response format from AI.', 'topranker-ai' )
			);
		}

		$suggestions = array();

		foreach ( $parsed['suggestions'] as $suggestion ) {
			if ( ! isset( $suggestion['description'] ) ) {
				continue;
			}

			$description = sanitize_text_field( $suggestion['description'] );

			// Trim if too long.
			if ( mb_strlen( $description ) > self::META_DESCRIPTION_MAX_LENGTH ) {
				$description = $this->smart_truncate( $description, self::META_DESCRIPTION_MAX_LENGTH );
			}

			$suggestions[] = array(
				'description' => $description,
				'length'      => mb_strlen( $description ),
				'rank'        => isset( $suggestion['rank'] ) ? (int) $suggestion['rank'] : count( $suggestions ) + 1,
			);
		}

		if ( empty( $suggestions ) ) {
			return new WP_Error(
				'no_suggestions',
				__( 'No valid suggestions generated.', 'topranker-ai' )
			);
		}

		// Sort by rank.
		usort(
			$suggestions,
			function( $a, $b ) {
				return $a['rank'] - $b['rank'];
			}
		);

		return array(
			'suggestions' => $suggestions,
		);
	}

	/**
	 * Generate optimized short description for a product.
	 *
	 * @since 1.0.0
	 * @param WP_Post $post         Post object.
	 * @param array   $product_data Product data array.
	 * @return array|WP_Error Array with 'short_description' key or WP_Error.
	 */
	public function generate_short_description( $post, $product_data ) {
		$api       = $this->get_api();
		$optimizer = $this->get_optimizer();
		$context   = $optimizer->build_context_prefix( $post );

		$prompt = $this->build_short_description_prompt( $product_data );

		$messages = array(
			array(
				'role'    => 'system',
				'content' => $context,
			),
			array(
				'role'    => 'user',
				'content' => $prompt,
			),
		);

		$response = $api->chat_completion( $messages );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$parsed = $api->parse_json_response( $response['content'] );

		if ( is_wp_error( $parsed ) ) {
			// Retry once.
			$response = $api->chat_completion( $messages );

			if ( is_wp_error( $response ) ) {
				return $response;
			}

			$parsed = $api->parse_json_response( $response['content'] );

			if ( is_wp_error( $parsed ) ) {
				return $parsed;
			}
		}

		return $this->validate_short_description_response( $parsed );
	}

	/**
	 * Build the short description prompt.
	 *
	 * @since 1.0.0
	 * @param array $product_data Product data array.
	 * @return string The formatted prompt.
	 */
	private function build_short_description_prompt( $product_data ) {
		$attributes_text = '';
		if ( ! empty( $product_data['attributes'] ) ) {
			$attr_parts = array();
			foreach ( $product_data['attributes'] as $name => $values ) {
				$display_name  = str_replace( array( 'pa_', '-', '_' ), array( '', ' ', ' ' ), $name );
				$display_name  = ucwords( $display_name );
				$values_string = is_array( $values ) ? implode( ', ', $values ) : $values;
				$attr_parts[]  = $display_name . ': ' . $values_string;
			}
			$attributes_text = implode( '; ', $attr_parts );
		}

		$prompt = sprintf(
			/* translators: 1: Product name, 2: Categories, 3: Attributes, 4: Full description, 5: Max length */
			__(
				'Generate an optimized product short description that highlights key selling points and encourages purchase.

PRODUCT DETAILS:
Product Name: %1$s
Categories: %2$s
Key Attributes: %3$s
Full Description: %4$s

Requirements:
- Maximum %5$d characters (strict limit)
- Focus on unique selling points and key benefits
- Use persuasive, benefit-focused language
- Include the most important features that differentiate this product
- Make it scannable - consider using bullet points or short sentences
- Avoid generic phrases; be specific about what makes this product valuable
- End with a subtle call-to-action if space allows

Respond with valid JSON only, in this exact format:
{
  "short_description": "The optimized short description text here"
}',
				'topranker-ai'
			),
			$product_data['name'],
			implode( ', ', $product_data['categories'] ),
			$attributes_text,
			mb_substr( wp_strip_all_tags( $product_data['description'] ), 0, 1000 ),
			self::SHORT_DESCRIPTION_MAX_LENGTH
		);

		return $prompt;
	}

	/**
	 * Validate and format short description response.
	 *
	 * @since 1.0.0
	 * @param array $parsed Parsed JSON response.
	 * @return array|WP_Error Formatted short description or error.
	 */
	private function validate_short_description_response( $parsed ) {
		if ( ! isset( $parsed['short_description'] ) || ! is_string( $parsed['short_description'] ) ) {
			return new WP_Error(
				'invalid_response',
				__( 'Invalid response format from AI.', 'topranker-ai' )
			);
		}

		$short_description = wp_kses_post( $parsed['short_description'] );

		// Trim if too long.
		if ( mb_strlen( wp_strip_all_tags( $short_description ) ) > self::SHORT_DESCRIPTION_MAX_LENGTH ) {
			$short_description = $this->smart_truncate( $short_description, self::SHORT_DESCRIPTION_MAX_LENGTH );
		}

		if ( empty( $short_description ) ) {
			return new WP_Error(
				'empty_short_description',
				__( 'Generated short description was empty.', 'topranker-ai' )
			);
		}

		return array(
			'short_description' => $short_description,
			'length'            => mb_strlen( wp_strip_all_tags( $short_description ) ),
		);
	}

	/**
	 * Smart truncate text at word boundary.
	 *
	 * @since 1.0.0
	 * @param string $text   Text to truncate.
	 * @param int    $length Maximum length.
	 * @return string Truncated text.
	 */
	private function smart_truncate( $text, $length ) {
		if ( mb_strlen( $text ) <= $length ) {
			return $text;
		}

		// Truncate to length.
		$truncated = mb_substr( $text, 0, $length );

		// Try to break at word boundary.
		$last_space = mb_strrpos( $truncated, ' ' );

		if ( false !== $last_space && $last_space > $length * 0.8 ) {
			$truncated = mb_substr( $truncated, 0, $last_space );
		}

		return trim( $truncated );
	}

	/**
	 * Save optimized short description to the product.
	 *
	 * @since 1.0.0
	 * @param int    $product_id       Product ID.
	 * @param string $short_description Short description to save.
	 * @return bool True on success.
	 */
	public function save_short_description( $product_id, $short_description ) {
		$product = wc_get_product( $product_id );

		if ( ! $product ) {
			return false;
		}

		$short_description = wp_kses_post( $short_description );

		if ( mb_strlen( wp_strip_all_tags( $short_description ) ) > self::SHORT_DESCRIPTION_MAX_LENGTH ) {
			$short_description = $this->smart_truncate( $short_description, self::SHORT_DESCRIPTION_MAX_LENGTH );
		}

		$product->set_short_description( $short_description );

		return (bool) $product->save();
	}

	/**
	 * Add product context to the optimization context.
	 *
	 * Filter callback for 'topranker_post_context'.
	 *
	 * @since 1.0.0
	 * @param array   $context Existing context array.
	 * @param WP_Post $post    Post object.
	 * @return array Modified context array.
	 */
	public function add_product_context( $context, $post ) {
		if ( ! $this->is_product( $post ) ) {
			return $context;
		}

		$product_data = $this->get_product_data( $post );

		if ( is_wp_error( $product_data ) ) {
			return $context;
		}

		$context['product_data'] = $product_data;
		$context['is_product']   = true;

		return $context;
	}

	/**
	 * Get cached product optimization results.
	 *
	 * @since 1.0.0
	 * @param int $product_id Product ID.
	 * @return array|null Cached results or null.
	 */
	public function get_cached_results( $product_id ) {
		$cached = get_post_meta( $product_id, '_topranker_product_optimization', true );

		if ( ! is_array( $cached ) || empty( $cached['results'] ) ) {
			return null;
		}

		return $cached['results'];
	}

	/**
	 * Cache product optimization results.
	 *
	 * @since 1.0.0
	 * @param int   $product_id Product ID.
	 * @param array $results    Optimization results.
	 * @return bool True on success.
	 */
	public function cache_results( $product_id, $results ) {
		$cache_data = array(
			'results'   => $results,
			'timestamp' => time(),
		);

		return (bool) update_post_meta( $product_id, '_topranker_product_optimization', $cache_data );
	}

	/**
	 * Clear cached product optimization results.
	 *
	 * @since 1.0.0
	 * @param int $product_id Product ID.
	 * @return bool True on success.
	 */
	public function clear_cached_results( $product_id ) {
		return delete_post_meta( $product_id, '_topranker_product_optimization' );
	}

	/**
	 * Check if product optimization is available.
	 *
	 * @since  1.0.0
	 * @return bool True if WooCommerce is active and Pro is licensed.
	 */
	public static function is_available() {
		return class_exists( 'WooCommerce' )
			&& function_exists( 'topranker_is_pro' )
			&& topranker_is_pro();
	}
}
